<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Collections\ProductAttributeCollection;
use App\Collections\ProductBrandCollection;
use App\Collections\ProductCollection;
use App\Collections\ProductImageCollection;
use App\Collections\ProductPricingCollection;
use App\Collections\SupplierCollection;
use App\Data\SupplierInventoryData;
use App\Data\SupplierProductData;
use App\DTO\ProductAttributeDto;
use App\DTO\ProductDto;
use App\DTO\ProductImageDto;
use App\DTO\ProductPricingDto;
use App\DTO\WarehouseDto;
use App\Exceptions\MultipleInitialCountFifoLayersException;
use App\Helpers;
use App\Models\FifoLayer;
use App\Models\IntegrationInstance;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductAttribute;
use App\Models\ProductBrand;
use App\Models\ProductComponent;
use App\Models\ProductImage;
use App\Models\ProductInventory;
use App\Models\ProductPricing;
use App\Models\ProductPricingTier;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Models\SupplierPricingTier;
use App\Models\SupplierProduct;
use App\Models\SupplierProductPricing;
use App\Models\Warehouse;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Collection;
use Illuminate\Support\Enumerable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redis;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;

class ProductRepository extends AbstractRepository
{
    private SupplierRepository $supplierRepository;

    private ProductImageRepository $productImageRepository;

    private ProductPricingRepository $productPricingRepository;

    private ProductAttributeRepository $productAttributeRepository;

    private ProductBrandRepository $productBrandRepository;

    public function __construct()
    {
        $this->supplierRepository = new SupplierRepository();
        $this->productImageRepository = new ProductImageRepository();
        $this->productPricingRepository = new ProductPricingRepository();
        $this->productAttributeRepository = new ProductAttributeRepository();
        $this->productBrandRepository = new ProductBrandRepository();
    }

    public function getProductBySku(string $sku): ?Product
    {
        return Product::whereSku($sku)->first();
    }

    public function getProductBySkuForAmazonFbaProductListing(string $sku): ?Product
    {
        return Product::whereSku($sku)
            ->whereIn('type', [
                Product::TYPE_STANDARD,
                Product::TYPE_KIT,
                Product::TYPE_BLEMISHED,
                Product::TYPE_BUNDLE,
            ])
            ->first();
    }

    public function getProductBySkuForAmazonNonFbaProductListing(string $sku): ?Product
    {
        return Product::whereSku($sku)
            ->whereIn('type', [
                Product::TYPE_STANDARD,
                Product::TYPE_KIT,
                Product::TYPE_BLEMISHED,
            ])
            ->first();
    }
    //
    //    public function save(ProductDto $productDto): Product
    //    {
    //        $product = Product::firstOrCreate([
    //            'sku' => $productDto->sku,
    //        ], $productDto->toArray());
    //
    //        //Save product brand
    //        if ($brandName = $productDto?->product_brand?->name) {
    //            $productBrand = $this->productBrandRepository->getProductBrandByName($brandName);
    //
    //            if (is_null($productBrand)) {
    //                $productBrand = $this->productBrandRepository->save($productDto->product_brand);
    //            }
    //
    //            $product->brand_id = $productBrand->id;
    //            $product->update();
    //        }
    //
    //        //Save product prices
    //        if ($productDto?->product_prices?->count() > 0) {
    //            $this->productPricingRepository->save($product, $productDto->product_prices);
    //        }
    //
    //        //Save product attributes
    //        if ($productDto?->product_attributes?->count() > 0) {
    //            $this->productAttributeRepository->save($product, $productDto->product_attributes);
    //        }
    //
    //        if (!empty($productDto->images) && $productDto->images?->count() > 0) {
    //            $this->productImageRepository->save($product, $productDto->images);
    //        }
    //
    //        if ($productDto->supplier?->name) {
    //            $supplier = Supplier::firstOrCreate([
    //                'name' => $productDto->supplier?->name,
    //            ], [
    //                'name' => $productDto->supplier?->name,
    //            ]);
    //
    //            SupplierProduct::firstOrCreate([
    //                'supplier_id' => $supplier->id,
    //                'product_id' => $product->id,
    //            ], [
    //                'supplier_id' => $supplier->id,
    //                'product_id' => $product->id,
    //                'is_default' => true,
    //            ]);
    //        }
    //
    //        return $product;
    //    }

    public function saveBulk(ProductCollection $data, bool $insert, bool $update, $metadata): array
    {
        $mappings = [
            [
                'expected_column_name' => 'type',
                'data_column_name' => 'type',
                'expected_column_type' => 'string',
            ],
            [
                'expected_column_name' => 'sku',
                'data_column_name' => 'sku',
                'expected_column_type' => 'string',
            ],
            [
                'expected_column_name' => 'barcode',
                'data_column_name' => 'barcode',
                'expected_column_type' => 'string',
            ],
            [
                'expected_column_name' => 'name',
                'data_column_name' => 'name',
                'expected_column_type' => 'string',
            ],
            [
                'expected_column_name' => 'weight',
                'data_column_name' => 'weight',
                'expected_column_type' => 'string',
            ],
            [
                'expected_column_name' => 'weight_unit',
                'data_column_name' => 'weight_unit',
                'expected_column_type' => 'string',
            ],
            [
                'expected_column_name' => 'unit_cost',
                'data_column_name' => 'unit_cost',
                'expected_column_type' => 'string',
            ],
        ];

        if (@$metadata['product_brand']['insert']) {
            $productBrandMetadata = $metadata['product_brand'];
            (new ProductBrandRepository())->saveBulk((new ProductBrandCollection($data->pluck('product_brand')->filter()->values())), $productBrandMetadata['insert'], $productBrandMetadata['update']);

            $mappings[] = [
                'expected_column_name' => 'product_brand_name',
                'data_column_name' => 'product_brand.->.name',
                'expected_column_type' => 'string',
                'is_importable' => false,
            ];

            $mappings[] = [
                'joins' => [
                    [
                        'table' => (new ProductBrand())->getTable(),
                        'type' => 'leftJoin',
                        'on' => [
                            [
                                'database_column' => 'name',
                                'stage_column_to_join' => 'product_brand_name',
                            ],
                        ],
                        'column_type' => 'integer',
                        'add_select' => [
                            [
                                'expected_column_name' => 'brand_id',
                                'data_column_name' => 'id',
                                'expected_column_type' => 'integer',
                                'is_importable' => true,
                            ],

                        ],
                    ],
                ],
            ];
        }

        $tempTable = Product::dataFeedBulkImport([
            'data_to_import' => [
                'type' => 'json',
                'data' => $data->toJson(),
            ],
            'insert' => $insert,
            'update' => $update,
            'mappings' => $mappings,
            'default_columns' => [
                [
                    'expected_column_name' => 'created_at',
                    'default_value' => now(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'updated_at',
                    'default_value' => now(),
                    'expected_column_type' => 'datetime',
                ],
            ],
            'unique_by_columns' => [
                'sku',
            ],
        ]);

        //Link supplier and products
        if (@$metadata['supplier']) {
            $supplierMetadata = $metadata['supplier'];
            if (@$supplierMetadata['insert'] || @$supplierMetadata['update']) {
                $this->supplierRepository->saveBulk((new SupplierCollection($data->pluck('supplier')->filter()->values())), $supplierMetadata['insert'], $supplierMetadata['update'], []);
            }

            SupplierProduct::dataFeedBulkImport([
                'data_to_import' => [
                    'type' => 'json',
                    'data' => $data->where('supplier', '!=', null)->toJson(),
                ],
                'insert' => true,
                'update' => false,
                'mappings' => [
                    [
                        'expected_column_name' => 'supplier_name',
                        'data_column_name' => 'supplier.->.name',
                        'expected_column_type' => 'string',
                        'is_importable' => false,
                    ],
                    [
                        'expected_column_name' => 'sku',
                        'data_column_name' => 'sku',
                        'expected_column_type' => 'string',
                        'is_importable' => false,
                    ],
                    [
                        'joins' => [
                            [
                                'table' => 'suppliers',
                                'type' => 'leftJoin',
                                'on' => [
                                    [
                                        'database_column' => 'name',
                                        'stage_column_to_join' => 'supplier_name',
                                    ],
                                ],
                                'column_type' => 'integer',
                                'add_select' => [
                                    [
                                        'expected_column_name' => 'supplier_id',
                                        'data_column_name' => 'id',
                                        'expected_column_type' => 'integer',
                                        'is_importable' => true,
                                    ],
                                ],
                            ],
                        ],
                    ],
                    [
                        'joins' => [
                            [
                                'table' => 'products',
                                'type' => 'leftJoin',
                                'on' => [
                                    [
                                        'database_column' => 'sku',
                                        'stage_column_to_join' => 'sku',
                                    ],
                                ],
                                'column_type' => 'integer',
                                'add_select' => [
                                    [
                                        'expected_column_name' => 'product_id',
                                        'data_column_name' => 'id',
                                        'expected_column_type' => 'integer',
                                        'is_importable' => true,
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
                'default_columns' => [
                    [
                        'expected_column_name' => 'created_at',
                        'default_value' => now(),
                        'expected_column_type' => 'datetime',
                    ],
                    [
                        'expected_column_name' => 'updated_at',
                        'default_value' => now(),
                        'expected_column_type' => 'datetime',
                    ],
                ],
                'unique_by_columns' => [
                    'supplier_id',
                    'product_id',
                ],
            ]);

            SupplierInventory::dataFeedBulkImport([
                'data_to_import' => [
                    'type' => 'json',
                    'data' => $data->where('supplier_name', '!=', null)->toJson(),
                ],
                'insert' => true,
                'update' => false,
                'mappings' => [
                    [
                        'expected_column_name' => 'supplier_name',
                        'data_column_name' => 'supplier.->.name',
                        'expected_column_type' => 'string',
                        'is_importable' => false,
                    ],
                    [
                        'expected_column_name' => 'sku',
                        'data_column_name' => 'sku',
                        'expected_column_type' => 'string',
                        'is_importable' => false,
                    ],
                    [
                        'joins' => [
                            [
                                'table' => 'suppliers',
                                'type' => 'leftJoin',
                                'on' => [
                                    [
                                        'database_column' => 'name',
                                        'stage_column_to_join' => 'supplier_name',
                                    ],
                                ],
                                'column_type' => 'integer',
                                'add_select' => [
                                    [
                                        'expected_column_name' => 'supplier_id',
                                        'data_column_name' => 'id',
                                        'expected_column_type' => 'integer',
                                        'is_importable' => true,
                                    ],
                                    [
                                        'expected_column_name' => 'warehouse_id',
                                        'data_column_name' => 'default_warehouse_id',
                                        'expected_column_type' => 'integer',
                                        'is_importable' => true,
                                    ],
                                ],
                            ],
                        ],
                    ],
                    [
                        'joins' => [
                            [
                                'table' => 'products',
                                'type' => 'leftJoin',
                                'on' => [
                                    [
                                        'database_column' => 'sku',
                                        'stage_column_to_join' => 'sku',
                                    ],
                                ],
                                'column_type' => 'integer',
                                'add_select' => [
                                    [
                                        'expected_column_name' => 'product_id',
                                        'data_column_name' => 'id',
                                        'expected_column_type' => 'integer',
                                        'is_importable' => true,
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
                'default_columns' => [
                    [
                        'expected_column_name' => 'created_at',
                        'default_value' => now(),
                        'expected_column_type' => 'datetime',
                    ],
                    [
                        'expected_column_name' => 'updated_at',
                        'default_value' => now(),
                        'expected_column_type' => 'datetime',
                    ],
                ],
                'unique_by_columns' => [
                    'supplier_id',
                    'product_id',
                ],
            ]);
        }

        //Save images
        if (@$metadata['images']) {
            $imageMetadata = $metadata['images'];
            if (@$imageMetadata['insert']) {
                $productImagesCollection = new ProductImageCollection();
                foreach ($data->pluck('images')->filter()->values() as $imagesArray) {
                    $productImagesCollection = $productImagesCollection->merge($imagesArray);
                }

                $this->productImageRepository->saveBulk($productImagesCollection, $imageMetadata['insert'], $imageMetadata['update'], []);
            }
        }

        //Save product pricing
        if (@$metadata['product_pricing']) {
            $productPricingMetadata = $metadata['product_pricing'];
            if (@$productPricingMetadata['insert'] || @$productPricingMetadata['update']) {
                $this->productPricingRepository->saveBulk((new ProductPricingCollection($data->pluck('product_pricing')->filter()->values())), $productPricingMetadata['insert'], $productPricingMetadata['update'], $productPricingMetadata);
            }
        }

        //Save product prices
        if (@$metadata['product_prices']) {
            $productPricesMetadata = $metadata['product_prices'];
            if (@$productPricesMetadata['insert'] || @$productPricesMetadata['update']) {
                $productPricesCollection = new ProductPricingCollection();
                foreach ($data->pluck('product_prices')->filter()->values() as $pricesArray) {
                    $productPricesCollection = $productPricesCollection->merge($pricesArray);
                }

                if ($productPricesCollection->count() > 0) {
                    $this->productPricingRepository->saveBulk($productPricesCollection, $productPricesMetadata['insert'], $productPricesMetadata['update'], $productPricesMetadata);
                }
            }
        }

        //Save product Attributes
        if (@$metadata['product_attributes']) {
            $productAttributesMetadata = $metadata['product_attributes'];
            if (@$productAttributesMetadata['insert'] || @$productAttributesMetadata['update']) {
                $productAttributeCollection = new ProductAttributeCollection();
                foreach ($data->pluck('product_attributes')->filter()->values() as $attributeArray) {
                    $productAttributeCollection = $productAttributeCollection->merge($attributeArray);
                }

                if ($productAttributeCollection->count() > 0) {
                    $this->productAttributeRepository->saveBulk($productAttributeCollection, $productAttributesMetadata['insert'], $productAttributesMetadata['update'], $productAttributesMetadata);
                }
            }
        }

        return [
            'temp_table' => $tempTable,
            'status' => true,
        ];
    }

    public function bulkSaveDailyAverageConsumption(Collection $data): void
    {

        $data->chunk(25000)->each(function ($chunk) {
            batch()->update(new Product(), $chunk->toArray(), 'id');
        });
    }

    public function clearInvalidDailyAverageConsumptionCache(array $productIds = []): void
    {
        if (! empty($productIds)) {
            Redis::connection('cache')->srem(Product::INVALID_DAILY_AVERAGE_CONSUMPTION_KEY, ...$productIds);
        } else {
            Redis::connection('cache')->del(Product::INVALID_DAILY_AVERAGE_CONSUMPTION_KEY);
        }
    }

    public function getQuantitySoldSinceDate(Carbon $fromDate, array $productIds): Collection
    {
        $query = Product::query()
            ->whereIn('products.id', Redis::connection('cache')->smembers(Product::INVALID_DAILY_AVERAGE_CONSUMPTION_KEY))
            ->leftJoin('reporting_daily_financials', function (JoinClause $join) use ($fromDate) {
                $join->on('products.id', '=', 'reporting_daily_financials.reportable_id')
                    ->where('reporting_daily_financials.reportable_type', '=', Product::class)
                    ->where('reporting_daily_financials.date', '>=', $fromDate->format('Y-m-d'));
                // TODO: fromDate should be full format once converting over.  Until then the sales records could be inaccurate
            })
            ->groupBy(['products.id', 'products.sku'])
            ->select([
                'products.id as id',
                'products.sku',
                DB::raw(
                    '(IFNULL((SUM(`reporting_daily_financials`.`quantity`)), 0) / 
                    '.Helpers::setting(Setting::KEY_DAYS_SALES_HISTORY).') as daily_average_consumption'
                ),
            ])
            ->when($productIds, function (Builder $builder) use ($productIds) {
                return
                    $builder->whereIn('reporting_daily_financials.reportable_id', $productIds)
                        ->where('reporting_daily_financials.reportable_type', Product::class);
            });

        return $query->get();
    }

    public function invalidateDailyAverageConsumptionCache(array $productIds): void
    {
        if (empty($productIds)) {
            return;
        }
        Redis::connection('cache')->sadd(Product::INVALID_DAILY_AVERAGE_CONSUMPTION_KEY, ...$productIds);
    }

    public function getForIdOrFail(int $productId): Product|Model
    {
        return Product::query()->findOrFail($productId);
    }

    /*
     * @param  Product  $product
     * @param  int  $warehouseId
     * @param  int  $supplier_id
     *
     * @return float
     */
    public function getCostForSupplier(Product $product, int $warehouseId, int $supplier_id): float
    {
        /** @var Supplier $supplier */
        $supplier = Supplier::query()->findOrFail($supplier_id);

        $supplierPricingTierId = $supplier->defaultPricingTier?->id ?? SupplierPricingTier::default()->id;
        if ($supplierPricingTierId) {
            /** @var SupplierProductPricing $defaultSupplierPricing */
            $defaultSupplierPricing = $product->supplierProducts()
                ->where('supplier_id', $supplier->id)
                ->first()
                ?->getDefaultSupplierPricing($supplierPricingTierId);
            if ($defaultSupplierPricing && $defaultSupplierPricing->price > 0) {
                return $defaultSupplierPricing->price;
            }
        }

        if ($product->unit_cost > 0) {
            return $product->unit_cost;
        }

        /** @var FifoLayer $activeFifoLayer */
        $activeFifoLayer = $product->activeFifoLayers()
            ->where('warehouse_id', $warehouseId)
            ->where('total_cost', '>', 0)
            ->first();

        if ($activeFifoLayer) {
            return $activeFifoLayer->avg_cost;
        }

        // No active fifo layer,
        // we use average cost if there are used fifo layers.
        if ($product->fifoLayers()->count() > 0 && $product->average_cost > 0) {
            return $product->average_cost;
        }

        return 0.00;
    }

    /**
     * Create a kit from a bundle.
     */
    public function createKitFromBundle(Product $bundle, array $kitData): ?Product
    {
        // Create the kit product
        $kit = Product::create([
            'sku' => $kitData['sku'],
            'name' => $kitData['name'],
            'type' => Product::TYPE_KIT,
        ]);

        if (! $kit) {
            return null;
        }

        $bundle->update(['parent_id' => $kit->id]);
        $defaultPricingTier = ProductPricingTier::default();

        $kit->setPricing([
            [
                'product_pricing_tier_id' => $defaultPricingTier->id,
                'price' => $bundle->price,
            ],
        ]);
        // Get the bundle components
        $components = $bundle->productComponents;
        if ($components) {
            foreach ($components as $component) {
                $componentProduct = $component->childProduct;

                if ($componentProduct) {
                    // Create kit components from the bundle components
                    ProductComponent::create([
                        'parent_product_id' => $kit->id,
                        'component_product_id' => $componentProduct->id,
                        'quantity' => $component->quantity,
                    ]);
                    $componentProduct->parent_id = $kit->id;
                    $componentProduct->save();
                    $component->childProduct->setPricing([
                        [
                            'product_pricing_tier_id' => $defaultPricingTier->id,
                            'price' => $componentProduct->price,
                        ],
                    ]);
                }
            }
        }

        return $kit;
    }

    /**
     * @deprecated replaced due to different logic requirements by SalesOrderLineRepository::withoutDailyFinancialsQuery()
     */
    public function soldWithoutDailyFinancialQuery(string $userTimezone): Builder
    {
        return Product::query()
            ->distinct()
            ->select('products.*')
            ->join('sales_order_lines', 'products.id', '=', 'sales_order_lines.product_id')
            ->join('sales_orders', 'sales_order_lines.sales_order_id', '=', 'sales_orders.id')
            ->leftJoin('reporting_daily_financials', function ($join) use ($userTimezone) {
                $join->on(
                    DB::raw("CONVERT_TZ(DATE(CONVERT_TZ(sales_orders.order_date, 'UTC', '$userTimezone')), '$userTimezone', 'UTC')"),
                    '=',
                    'reporting_daily_financials.date'
                );

                $join->on('reporting_daily_financials.reportable_id', '=', 'sales_order_lines.product_id');

                $join->where('reporting_daily_financials.reportable_type', '=', Product::class);
            })
            ->where('sales_orders.order_status', '!=', 'draft')
            ->whereNull('reporting_daily_financials.id');
    }

    public function getProductIdsNeedingListingCacheUpdate(IntegrationInstance $integrationInstance): array
    {
        return ProductInventory::join('product_listings as pl', 'pl.product_id', '=', 'products_inventory.product_id')
            ->whereColumn('products_inventory.updated_at', '>', 'pl.updated_at')
            ->where('pl.sales_channel_id', $integrationInstance->salesChannel->id)
            ->distinct()
            ->pluck('products_inventory.product_id')
            ->toArray();
    }

    public function getInboundQuantityForProduct(Product $product, ?Warehouse $warehouse = null, array $excludeWarehouseTypes = []): float
    {

        return PurchaseOrderLine::query()
            ->selectRaw('sum(unreceived_quantity) as inbound_quantity')
            ->join('purchase_orders', 'purchase_orders.id', '=', 'purchase_order_lines.purchase_order_id')
            ->join('warehouses', 'warehouses.id', '=', 'purchase_orders.destination_warehouse_id')
            ->where('purchase_order_lines.product_id', $product->id)
            ->where('purchase_orders.order_status', PurchaseOrder::STATUS_OPEN)
            ->when($warehouse, function ($query) use ($warehouse) {
                $query->where('purchase_orders.destination_warehouse_id', $warehouse->id);
            })
            ->when(!empty($excludeWarehouseTypes), function ($query) use ($excludeWarehouseTypes) {
                $query->whereNotIn('warehouses.type', $excludeWarehouseTypes);
            })
            ->value('inbound_quantity') ?? 0;
    }

    /**
     * @throws Exception
     */
    public function saveComponentLineFromBundle(SalesOrderLine $bundleLine, Product $bundleProduct, Product $bundleComponent): SalesOrderLine
    {
        $componentLine = $bundleLine->replicate();
        $componentLine->bundle_id = $bundleProduct->id;
        $componentLine->product_id = $bundleComponent->id;
        $componentLine->description = $bundleComponent->name;
        $componentLine->amount = $bundleLine->amount * $bundleComponent->getBundlePriceProration($bundleProduct);
        $componentLine->quantity = $bundleLine->quantity * $bundleComponent->pivot->quantity;
        $componentLine->save();

        return $componentLine;
    }

    public function saveWithRelations(#[DataCollectionOf(ProductDto::class)] DataCollection $data): Collection
    {
        $data = $data->toCollection();

        // Save brands
        $brandCollection = app(ProductBrandRepository::class)->save($data->pluck('product_brand'), ProductBrand::class);

        // Save products

        $productCollection = $data->map(function ($product) use ($brandCollection) {
            $product->brand_id = $product->product_brand instanceof Optional
                ? null
                : $brandCollection->where('name', $product->product_brand?->name)->first()['id'];

            return ProductDto::from(new Product($product->toArray()));
        });

        $productCollection = $this->save($productCollection, Product::class);

        $this->saveSupplierRelations($data, $productCollection);
        $this->saveImages($data, $productCollection);
        $this->savePricing($data, $productCollection);
        $this->saveAttributes($data, $productCollection);

        return $productCollection;
    }

    private function saveSupplierRelations(Enumerable $data, Collection $productCollection): void
    {
        $supplierCollection = app(SupplierRepository::class)->save($data->pluck('supplier'), Supplier::class);

        $supplierWarehouseCollection = $supplierCollection->map(function ($supplier) {
            return WarehouseDto::from([
                'name' => 'Main Warehouse',
                'supplier_id' => $supplier['id'],
                'integration_instance_id' => null,
                'type' => Warehouse::TYPE_SUPPLIER,
            ]);
        });

        app(WarehouseRepository::class)->save($supplierWarehouseCollection, Warehouse::class);

        $supplierProductCollection = $data->reject(fn ($product) => $product->supplier instanceof Optional)
            ->map(function ($product) use ($supplierCollection, $productCollection) {
                return SupplierProductData::from([
                    'supplier_id' => $supplierCollection->where('name', $product->supplier?->name)->first()['id'],
                    'product_id' => $productCollection->where('sku', $product->sku)->first()['id'],
                    'is_default' => true,
                ]);
            });

        $supplierProductCollection = app(SupplierProductRepository::class)->save($supplierProductCollection, SupplierProduct::class);

        $supplierInventoryCollection = collect();
        $supplierProductCollection->each(function ($supplierProduct) use ($supplierInventoryCollection) {
            app(WarehouseRepository::class)
                ->getWarehousesFromSupplierId($supplierProduct['supplier_id'])->each(function (Warehouse $warehouse) use ($supplierInventoryCollection, $supplierProduct) {
                    $supplierInventoryCollection->add(SupplierInventoryData::from([
                        'supplier_id' => $supplierProduct['supplier_id'],
                        'product_id' => $supplierProduct['product_id'],
                        'warehouse_id' => $warehouse->id,
                    ]));
                });
        });

        app(SupplierInventoryRepository::class)->save($supplierInventoryCollection, SupplierInventory::class);
    }

    private function saveImages(Enumerable $data, Collection $productCollection): void
    {
        $productImageCollection = $data->reject(fn ($product) => $product->images instanceof Optional)
            ->map(function ($product) use ($productCollection) {
                return $product->images->toCollection()->map(function ($image) use ($productCollection, $product) {
                    return ProductImageDto::from([
                        'product_id' => $productCollection->where('sku', $product->sku)->first()['id'],
                        'url' => $image->url,
                    ]);
                });
            })->flatten();

        app(ProductImageRepository::class)->save($productImageCollection, ProductImage::class);
    }

    private function savePricing(Enumerable $data, Collection $productCollection): void
    {
        $productPricingCollection = $data->reject(fn ($product) => $product->pricing instanceof Optional)
            ->map(function ($product) use ($productCollection) {
                return $product->pricing->toCollection()->map(function ($pricing) use ($productCollection, $product) {
                    return ProductPricingDto::from([
                        'product_id' => $productCollection->where('sku', $product->sku)->first()['id'],
                        'product_pricing_tier_id' => $pricing->product_pricing_tier_id,
                        'price' => $pricing->price,
                    ]);
                });
            })->flatten();

        app(ProductPricingRepository::class)->save($productPricingCollection, ProductPricing::class);
    }

    private function saveAttributes(Enumerable $data, Collection $productCollection): void
    {
        $productAttributeCollection = $data->reject(fn ($product) => $product->attributes instanceof Optional)
            ->map(function ($product) use ($productCollection) {
                return $product->attributes->toCollection()->map(function ($attribute) use (
                    $productCollection,
                    $product
                ) {
                    return ProductAttributeDto::from([
                        'product_id' => $productCollection->where('sku', $product->sku)->first()['id'],
                        'attribute_id' => $attribute->attribute_id,
                        'value' => $attribute->value,
                    ]);
                });
            })->flatten();

        app(ProductAttributeRepository::class)->save($productAttributeCollection, ProductAttribute::class);
    }

    public function skusForMatchingProductTypes(array $skus, array $types): array
    {
        $resultSkus = [];
        foreach (array_chunk($skus, 10000) as $chunk) {
            $chunkResult = Product::query()
                ->whereIn('sku', $chunk)
                ->whereIn('type', $types)
                ->pluck('sku')
                ->toArray();
            $resultSkus = array_merge($resultSkus, $chunkResult);
        }

        return $resultSkus;
    }

    public function getForMatchingIdsAndType(array $ids, string $type): EloquentCollection
    {
        $results = new EloquentCollection();
        foreach (array_chunk($ids, 10000) as $chunk) {
            $chunkResult = Product::query()
                ->whereIn('id', $chunk)
                ->where('type', $type)
                ->get();
            $results = $results->merge($chunkResult);
        }

        return $results;
    }

    public function getProductsFromSkus(array $skus): EloquentCollection
    {
        $chunkSize = 10000;
        $products = new EloquentCollection();

        foreach (array_chunk($skus, $chunkSize) as $chunk) {
            $productsChunk = Product::query()
                ->whereIn('sku', $chunk)
                ->get();

            $products = $products->merge($productsChunk);
        }

        return $products;
    }

    public function getSalesChannelCoverage(): Collection
    {
        return Product::with([
            'productInventory',
            'productListings.ebayLegacyProduct',
            'productListings.salesChannel.integrationInstance.integration',
            'parentProducts.productListings.salesChannel.integrationInstance.integration',
            'parentProducts.productListings.document',
            'defaultSupplierProduct.supplier',
            'activeFifoLayers'
        ])
            ->whereIn("type", [
                Product::TYPE_STANDARD,
                Product::TYPE_KIT,
                Product::TYPE_BLEMISHED,
            ])
            ->whereHas('productInventory', function (Builder $query) {
                $query->where('warehouse_id', 0);
                $query->where("inventory_available", ">", 0);
            })
            ->get();
    }

    public function getAvailableQuantityInWarehouse(Product $product, Warehouse $warehouse): float
    {
        return $product->inventoryMovements()
            ->where('warehouse_id', $warehouse->id)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->sum('quantity');
    }
}
