<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelIntegrationInstance;
use App\Data\IntegrationInstanceInventoryData;
use App\Data\InventoryLocationData;
use App\DTO\ProductDto;
use App\DTO\ProductListingDto;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\ProductListingInventoryLocation;
use App\Models\Warehouse;
use Illuminate\Contracts\Database\Query\Expression;
use Illuminate\Database\Query\Builder;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Entities\AmazonProduct;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;
use Throwable;


/**
 * Class ProductListingRepository
 * @package App\Repositories
 */
class ProductListingRepository extends AbstractRepository
{
    const TEMP_INVENTORY_TABLE = 'temp_inventory';

    private ?string $tempInventoryTableName = null;

    /**
     * @throws Throwable
     */
    public function saveWithRelations(#[DataCollectionOf(ProductListingDto::class)] DataCollection $data): Collection
    {
        $data = $data->toCollection();

        return DB::transaction(function () use ($data) {
            $productCollection = $data->pluck('product')->reject(fn ($product) => $product instanceof Optional);
            if ($productCollection->count() > 0) {
                $productCollection = app(ProductRepository::class)->saveWithRelations(ProductDto::collection($data->pluck('product')));
            }

            $productListingCollection = $data->map(function ($productListing) use ($productCollection) {
                $productListingModel = new ProductListing($productListing->toArray());
                // If product collection has items, they are newly created, otherwise the data must already have a product id
                if ($productCollection->count() > 0)
                {
                    $productListingModel->product_id = $productCollection->where('sku', $productListing->product->sku)->first()['id'];
                }
                return ProductListingDto::from(new ProductListing($productListingModel->toArray()));
            });

            return $this->save($productListingCollection, ProductListing::class);
        });
    }

    /**
     * @param  int  $salesChannelId
     * @param  array  $salesChannelListingIds
     * @return Collection
     */
    public function getFromSalesChannelIdForSalesChannelListingIds(int $salesChannelId, array $salesChannelListingIds): Collection
    {
        $productListingsCollection = collect();
        foreach (array_chunk($salesChannelListingIds, 10000) as $salesChannelListingIdsChunk) {
            $productListingsCollection = $productListingsCollection->merge(
                ProductListing::query()
                    ->where('sales_channel_id', $salesChannelId)
                    ->whereIn('sales_channel_listing_id', $salesChannelListingIdsChunk)
                    ->get()
            );
        }
        return $productListingsCollection;
    }

    /**
     * @param  int  $salesChannelId
     * @param  array  $uniqueIds
     * @return Collection
     */
    public function getFromSalesChannelIdFromUniqueSalesChannelProductIds(int $salesChannelId, array $uniqueIds): Collection
    {
        $productListingsCollection = collect();
        foreach (array_chunk($uniqueIds, 10000) as $uniqueIdsChunk) {
            $productListingsCollection = $productListingsCollection->merge(
                ProductListing::query()
                    ->where('sales_channel_id', $salesChannelId)
                    ->whereIn('sales_channel_listing_id', $uniqueIdsChunk)
                    ->get()
            );
        }
        return $productListingsCollection;
    }

    /**
     * @param  array  $ids
     * @return void
     */
    public function deleteProductListingsForIds(array $ids): void
    {
        $chunkSize = 10000;

        foreach (array_chunk($ids, $chunkSize) as $chunk) {
            ProductListing::whereIn('id', $chunk)->delete();
        }
    }

    /**
     * @param  DataCollection  $data
     * @return void
     */
    public function updateInventorySyncedForIds(DataCollection $data): void
    {
        batch()->update(new ProductListing(), $data->toArray(), 'id');
    }

    /**
     * @param  AbstractSalesChannelIntegrationInstance|IntegrationInstance  $integrationInstance
     * @param  array|null  $productIds
     * @return void
     */
    public function cacheProductListingQuantity(
        AbstractSalesChannelIntegrationInstance|IntegrationInstance $integrationInstance,
        ?array $productIds = []
    ): void{

        $inventoryRules = $integrationInstance->getInventoryData();

        // Non-bundled products
        $this->updateListingQuantityForNonBundledProducts($integrationInstance, $inventoryRules, $productIds);

        // Bundled products
        $this->updateListingQuantityForBundledProducts($integrationInstance, $inventoryRules, $productIds);

    }

    /**
     * @param  AbstractSalesChannelIntegrationInstance|IntegrationInstance $integrationInstance
     * @param  array|null  $productIds
     * @return void
     */
    public function cacheListingQuantityForInventoryLocations(
        AbstractSalesChannelIntegrationInstance|IntegrationInstance $integrationInstance,
        ?array $productIds = []
    ): void
    {

        $inventoryRules = $integrationInstance->getInventoryData();

        if($inventoryRules->locations instanceof Optional){
            return;
        }

        // Seed inventory locations.
        $this->seedInventoryLocations($integrationInstance, $inventoryRules, $productIds);

        // Update the quantity for each location.
        /** @var InventoryLocationData $location */
        foreach ($inventoryRules->locations as $location) {
            if (!isset($location->id) || ($location->masterOfStock ?? '') !== ProductListing::MASTER_SKU) {
                continue;
            }

            $this->updateLocationListingQuantityForNonBundledProducts(
                location: $location,
                integrationInstance: $integrationInstance,
                productIds: $productIds
            );

            $this->updateLocationListingQuantityForBundledProducts(
                location: $location,
                integrationInstance: $integrationInstance,
                inventoryRules: $inventoryRules,
                productIds: $productIds
            );

            $this->clearTempInventoryTable();
        }
    }

    /**
     * @param  AbstractSalesChannelIntegrationInstance|IntegrationInstance  $integrationInstance
     * @param  array|null  $productIds
     * @return void
     */
    public function cacheProductListingQuantityFromInventoryLocations(
        AbstractSalesChannelIntegrationInstance|IntegrationInstance $integrationInstance,
        ?array $productIds = []
    ): void
    {
        ProductListing::query()
            ->where('sales_channel_id', $integrationInstance->salesChannel->id)
            ->when(!empty($productIds), function ($query) use ($productIds) {
                $query->whereIn('product_id', $productIds);
            })
            ->join('product_listing_inventory_locations as plil', 'plil.product_listing_id', '=', 'product_listings.id')
            ->groupBy('product_listings.id')
            ->update([
                'product_listings.quantity' => DB::raw('(
                    SELECT SUM(plil.quantity) 
                    FROM product_listing_inventory_locations AS plil 
                    WHERE plil.product_listing_id = product_listings.id
                    GROUP BY plil.product_listing_id
                )')
            ]);

    }


    /**
     * @param  InventoryLocationData  $location
     * @param  AbstractSalesChannelIntegrationInstance|IntegrationInstance  $integrationInstance
     * @param  IntegrationInstanceInventoryData  $inventoryRules
     * @param  array|null  $productIds
     * @return void
     */
    private function updateLocationListingQuantityForNonBundledProducts(
        InventoryLocationData $location,
        AbstractSalesChannelIntegrationInstance|IntegrationInstance $integrationInstance,
        ?array $productIds = []
    ): void
    {
        $query = $this->getNonBundledProductsListingsQuery(
            $integrationInstance, $location, $productIds
        )
            ->join('product_listing_inventory_locations', function (JoinClause $join) use ($location) {
                $join->on('product_listing_id', '=', 'product_listings.id')
                    ->where('product_listing_inventory_locations.sales_channel_location_id', $location->id);
            });

        $query->update([
            'product_listing_inventory_locations.quantity' => $this->getUpdateQuery(
                inventoryRules: $location,
                baseQuery: $query,
                integrationInstance: $integrationInstance
            ),
        ]);
    }


    /**
     * @param  InventoryLocationData  $location
     * @param  AbstractSalesChannelIntegrationInstance|IntegrationInstance  $integrationInstance
     * @param  IntegrationInstanceInventoryData  $inventoryRules
     * @param  array|null  $productIds
     * @return void
     */
    private function updateLocationListingQuantityForBundledProducts(
        InventoryLocationData $location,
        AbstractSalesChannelIntegrationInstance|IntegrationInstance $integrationInstance,
        IntegrationInstanceInventoryData $inventoryRules,
        ?array $productIds = []
    ): void
    {

        $query = $this->getBundledProductsListingsQuery(
            $integrationInstance, $inventoryRules, $productIds
        )
        ->join('product_listing_inventory_locations', function (JoinClause $join) use ($location) {
            $join->on('product_listing_id', '=', 'product_listings.id')
                ->where('product_listing_inventory_locations.sales_channel_location_id', $location->id);
        });

        // Update the product listings with the calculated
        $query->update([
            'product_listing_inventory_locations.quantity' => $this->getUpdateQuery(
                inventoryRules: $location,
                baseQuery: $query,
                integrationInstance: $integrationInstance
            ),
        ]);

    }


    /**
     * @param  AbstractSalesChannelIntegrationInstance|IntegrationInstance  $integrationInstance
     * @param  IntegrationInstanceInventoryData  $inventoryRules
     * @param  array|null  $productIds
     * @return void
     */
    private function seedInventoryLocations(
        AbstractSalesChannelIntegrationInstance|IntegrationInstance $integrationInstance,
        IntegrationInstanceInventoryData $inventoryRules,
        ?array $productIds = []
    ): void
    {

        $locations = $inventoryRules->locations;

        /** @var InventoryLocationData $location */
        foreach ($locations as $location) {

            if (($location->masterOfStock ?? '') !== ProductListing::MASTER_SKU) {

                // Delete all inventory locations for this location.
                ProductListingInventoryLocation::query()
                    ->where('sales_channel_location_id', $location->id)
                    ->delete();

                continue;
            }

            $productListingQuery = ProductListing::query()
                ->where('sales_channel_id', $integrationInstance->salesChannel->id)
                ->when(! empty($productIds), function ($query) use ($productIds) {
                    $query->whereIn('product_listings.product_id', $productIds);
                });
            $productListingQuery
                ->leftJoin('product_listing_inventory_locations', function (JoinClause $join) use ($location) {
                    $join->on('product_listing_id', '=', 'product_listings.id')
                        ->where('product_listing_inventory_locations.sales_channel_location_id', $location->id);
                })
                ->whereNull('product_listing_inventory_locations.id')
                ->select('product_listings.*')
                ->chunk(1000, function ($listings) use ($location) {
                    $data = [];
                    foreach ($listings as $listing) {
                        $data[] = [
                            'sales_channel_location_id' => $location->id,
                            'product_listing_id' => $listing->id,
                            'inventory_rules' => json_encode($location),
                        ];
                    }

                    if (! empty($data)) {
                        $chunkData = collect($data);
                        ProductListingInventoryLocation::query()->whereNull('inventory_rules')->upsert(
                            $chunkData->toArray(),
                            ['sales_channel_location_id', 'product_listing_id'],
                            ['inventory_rules']
                        );
                    }
                });
        }
    }


    /**
     * @param  AbstractSalesChannelIntegrationInstance|IntegrationInstance  $integrationInstance
     * @param  IntegrationInstanceInventoryData  $inventoryRules
     * @param  array|null  $productIds
     * @return void
     */
    private function updateListingQuantityForNonBundledProducts(
        AbstractSalesChannelIntegrationInstance|IntegrationInstance $integrationInstance,
        IntegrationInstanceInventoryData $inventoryRules,
        ?array $productIds = []
    ): void
    {
        $query = $this->getNonBundledProductsListingsQuery($integrationInstance, $inventoryRules, $productIds);

        $query->update([
            'product_listings.quantity' => $this->getUpdateQuery(
                inventoryRules: $inventoryRules,
                baseQuery: $query,
                integrationInstance: $integrationInstance
            ),
        ]);

        // Clear the temporary inventory table
        $this->clearTempInventoryTable();
    }


    /**
     * @param  AbstractSalesChannelIntegrationInstance|IntegrationInstance  $integrationInstance
     * @param  IntegrationInstanceInventoryData|InventoryLocationData  $inventoryRules
     * @param  array|null  $productIds
     * @return Builder|\Illuminate\Database\Eloquent\Builder
     */
    private function getNonBundledProductsListingsQuery(
        AbstractSalesChannelIntegrationInstance|IntegrationInstance $integrationInstance,
        IntegrationInstanceInventoryData|InventoryLocationData $inventoryRules,
        ?array $productIds = []
    ): Builder | \Illuminate\Database\Eloquent\Builder
    {

        // Create a temporary table to store the available inventory for each product
        $this->loadProductInventoryIntoTemporaryTable(
            inventoryRules: $inventoryRules,
            productIds: $productIds
        );

        return DB::query()->from($this->tempInventoryTableName, 'products_inventory')
            ->join('product_listings', 'product_listings.product_id',
                'products_inventory.product_id')
            ->join('products', 'product_listings.product_id',
                'products.id')
            ->where('product_listings.sales_channel_id', $integrationInstance->salesChannel->id)
            ->where('products.type', '!=', Product::TYPE_BUNDLE)
            ->when(! empty($productIds), function ($query) use ($productIds) {
                $query->whereIn('product_listings.product_id', $productIds);
            })
            ->groupBy('products_inventory.product_id')
            ->select(
                [
                    'product_listings.product_id',
                    'product_listings.inventory_rules',
                    DB::raw('coalesce(SUM(products_inventory.inventory_available), 0) inventory_available'),
                ]
            );
    }


    /**
     * @param  AbstractSalesChannelIntegrationInstance|IntegrationInstance  $integrationInstance
     * @param  IntegrationInstanceInventoryData  $inventoryRules
     * @param  array|null  $productIds
     * @return void
     */
    private function updateListingQuantityForBundledProducts(
        AbstractSalesChannelIntegrationInstance|IntegrationInstance $integrationInstance,
        IntegrationInstanceInventoryData $inventoryRules,
        ?array $productIds = []
    ): void
    {
        $query = $this->getBundledProductsListingsQuery(
            $integrationInstance, $inventoryRules, $productIds
        );

        // Update the product listings with the calculated quantity
        $query->update([
            'product_listings.quantity' => $this->getUpdateQuery(
                inventoryRules: $inventoryRules,
                baseQuery: $query,
                integrationInstance: $integrationInstance
            ),
        ]);

        // Clear the temporary inventory table
        $this->clearTempInventoryTable();
    }


    /**
     * @param  AbstractSalesChannelIntegrationInstance|IntegrationInstance  $integrationInstance
     * @param  IntegrationInstanceInventoryData  $inventoryRules
     * @param  array|null  $productIds
     * @return Builder|\Illuminate\Database\Eloquent\Builder
     */
    private function getBundledProductsListingsQuery(
        AbstractSalesChannelIntegrationInstance|IntegrationInstance $integrationInstance,
        IntegrationInstanceInventoryData $inventoryRules,
        ?array $productIds = []
    ): Builder | \Illuminate\Database\Eloquent\Builder
    {

        // Create a temporary table to store the available inventory for each product
        $this->loadProductInventoryIntoTemporaryTable(
            inventoryRules: $inventoryRules,
            productIds: $productIds
        );

        $query = DB::query()->from('product_listings')
            ->join('products', 'product_listings.product_id', '=', 'products.id')
            ->join('product_components', 'product_components.parent_product_id', '=', 'product_listings.product_id')
            ->join($this->tempInventoryTableName . ' as products_inventory', 'products_inventory.product_id', '=', 'product_components.component_product_id')
            ->where('product_listings.sales_channel_id', $integrationInstance->salesChannel->id)
            ->where('products.type', Product::TYPE_BUNDLE)
            ->when(! empty($productIds), function ($query) use ($productIds) {
                $query->whereIn('product_listings.product_id', $productIds);
            })
            ->groupBy('product_listings.product_id', 'product_components.component_product_id',
                'product_components.quantity')
            ->select([
                'product_listings.product_id',
                'product_components.component_product_id',
                'product_listings.inventory_rules',
                DB::raw('cast(product_components.quantity as float) AS child_bundle_quantity'),
                DB::raw('SUM(IFNULL(products_inventory.inventory_available, 0)) AS child_inventory_available'),
            ]);

        // Get the minimum available inventory for each product and warehouse
        $query = DB::query()
            ->fromSub($query, 'sq')
            ->groupBy('product_id')
            ->select([
                'product_id',
                'inventory_rules',
                DB::raw('MIN(FLOOR(child_inventory_available / child_bundle_quantity)) inventory_available'),
            ]);

        // Get the sum of the minimum available inventory for each product
        return DB::query()->from('product_listings')
            ->joinSub($query, 'results', 'product_listings.product_id', '=', 'results.product_id')
            ->groupBy('product_listings.product_id')
            ->select(
                [
                    'product_listings.product_id',
                    'product_listings.inventory_rules',
                    DB::raw('SUM(results.inventory_available) inventory_available'),
                ]
            );
    }


    private function loadProductInventoryIntoTemporaryTable(
        IntegrationInstanceInventoryData|InventoryLocationData $inventoryRules,
        array $productIds = []
    ){
        $nonFBAInventory = DB::table('products_inventory')
            ->join('warehouses', 'products_inventory.warehouse_id', 'warehouses.id')
            ->whereNotIn('warehouses.type', [Warehouse::TYPE_AMAZON_FBA, Warehouse::TYPE_SUPPLIER])
            ->when(!empty($productIds), function($query) use ($productIds){
                $query->whereIn('products_inventory.product_id', $productIds);
            })
            ->when(!$inventoryRules->selectedWarehouses instanceof Optional, function ($query) use ($inventoryRules) {
                $query->whereIn('products_inventory.warehouse_id', $inventoryRules->selectedWarehouses);
            })
            ->select([
                'products_inventory.product_id',
                'products_inventory.warehouse_id',
                'products_inventory.inventory_available',
                'products_inventory.inventory_in_transit'
            ]);


        // Amazon FBA Warehouses
        $fbaInventory = DB::table('warehouses')
            ->selectRaw('
                products.id as product_id,
                warehouses.id as warehouse_id,
                cast(coalesce(sum(afri.afn_fulfillable_quantity), 0) as signed) as inventory_available,
                cast(0 as unsigned) as inventory_in_transit
            ')
            ->join('amazon_fba_report_inventory as afri', 'afri.integration_instance_id', 'warehouses.integration_instance_id')
            ->join('amazon_products', function(JoinClause $q){
                $q->on('amazon_products.integration_instance_id', 'warehouses.integration_instance_id')
                    ->whereColumn('amazon_products.seller_sku', 'afri.sku');
            })
            ->join('product_listings', function(JoinClause $q){
                $q->on('product_listings.document_id', 'amazon_products.id')
                    ->where('product_listings.document_type', AmazonProduct::class);
            })
            ->join('products', 'products.id', 'product_listings.product_id')
            ->where('warehouses.type', Warehouse::TYPE_AMAZON_FBA)
            ->when(!empty($productIds), function($query) use ($productIds){
                $query->whereIn('products.id', $productIds);
            })
            ->when(!$inventoryRules->selectedWarehouses instanceof Optional, function ($query) use ($inventoryRules) {
                $query->whereIn('warehouses.id', $inventoryRules->selectedWarehouses);
            })
            ->groupBy('products.id');

        $this->clearTempInventoryTable();

        $this->setTempInventoryTableName();

        DB::statement('CREATE TEMPORARY TABLE '. $this->tempInventoryTableName . ' AS ' .
            DB::table(DB::raw("({$nonFBAInventory->union($fbaInventory)->toSql()}) as subquery"))
                ->select('product_id', DB::raw('SUM(inventory_available) as inventory_available'))
                ->groupBy('product_id')
                ->toSql(),
            $nonFBAInventory->getBindings()
        );
    }

    private function setTempInventoryTableName(): void {
        $this->tempInventoryTableName = self::TEMP_INVENTORY_TABLE . '_' . uniqid();
    }


    /**
     * @param  IntegrationInstanceInventoryData|InventoryLocationData  $inventoryRules
     * @param  \Illuminate\Database\Eloquent\Builder|Builder  $baseQuery
     * @param  IntegrationInstance  $integrationInstance
     * @return \Illuminate\Database\Query\Expression|Expression
     */
    public function getUpdateQuery(
        IntegrationInstanceInventoryData|InventoryLocationData $inventoryRules,
        \Illuminate\Database\Eloquent\Builder|Builder $baseQuery,
        IntegrationInstance $integrationInstance,
    ): \Illuminate\Database\Query\Expression|Expression
    {
        if(!$this->inventoryRulesExists($inventoryRules) && $this->noListingHasRuleOverride($baseQuery)){
            // There are no inventory rules, not on the integration instance or on none of the product listings.
            // We can just update the quantity to the available inventory.
            return DB::raw("inventory_available");
        } else {
            // There are inventory rules, either on integration instance or on some product listings.
            // We need to calculate the quantity based on the rules.

            if($inventoryRules instanceof InventoryLocationData && $integrationInstance->hasInventoryLocations()){

                $locationIndex = 0;
                foreach ($integrationInstance->getInventoryData()->locations as $key => $location){
                    if($location->id === $inventoryRules->id){
                        $locationIndex = $key;
                        break;
                    }
                }

                $listingInventoryQuery = "JSON_UNQUOTE(JSON_EXTRACT(product_listings.inventory_rules, CONCAT('$.locations[', $locationIndex, '].inventoryModificationRules')))";
            } else {
                $listingInventoryQuery = 'JSON_UNQUOTE(JSON_EXTRACT(product_listings.inventory_rules, "$.inventoryModificationRules"))';
            }

            $inventoryRules = json_encode($inventoryRules->inventoryModificationRules);

            return DB::raw(
                "GREATEST(0, 
                    getProductListingQuantity(
                        $listingInventoryQuery, 
                        '$inventoryRules', 
                        `inventory_available`
                    )
                )"
            );
        }

    }

    /**
     * @param  IntegrationInstanceInventoryData|InventoryLocationData  $inventoryRules
     * @return bool
     */
    private function inventoryRulesExists(IntegrationInstanceInventoryData|InventoryLocationData $inventoryRules): bool
    {
        try{
            if ($inventoryRules->inventoryModificationRules instanceof Optional) {
                return false;
            }
            return ($inventoryRules->inventoryModificationRules->maxRuleType &&
                $inventoryRules->inventoryModificationRules->maxRuleTypeValue !== 'None') ||
                (isset($inventoryRules->inventoryModificationRules->minRuleType) &&
                    $inventoryRules->inventoryModificationRules->minRuleTypeValue !== 'None') ||
                (isset($inventoryRules->inventoryModificationRules->substractBufferStock) &&
                    ((float)$inventoryRules->inventoryModificationRules->substractBufferStock) > 0);
        }catch (Throwable $e){
            return false;
        }
    }

    /**
     * @param  \Illuminate\Database\Eloquent\Builder|Builder  $baseQuery
     * @return bool
     */
    private function noListingHasRuleOverride(\Illuminate\Database\Eloquent\Builder|Builder $baseQuery): bool
    {
        return $baseQuery
            ->clone()
            ->whereNotNull('product_listings.inventory_rules')
            ->doesntExist();
    }

    private function clearTempInventoryTable(): void
    {
        if(is_null($this->tempInventoryTableName)){
            return;
        }
        DB::statement('DROP TEMPORARY TABLE IF EXISTS ' . $this->tempInventoryTableName);
        $this->tempInventoryTableName = null;
    }

}
