<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Models\Currency;
use App\Models\IntegrationInstance;
use App\Models\Payment;
use App\Models\PaymentType;
use App\Models\SalesOrder;
use Illuminate\Support\Collection;

class PaymentRepository extends AbstractRepository
{
    public function saveBulk(Collection $data, IntegrationInstance $integrationInstance, bool $insert, bool $update): array
    {
        $tempTable = Payment::dataFeedBulkImport([
            'data_to_import' => [
                'type' => 'json',
                'data' => $data->toJson(),
            ],
            'insert' => $insert,
            'update' => $update,
            'purgeDatabaseConnection' => false,
            'mappings' => [
                [
                    'expected_column_name' => 'sales_order_number',
                    'data_column_name' => 'sales_order_number',
                    'expected_column_type' => 'string',
                    'is_importable' => false,
                ],
                [
                    'expected_column_name' => 'payment_date',
                    'data_column_name' => 'payment_date',
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'currency_code',
                    'data_column_name' => 'currency_code',
                    'expected_column_type' => 'string',
                    'is_importable' => false,
                ],
                [
                    'expected_column_name' => 'payment_type',
                    'data_column_name' => 'payment_type',
                    'expected_column_type' => 'string',
                    'is_importable' => false,
                ],
                [
                    'expected_column_name' => 'amount',
                    'data_column_name' => 'amount',
                    'expected_column_type' => 'decimal',
                ],
                [
                    'joins' => [
                        [
                            'table' => (new PaymentType())->getTable(),
                            'type' => 'join',
                            'on' => [
                                [
                                    'database_column' => 'name',
                                    'stage_column_to_join' => 'payment_type',
                                ],
                            ],
                            'column_type' => 'integer',
                            'add_select' => [
                                [
                                    'expected_column_name' => 'payment_type_id',
                                    'data_column_name' => 'id',
                                    'expected_column_type' => 'integer',
                                    'is_importable' => true,
                                ],
                            ],
                        ],
                    ],
                ],
                [
                    'joins' => [
                        [
                            'table' => (new SalesOrder())->getTable(),
                            'type' => 'leftJoin',
                            'on' => [
                                [
                                    'database_column' => 'sales_order_number',
                                    'stage_column_to_join' => 'sales_order_number',
                                ],
                                [
                                    'database_column' => 'sales_channel_id',
                                    'stage_column_to_join' => 'sales_channel_id',
                                ],
                            ],
                            'column_type' => 'integer',
                            'add_select' => [
                                [
                                    'expected_column_name' => 'link_id',
                                    'data_column_name' => 'id',
                                    'expected_column_type' => 'integer',
                                    'is_importable' => true,
                                ],
                                [
                                    'expected_column_name' => 'order_status',
                                    'data_column_name' => 'order_status',
                                    'expected_column_type' => 'string',
                                    'is_importable' => false,
                                ],
                            ],
                        ],
                    ],
                ],
                [
                    'joins' => [
                        [
                            'table' => (new Currency())->getTable(),
                            'type' => 'leftJoin',
                            'on' => [
                                [
                                    'database_column' => 'code',
                                    'stage_column_to_join' => 'currency_code',
                                ],
                            ],
                            'column_type' => 'integer',
                            'add_select' => [
                                [
                                    'expected_column_name' => 'currency_id',
                                    'data_column_name' => 'id',
                                    'expected_column_type' => 'integer',
                                    'is_importable' => true,
                                ],
                                [
                                    'expected_column_name' => 'currency_rate',
                                    'data_column_name' => 'conversion',
                                    'expected_column_type' => 'decimal',
                                    'is_importable' => true,
                                ],
                            ],
                        ],
                    ],
                ],
            ],

            'default_columns' => [
                [
                    'expected_column_name' => 'link_type',
                    'default_value' => getClassNameForBulkImport(new SalesOrder()),
                    'expected_column_type' => 'string',
                ],
                [
                    'expected_column_name' => 'updated_at',
                    'default_value' => now(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'sales_channel_id',
                    'default_value' => $integrationInstance->salesChannel->id,
                    'expected_column_type' => 'integer',
                    'is_importable' => false,
                ],
                [
                    'expected_column_name' => 'currency_id_tenant_snapshot',
                    'default_value' => Currency::default()->id,
                    'expected_column_type' => 'integer',
                ],
            ],
            'unique_by_columns' => [
                'link_id',
                'link_type',
            ],
        ]);

        return [
            'status' => true,
            'temp_table' => $tempTable,
        ];
    }
}
