<?php

namespace App\Repositories;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Enums\IntegrationInstanceSyncStatusEnum;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Modules\Qbo\Entities\QboIntegrationInstance;
use Modules\Xero\Entities\XeroIntegrationInstance;

class IntegrationInstanceRepository
{
    public function getById(int $id): IntegrationInstance|Model
    {
        return IntegrationInstance::query()->where('id', $id)->firstOrFail();
    }

    public function initializeSettings(array $settings, bool $salesChannelOnly = false): void
    {
        $query = IntegrationInstance::query();

        if ($salesChannelOnly) {
            $query
                ->whereHas('salesChannel')
                ->where('name', '!=', 'SKU.io');
        }

        $query->each(function (IntegrationInstance $integrationInstance) use ($settings) {
            $integrationSettings = $integrationInstance->integration_settings;

            foreach ($settings as $setting) {
                if (! isset($integrationSettings[$setting])) {
                    $integrationSettings[$setting] = null;
                }
            }

            $integrationInstance->integration_settings = $integrationSettings;
            $integrationInstance->save();
        });
    }

    public function deInitializeSettings(array $settings, bool $salesChannelOnly = false): void
    {
        $query = IntegrationInstance::query();

        if ($salesChannelOnly) {
            $query
                ->whereHas('salesChannel')
                ->where('name', '!=', 'SKU.io');
        }

        $query->each(function (IntegrationInstance $integrationInstance) use ($settings) {
            $integrationSettings = $integrationInstance->integration_settings;

            foreach ($settings as $setting) {
                unset($integrationSettings[$setting]);
            }

            $integrationInstance->integration_settings = $integrationSettings;
            $integrationInstance->save();
        });
    }

    /**
     * @throws Exception
     */
    public function getAccountingInstance(): ?IntegrationInstance
    {
        if (! $accountingInstance = IntegrationInstance::query()
            ->whereHas('integration', function (Builder $query) {
                $query->where('integration_type', Integration::TYPE_ACCOUNTING);
            })
            ->first()) {
            return null;
        }

        return match ($accountingInstance->integration->name) {
            Integration::NAME_XERO => XeroIntegrationInstance::first(),
            Integration::NAME_QBO => QboIntegrationInstance::first(),
            default => throw new Exception('Accounting integration not implemented for '.$accountingInstance->integration->name),
        };
    }

    public function getQboAccountingInstance(): ?QboIntegrationInstance
    {
        if (! $accountingInstance = IntegrationInstance::query()
            ->whereHas('integration', function (Builder $query) {
                $query->where('integration_type', Integration::TYPE_ACCOUNTING)->where('name', Integration::NAME_QBO);
            })
            ->first()) {
            return null;
        }

        return match ($accountingInstance->integration->name) {
            Integration::NAME_XERO => XeroIntegrationInstance::first(),
            Integration::NAME_QBO => QboIntegrationInstance::first(),
            default => throw new Exception('Accounting integration not implemented for '.$accountingInstance->integration->name),
        };
    }

    public function getAutoFulfillmentShippingProviders(): Collection
    {
        return IntegrationInstance::with(['integration'])
            ->whereHas('integration', function (Builder $builder) {
                $builder->where('integration_type', 'shipping_provider');
            })
            ->get()
            ->map(function (IntegrationInstance $instance) {
                $settings = is_string($instance->integration_settings) ? [] : $instance->integration_settings;

                if($instance->isVeracore() || $instance->isSMO()){
                    if(@$settings['automate_fulfillment']){
                        return [
                            [
                                'warehouse_id' => $settings['linked_warehouse_id'],
                                'integration' => [
                                    'instance_id' => $instance->id,
                                    'name' => $instance->integration->name,
                                ],
                            ],
                        ];
                    } else {
                        return false;
                    }
                }

                return collect((@$settings['fulfillment']['automatedWarehousesIds'] ?: []))
                    ->map(function ($warehouseId) use ($instance) {
                        return [
                            'warehouse_id' => $warehouseId,
                            'integration' => [
                                'instance_id' => $instance->id,
                                'name' => $instance->integration->name,
                            ],
                        ];
                    });
            })
            ->collapse();
    }

    public function getSalesChannelIntegrationInstances(): Collection
    {
        return IntegrationInstance::with(['integration'])
            ->whereHas('integration', function (Builder $builder) {
                $builder->where('integration_type', 'sales_channel');
            })
            ->whereHas('salesChannel')
            ->get();
    }

    public function updateSyncStatus(IntegrationInstance $integrationInstance, IntegrationInstanceSyncStatusEnum $syncStatus): IntegrationInstance
    {
        $integrationInstance->sync_status = $syncStatus;
        $integrationInstance->is_automatic_sync_enabled = false;
        $integrationInstance->update();

        return $integrationInstance;
    }

    public function checkSkuIsMasterOfStock(IntegrationInstanceInterface $integrationInstance): bool
    {
        $settings = $integrationInstance->integration_settings;
        // TODO: May need override for Shopify which stores master of stock settings per location
        if (@$settings['inventory']['masterOfStock'] === 'sku.io') {
            return true;
        }
        return false;
    }
}
