<?php

namespace App\Repositories;

use App\DTO\FinancialLineTypeDto;
use App\DTO\ReportingDailyFinancialDto;
use App\Helpers;
use App\Models\FinancialLine;
use App\Models\FinancialLineType;
use App\Models\SalesOrder;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Spatie\QueryBuilder\QueryBuilder;

class FinancialLineTypeRepository
{
    public function save(FinancialLineTypeDto $financialLineTypeDto): FinancialLineType
    {
        /** @var FinancialLineType $financialLineType */
        $financialLineType = FinancialLineType::query()->updateOrCreate(['id' => $financialLineTypeDto->id], $financialLineTypeDto->toArray());

        return $financialLineType;
    }

    public function query(): EloquentCollection
    {
        return QueryBuilder::for(FinancialLineType::class)
            ->allowedFilters(['classification', 'name'])
            ->get();
    }

    public function getFinancialLineFinancialsSummaryForFinancialLineTypeDate(FinancialLineType $financialLineType, Carbon $date): ?ReportingDailyFinancialDto
    {
        $utcDate = $date->copy()->setTimezone('UTC');
        $data = FinancialLineType::query()
            ->selectRaw('
                SUM(financial_lines.quantity) as quantity,
                SUM(financial_lines.extended_amount) as '.$financialLineType->classification->value.'
            ')
            ->leftJoin('financial_lines', 'financial_lines.financial_line_type_id', '=', 'financial_line_types.id')
            ->join('sales_orders', 'sales_orders.id', '=', 'financial_lines.sales_order_id')
            ->where('sales_orders.order_status', '!=', SalesOrder::STATUS_DRAFT)
            ->where('financial_lines.financial_line_type_id', $financialLineType->id)
            ->whereRaw('DATE(CONVERT_TZ(sales_orders.order_date, "UTC", "'.Helpers::getAppTimezone().'")) = "'.$utcDate->toDateString().'"')
            ->whereNull('financial_lines.allocate_to_id') // Only include non allocated financial lines to avoid double-counting
            ->groupBy('financial_line_types.id')
            ->first();

        if (! $data) {
            return null;
        }

        $reportingDailyFinancialDto = ReportingDailyFinancialDto::from($data);

        $reportingDailyFinancialDto->date = $date->toDateTimeString();
        $reportingDailyFinancialDto->reportable_id = $financialLineType->id;
        $reportingDailyFinancialDto->reportable_type = FinancialLineType::class;

        return $reportingDailyFinancialDto;
    }

    public function unallocatedLinesWithoutDailyFinancialQuery(string $userTimezone): Builder
    {
        return FinancialLineType::with('financialLines.salesOrder')
            ->select('financial_line_types.id', 'sales_orders.order_date')
            ->leftJoin('financial_lines', 'financial_lines.financial_line_type_id', '=', 'financial_line_types.id')
            ->join('sales_orders', 'sales_orders.id', '=', 'financial_lines.sales_order_id')
            ->leftJoin('reporting_daily_financials', function ($join) use ($userTimezone) {
                $join->on('reporting_daily_financials.reportable_id', '=', 'financial_line_types.id')
                    ->where('reporting_daily_financials.reportable_type', FinancialLineType::class)
                    ->whereRaw("CONVERT_TZ(DATE(CONVERT_TZ(sales_orders.order_date, 'UTC', '$userTimezone')), '$userTimezone', 'UTC') = reporting_daily_financials.date");
            })
            // No allocatable exists from this financial line (aka, it has not been allocated)
            ->leftJoin('financial_allocatables', function ($join) {
                $join->on('financial_allocatables.allocatable_from_id', '=', 'financial_lines.id')
                    ->where('financial_allocatables.allocatable_from_type', FinancialLine::class);
            })
            ->where('sales_orders.order_status', '!=', SalesOrder::STATUS_DRAFT)
            ->whereNull('financial_allocatables.id')
            ->whereNull('reporting_daily_financials.id');
    }

    public function getOrCreateFinancialLineType()
    {

    }
}
