<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Enums\FinancialAllocatableTypeEnum;
use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Models\FinancialAllocatable;
use App\Models\FinancialLine;
use App\Models\FinancialLineType;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Spatie\QueryBuilder\QueryBuilder;

class FinancialLineRepository extends AbstractRepository
{
    public function query(SalesOrder $salesOrder): EloquentCollection
    {
        return QueryBuilder::for(FinancialLine::class)
            ->with('financialLineType:id,classification')
            ->where('sales_order_id', $salesOrder->id)
            ->allowedFilters(['sales_order_id', 'financialLineType.classification'])
            ->get();
    }

    public function getOrCreateFinancialLineType(string $name, FinancialLineClassificationEnum $classification, ?int $nominalCodeId = null, ?FinancialLineProrationStrategyEnum $prorationStrategy = null, bool $allocateToProducts = false): FinancialLineType
    {
        /** @var FinancialLineType $financialLineType */
        $financialLineType = FinancialLineType::query()->firstOrCreate([
            'name' => $name,
            'classification' => $classification,
            'proration_strategy' => $prorationStrategy ?? ($classification == FinancialLineClassificationEnum::REVENUE ?
                FinancialLineProrationStrategyEnum::REVENUE_BASED :
                FinancialLineProrationStrategyEnum::COST_BASED),
        ], [
            'nominal_code_id' => $nominalCodeId,
            'allocate_to_products' => $allocateToProducts,
        ]);

        return $financialLineType;
    }

    public function allocateLinesToSalesOrderLines(Collection $salesOrders, Collection $financialsDataCollection): array
    {
        $salesOrderLineAllocationTotals = [];

        $salesOrders->each(function (SalesOrder $salesOrder) use ($financialsDataCollection, &$salesOrderLineAllocationTotals) {
            $financialLines = $salesOrder->financialLines()->allocatable()->get();

            if ($financialLines->isEmpty()) {
                return [];
            }

            $salesOrderLineTotals = $salesOrder->getSalesOrderLineTotals();

            $prorationStrategies = [
                FinancialLineProrationStrategyEnum::REVENUE_BASED->value => fn ($line, $financialLine) => safeDivide($financialLine->revenue, $salesOrderLineTotals->revenue),
                FinancialLineProrationStrategyEnum::COST_BASED->value => fn ($line, $financialLine) => safeDivide($financialLine->cogs, $salesOrderLineTotals->cost),
                FinancialLineProrationStrategyEnum::WEIGHT_BASED->value => fn ($line, $financialLine) => safeDivide($line->weight_extended, $salesOrderLineTotals->weight),
                FinancialLineProrationStrategyEnum::VOLUME_BASED->value => fn ($line, $financialLine) => safeDivide($line->volume_extended, $salesOrderLineTotals->volume),
                FinancialLineProrationStrategyEnum::QUANTITY_BASED->value => fn ($line, $financialLine) => safeDivide($line->quantity, $salesOrderLineTotals->quantity),
            ];

            $allocations = [];

            foreach ($salesOrder->salesOrderLines as $salesOrderLine) {
                $salesOrderFinancialLine = $financialsDataCollection->firstWhere('sales_order_line_id', $salesOrderLine->id);
                foreach ($financialLines as $financialLine) {
                    // If financial line is to be allocated to a specific line, skip if not the current line
                    if ($financialLine->proration_strategy === FinancialLineProrationStrategyEnum::SPECIFIC_LINE) {
                        if ($financialLine->allocate_to_id == $salesOrderLine->id && $financialLine->allocate_to_type == SalesOrderLine::class) {
                            $proration = 1;
                        } else {
                            continue;
                        }
                    } else {
                        customlog('5820', 'Calculating proration for financial line '.$financialLine->id, $salesOrderFinancialLine->toArray());
                        $proration = $prorationStrategies[$financialLine->proration_strategy->value]($salesOrderLine, $salesOrderFinancialLine);
                    }

                    $classification = $financialLine->financialLineType->classification;
                    $financialLineType = $classification == FinancialLineClassificationEnum::REVENUE ?
                        FinancialAllocatableTypeEnum::FINANCIAL_LINE_REVENUE :
                        FinancialAllocatableTypeEnum::FINANCIAL_LINE_COST;
                    $allocations[] = [
                        'allocatable_type' => $financialLineType,
                        'allocatable_to_id' => $salesOrderLine->id,
                        'allocatable_to_type' => SalesOrderLine::class,
                        'allocatable_from_id' => $financialLine->id,
                        'allocatable_from_type' => FinancialLine::class,
                        'amount' => $proration * $financialLine->extended_amount_in_tenant_currency,
                    ];
                    uninitializedArrayToZero($salesOrderLineAllocationTotals, $salesOrderLine->id, $classification->value);
                    $salesOrderLineAllocationTotals[$salesOrderLine->id][$classification->value] += $proration * $financialLine->extended_amount_in_tenant_currency;

                    // Sync allocations
                    FinancialAllocatable::query()->where('allocatable_to_id', $salesOrderLine->id)
                        ->where('allocatable_to_type', SalesOrderLine::class)
                        ->delete();
                }
            }
            FinancialAllocatable::query()->insert($allocations);
        });

        return $salesOrderLineAllocationTotals;
    }

    // Query total revenue/cost of a financial line for a sales order
    public function getSalesOrderFinancialLineTotal(SalesOrder $salesOrder, string $financialLineType, FinancialLineClassificationEnum $classification): float
    {
        return $salesOrder
            ->financialLines()
            ->whereHas('financialLineType', function (Builder $builder) use ($financialLineType, $classification) {
                $builder->where('name', $financialLineType);
                $builder->where('classification', $classification->value);
            })
            ->selectRaw('SUM(quantity * amount) as total')
            ->first()
            ->total ?? 0;
    }

    public function saveBulk(Collection $data, IntegrationInstance $integrationInstance, bool $insert, bool $update, array $metadata): array
    {
        $tempTable = FinancialLine::dataFeedBulkImport([
            'data_to_import' => [
                'type' => 'json',
                'data' => $data->toJson(),
            ],
            'insert' => $insert,
            'update' => $update,
            'purgeDatabaseConnection' => false,
            'mappings' => [
                [
                    'expected_column_name' => 'sales_order_number',
                    'data_column_name' => 'sales_order_number',
                    'expected_column_type' => 'string',
                    'is_importable' => false,
                ],
                [
                    'expected_column_name' => 'sales_order_id',
                    'data_column_name' => 'sales_order_id',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'financial_line_type_id',
                    'data_column_name' => 'financial_line_type_id',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'description',
                    'data_column_name' => 'description',
                    'expected_column_type' => 'string',
                ],
                [
                    'expected_column_name' => 'quantity',
                    'data_column_name' => 'quantity',
                    'expected_column_type' => 'decimal',
                ],
                [
                    'expected_column_name' => 'amount',
                    'data_column_name' => 'amount',
                    'expected_column_type' => 'decimal',
                ],
                [
                    'expected_column_name' => 'tax_allocation',
                    'data_column_name' => 'tax_allocation',
                    'expected_column_type' => 'decimal',
                ],
                [
                    'expected_column_name' => 'allocate_to_products',
                    'data_column_name' => 'allocate_to_products',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'proration_strategy',
                    'data_column_name' => 'proration_strategy',
                    'expected_column_type' => 'string',
                ],
                [
                    'expected_column_name' => 'allocate_to_id',
                    'data_column_name' => 'allocate_to_id',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'sales_channel_line_id',
                    'data_column_name' => 'sales_channel_line_id',
                    'expected_column_type' => 'integer',
                ],
                [
                    'joins' => [
                        [
                            'table' => (new SalesOrder())->getTable(),
                            'type' => 'leftJoin',
                            'on' => [
                                [
                                    'database_column' => 'sales_order_number',
                                    'stage_column_to_join' => 'sales_order_number',
                                ],
                                [
                                    'database_column' => 'sales_channel_id',
                                    'stage_column_to_join' => 'sales_channel_id',
                                ],
                            ],
                            'column_type' => 'integer',
                            'add_select' => [
                                [
                                    'expected_column_name' => 'sales_order_id',
                                    'data_column_name' => 'id',
                                    'expected_column_type' => 'integer',
                                    'is_importable' => true,
                                ],
                            ],
                        ],
                    ],
                ],
            ],

            'default_columns' => [
                [
                    'expected_column_name' => 'created_at',
                    'default_value' => now(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'updated_at',
                    'default_value' => now(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'sales_channel_id',
                    'default_value' => $integrationInstance->salesChannel->id,
                    'expected_column_type' => 'integer',
                    'is_importable' => false,
                ],
            ],
            'unique_by_columns' => [
                'sales_order_id',
            ],
        ]);

        return [
            'status' => true,
            'temp_table' => $tempTable,
        ];
    }
}
