<?php

namespace App\Repositories;

use App\Collections\FifoLayerCollection;
use App\DTO\BulkImportResponseDto;
use App\DTO\FifoLayerDto;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryAssemblyLine;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\Warehouse;
use App\Models\WarehouseTransferLine;
use App\Services\InventoryManagement\PositiveInventoryEvent;
use Exception;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Throwable;

class FifoLayerRepository
{
    public function recalculateTotalCosts(array $ids = [], $parameters = []): void
    {
        //        $fifoLayers = FifoLayer::query()
        //            ->where('link_type', '!=', 'initial_cost')
        //            ->where('link_type', '!=', InventoryAssemblyLine::class);

        // Only recalculation for purchase order receipt based fifo layers
        $fifoLayers = FifoLayer::query();

        if (@$parameters['withDiscountsOnly'] || @$parameters['withNonDefaultCurrencyOnly'])
        {
            $fifoLayers->whereHasMorph('link', PurchaseOrderShipmentReceiptLine::class, function ($query) use ($parameters) {
                $query->whereHas('purchaseOrderShipmentLine', function ($query) use ($parameters) {
                    $query->whereHas('purchaseOrderLine', function ($query) use ($parameters) {
                        if (@$parameters['withDiscountsOnly']) {
                            $query->where('discount_amount', '!=', 0);
                        }
                        $query->whereHas('purchaseOrder', function ($query) use ($parameters) {
                            if (@$parameters['withNonDefaultCurrencyOnly']) {
                                $query->whereHas('currency', function ($query) {
                                    $query->where('is_default', 0);
                                });
                            }
                        });
                    });
                });
            });
        }


        if (! empty($ids)) {
            $fifoLayers->whereIn('id', $ids);
        }

        if (@$parameters['startDate']) {
            $fifoLayers->whereDate('fifo_layer_date', '>=', $parameters['startDate']);
        }

        if (@$parameters['endDate']) {
            $fifoLayers->whereDate('fifo_layer_date', '<=', $parameters['endDate']);
        }

        customlog('fifo_layer_recalculations', 'There are '.$fifoLayers->count().' fifo layers to recalculate');

        $fifoLayers = $fifoLayers->cursor();

        $i = 0;

        $fifoLayers->each(/**
         * @throws OversubscribedFifoLayerException
         * @throws Exception
         * @throws Throwable
         */ function (FifoLayer $fifoLayer) use (&$i) {
            $i++;

            if ($i % 1000 === 0) {
                customlog('fifo_layer_recalculations', 'Recalculated '.$i.' fifo layers');
            }

            DB::transaction(function () use ($fifoLayer, &$i) {

                /** @var PositiveInventoryEvent $link */
                $link = $fifoLayer->link;

                $fifoLayer->total_cost = $link->getUnitCost() * $fifoLayer->original_quantity;

                $isDirty = $fifoLayer->isDirty();

                $fifoLayer->save();

                /*
                * Handle usages by making sure the implication of a FIFO layer update gets propagated down the line
                */
                if ($isDirty) {
                    $this->updateUsagesForFifoLayer($fifoLayer);
                }
            });
        });
    }

    public function updateUsagesForFifoLayer(FifoLayer $fifoLayer): void
    {
        $fifoLayer->inventoryMovements()
            ->where('quantity', '<', 0)
            ->where('link_type', '!=', WarehouseTransferLine::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)->each(function (InventoryMovement $inventoryMovement) use ($fifoLayer) {
                $inventoryMovement->link->refreshFromFifoCostUpdate($fifoLayer->avg_cost);
            });
    }

    /**
     * @throws Throwable
     */
    public function save(FifoLayerDto $data): FifoLayer
    {
        /** @var FifoLayer $fifoLayer */
        $fifoLayer = FifoLayer::query()->updateOrCreate(['id' => $data->id], $data->toArray());

        return $fifoLayer;
    }

    public function getFifoLayersForIds(array $ids): EloquentCollection
    {
        return FifoLayer::query()
            ->whereIn('id', $ids)
            ->get();
    }

    public function getFifoLayersForTypeAndIds(string $fifoLayerType, array $ids): EloquentCollection
    {
        return FifoLayer::query()
            ->where('link_type', $fifoLayerType)
            ->whereIn('link_id', $ids)
            ->get();
    }

    public function saveBulk(FifoLayerCollection $data): void
    {
        FifoLayer::query()->upsert(
            $data->toArray(),
            ['id']
        );
    }

    public function getAvailableFifoLayersForProductWarehousePairs(Collection $productWarehousePairs, array $fifoLayerIdsBeingDeleted = []): EloquentCollection
    {
        $query = FifoLayer::query()
            ->where(function ($query) use ($productWarehousePairs) {
                foreach ($productWarehousePairs as $pair) {
                    $query->orWhere(function ($query) use ($pair) {
                        $query->where('product_id', $pair['product_id'])
                            ->where('warehouse_id', $pair['warehouse_id']);
                    });
                }
            })
            ->where('available_quantity', '>', 0);

        if (count($fifoLayerIdsBeingDeleted) > 0) {
            $query->whereNotIn('id', $fifoLayerIdsBeingDeleted);
        }

        return $query->get();
    }

    public function getAgingInventory(): EloquentCollection
    {
        return FifoLayer::query()
            ->join('warehouses', 'warehouses.id', '=', 'fifo_layers.warehouse_id')
            ->select(
                'fifo_layers.warehouse_id',
                'warehouses.name',
                DB::raw("CASE 
                            WHEN DATEDIFF(CURRENT_DATE, fifo_layer_date) BETWEEN 0 AND 30 THEN '0-30 days'
                            WHEN DATEDIFF(CURRENT_DATE, fifo_layer_date) BETWEEN 31 AND 90 THEN '31-90 days'
                            WHEN DATEDIFF(CURRENT_DATE, fifo_layer_date) BETWEEN 91 AND 180 THEN '91-180 days'
                            WHEN DATEDIFF(CURRENT_DATE, fifo_layer_date) BETWEEN 181 AND 365 THEN '181-365 days'
                            ELSE '365+ days'
                         END AS age_category"),
                DB::raw('SUM(available_quantity) AS quantity'),
                DB::raw('SUM((total_cost / original_quantity) * available_quantity) as valuation')
            )
            ->where('available_quantity', '>', 0)
            ->groupBy('fifo_layers.warehouse_id', 'age_category')
            ->orderBy('fifo_layers.warehouse_id')
            ->orderByRaw("CASE 
                            WHEN age_category = '0-30 days' THEN 1
                            WHEN age_category = '31-90 days' THEN 2
                            WHEN age_category = '91-180 days' THEN 3
                            WHEN age_category = '181-365 days' THEN 4
                            ELSE 5
                          END")
            ->get();
    }

    /*
     * Note: this relies on cache.  If cache is inaccurate, it could cause oversubscribed FIFO layers
     * Inventory adjustment movements that have links are not eligible to move because they strictly need to match
     * COGS on the positive inventory event they correspond to
     */
    public function getRelocatableInventoryMovements(FifoLayer $fifoLayer): Collection
    {
        return $fifoLayer->inventoryMovements()
            ->where('inventory_status', 'active')
            ->where('quantity', '<', 0)
            // Eligible usages are all usages except inventory adjustments that are linked.
            ->where(function ($query) {
                $query->where('link_type', '!=', InventoryAdjustment::class);
                $query->orWhere(function($query) {
                    $query->whereHasMorph('link', [InventoryAdjustment::class], function($query) {
                        $query->whereNull('link_id');
                    });
                });
            })
            ->orderByDesc('created_at')
            ->get();
    }

    /**
     * @throws OversubscribedFifoLayerException
     */
    public function validateFifoLayerCache(FifoLayer $fifoLayer): void
    {
        $fifoLayer->fulfilled_quantity = abs($fifoLayer->inventoryMovements()
            ->where('inventory_status', 'active')
            ->where('quantity', '<', 0)
            ->sum('quantity'));

        $fifoLayer->save(allowOverage: true);
    }
}
