<?php

namespace App\Providers;

use App\Models\Integration;
use Illuminate\Session\Store;
use Illuminate\Support\Facades\App;
use League\OAuth2\Client\Token\AccessTokenInterface;
use Webfox\Xero\Oauth2Provider;
use Webfox\Xero\OauthCredentialManager;

class OAuthStorageProvider implements OauthCredentialManager
{
    /** @var Oauth2Provider */
    protected $oauthProvider;

    /** @var Store */
    protected $session;

    /** @var Integration */
    protected $integration;

    public function __construct(Store $session)
    {
        $integration = Integration::with([])->where('name', Integration::NAME_XERO)->first();
        $instance = $integration->integrationInstances()->first();
        if ($instance) {
            config(['xero.oauth.client_id' => $instance->connection_settings ? $instance->connection_settings['clientId'] : null]);
            config(['xero.oauth.client_secret' => $instance->connection_settings ? $instance->connection_settings['clientSecret'] : null]);
        }

        $oauthProvider = App::make(Oauth2Provider::class);

        $this->integration = $instance;
        $this->oauthProvider = $oauthProvider;
        $this->session = $session;
    }

    public function getAccessToken(): string
    {
        return $this->data('token') ?? '';
    }

    public function getRefreshToken(): string
    {
        return $this->data('refresh_token');
    }

    public function getTenantId(): string
    {
        return $this->data('tenant_id');
    }

    public function getExpires(): int
    {
        return $this->data('expires');
    }

    public function getState(): string
    {
        return $this->session->get('xero_oauth2_state');
    }

    public function getAuthorizationUrl(): string
    {
        $redirectUrl = $this->oauthProvider->getAuthorizationUrl([
            'scope' => config('xero.oauth.scopes'),
            'redirect_uri' => config('app.url').'/xero/auth/callback',
        ]);
        $this->session->put('xero_oauth2_state', $this->oauthProvider->getState());

        return $redirectUrl;
    }

    public function getData(): array
    {
        return $this->data();
    }

    public function exists(): bool
    {
        return $this->integration && $this->integration->connection_settings ? true : false;
    }

    public function isExpired(): bool
    {
        return time() >= $this->data('expires');
    }

    public function refresh(): void
    {
        $newAccessToken = $this->oauthProvider->getAccessToken('refresh_token', [
            'refresh_token' => $this->getRefreshToken(),
        ]);

        $this->store($newAccessToken);
    }

    public function store(AccessTokenInterface $token, ?string $tenantId = null): void
    {
        $this->integration->connection_settings = array_merge($this->data(), [
            'token' => $token->getToken(),
            'refresh_token' => $token->getRefreshToken(),
            'id_token' => $token->getValues()['id_token'],
            'expires' => $token->getExpires(),
            'tenant_id' => $tenantId ?? $this->getTenantId(),
        ]);
        $this->integration->save();
    }

    public function delete(): void
    {
        $this->integration->connection_settings = null;
        $this->integration->saveOrFail();
    }

    public function getUser(): ?array
    {
        try {
            $jwt = new \XeroAPI\XeroPHP\JWTClaims();
            $jwt->setTokenId($this->data('id_token'));
            $decodedToken = $jwt->decode();

            return [
                'given_name' => $decodedToken->getGivenName(),
                'family_name' => $decodedToken->getFamilyName(),
                'email' => $decodedToken->getEmail(),
                'user_id' => $decodedToken->getXeroUserId(),
                'username' => $decodedToken->getPreferredUsername(),
                'session_id' => $decodedToken->getGlobalSessionId(),
            ];
        } catch (\Throwable $e) {
            return null;
        }
    }

    protected function data($key = null)
    {
        if (! $this->exists()) {
            throw new \Exception('Xero oauth credentials are missing');
        }

        $cacheData = $this->integration->connection_settings;

        return empty($key) ? $cacheData : ($cacheData[$key] ?? null);
    }
}
