<?php

namespace App\Providers;

use App\FilesystemAdapters\Local;
use App\Helpers\FactoryHelper;
use App\Helpers\ImportFileHelper;
use App\Managers\AwsSecretsManager;
use App\Models\AccountingTransaction;
use App\Models\Address;
use App\Models\Attribute;
use App\Models\BackorderQueueRelease;
use App\Models\Customer;
use App\Models\FifoLayer;
use App\Models\IntegrationInstance;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\ShippingMethod;
use App\Models\Shopify\ShopifyOrder;
use App\Models\Store;
use App\Models\Supplier;
use App\Notifications\MonitoringMessage;
use App\Observers\AccountingObserver;
use App\Observers\AccountingTransactionObserver;
use App\Observers\AddPackingSlipQueueObserver;
use App\Observers\FifoLayerUpdateObserver;
use App\Observers\IntegrationInstanceObserver;
use App\Observers\InvalidateFinancialReportingCacheObserver;
use App\Observers\InventoryMovementUpdateObserver;
use App\Observers\SalesOrderLineFinancialObserver;
use App\Observers\SalesOrderObserver;
use App\Observers\Shopify\ShopifyOrderObserver;
use App\Support\Concurrency\ConcurrencyManager;
use App\Support\DatabaseConcurrencyManager;
use App\Validator;
use Dotenv\Dotenv;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Filesystem\FilesystemAdapter;
use Illuminate\Queue\Events\JobFailed;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Str;
use League\Flysystem\Filesystem;
use League\Flysystem\UnixVisibility\PortableVisibilityConverter;
use Staudenmeir\LaravelCte\Query\Builder as LaravelCteBuilder;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        Factory::guessFactoryNamesUsing(
            [FactoryHelper::class, 'guessFactoryName']
        );

        if (! $this->app->runningUnitTests() && ! $this->app->isProduction()) {
            $this->app->register(TelescopeServiceProvider::class);
            $this->app->register(HorizonServiceProvider::class);
        }

        $this->app->singleton(ConcurrencyManager::class, DatabaseConcurrencyManager::class);
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        if (!$this->app->runningUnitTests()) {
            AwsSecretsManager::setUpKeys();
        }

        URL::forceScheme('https');

        Builder::macro('toSqlWithBindings', function () {
            $bindings = array_map(
                fn ($value) => is_numeric($value) ? $value : "'{$value}'",
                $this->getBindings()
            );

            return Str::replaceArray('?', $bindings, $this->toSql());
        });

        LaravelCteBuilder::macro('toSqlWithBindings', function () {
            $bindings = array_map(
                fn ($value) => is_numeric($value) ? $value : "'{$value}'",
                $this->getBindings()
            );

            return Str::replaceArray('?', $bindings, $this->toSql());
        });

        //    ini_set('memory_limit','256M');
        // custom validator to return validation code(rule)
        App::make('validator')->resolver(function ($translator, $data, $rules, $messages, $customAttributes) {
            return new Validator($translator, $data, $rules, $messages, $customAttributes);
        });

        // replace the default local driver with SKU Local adapter
        Storage::extend('local', function ($app, $config) {
            $links = ($config['links'] ?? null) === 'skip' ? Local::SKIP_LINKS : Local::DISALLOW_LINKS;

            $adapter = new Local(
                $config['root'],
                PortableVisibilityConverter::fromArray(Local::$permissions),
                $config['lock'] ?? LOCK_EX,
                $links,
                null
            );

            return new FilesystemAdapter(
                new Filesystem($adapter, $config),
                $adapter,
                $config
            );
        });

        // load .ev.services
        try {
            Dotenv::createMutable(base_path(), '.env.services')->load();
        } catch (\Exception $exception) {
            Log::error('Error when loading ".env.services": '.$exception->getMessage());
        }

        // Notify about failed jobs
        Queue::failing(function (JobFailed $event) {
            // close mysql connections
            DB::connection('mysql')->disconnect();

            if (env('APP_ENV') === 'local') {
                return;
            }
            try {
                $url = env('APP_URL').'/horizon/failed/'.$event->job->getJobId();
                $message = "{$event->exception->getMessage()}\n{$event->job->resolveName()}\n$url";
                Notification::route('slack', config('slack.failed-jobs'))->notify(new MonitoringMessage($message));
            } catch (\Exception $e) {
            }
        });
        //
        //        Queue::after(function (JobProcessed $event) {
        //            // close mysql connections
        //            DB::connection('mysql')->disconnect();
        //        });

        // force HTTPS
        // URL::forceScheme( 'https' );

        Supplier::observe(AccountingObserver::class);

        /**
         * Invalidate inventory snapshots
         */
        // InventorySnapshot is not currently in use
        //InventoryMovement::observe(InvalidateInventorySnapshotObserver::class);

        /**
         * Add to reporting queue.
         */
        SalesOrder::observe(InvalidateFinancialReportingCacheObserver::class);
        SalesOrderLine::observe(InvalidateFinancialReportingCacheObserver::class);
        Product::observe(InvalidateFinancialReportingCacheObserver::class);

        // Related timestamp updates
        InventoryMovement::observe(InventoryMovementUpdateObserver::class);
        FifoLayer::observe(FifoLayerUpdateObserver::class);

        // Integration instance
        IntegrationInstance::observe(IntegrationInstanceObserver::class);

        // update product listings quantity by listening on product inventory model
        /*
         * This observer can lead to all kinds of problems.  For now, I'm disabling it.  I think a better option
         * may be to have a job that runs every 5 minutes or so that updates the product listings quantity based on any
         * newly updated product inventory records.
         */
        //ProductInventory::observe(UpdateProductListingQuantityObserver::class);

        /**
         * Add to packing slip queue
         *
         * @see SKU-3935
         */
        SalesOrder::observe(AddPackingSlipQueueObserver::class);
        SalesOrderLine::observe(AddPackingSlipQueueObserver::class);
        SalesOrderFulfillment::observe(AddPackingSlipQueueObserver::class);
        Store::observe(AddPackingSlipQueueObserver::class);
        Address::observe(AddPackingSlipQueueObserver::class);
        ShippingMethod::observe(AddPackingSlipQueueObserver::class);
        Customer::observe(AddPackingSlipQueueObserver::class);
        BackorderQueueRelease::observe(AddPackingSlipQueueObserver::class);
        Product::observe(AddPackingSlipQueueObserver::class);
        Attribute::observe(AddPackingSlipQueueObserver::class);

        /**
         * Shopify Order Observer for creating lines
         */
        ShopifyOrder::observe(ShopifyOrderObserver::class);

        $this->app->singleton('ImportFileHelper', function ($app) {
            return new ImportFileHelper();
        });

        SalesOrder::observe(SalesOrderObserver::class);

        // Seed sales order financial line with sales order line
        SalesOrderLine::observe(SalesOrderLineFinancialObserver::class);

        AccountingTransaction::observe(AccountingTransactionObserver::class);
    }
}
