<?php

namespace App\Observers;

use App\Helpers;
use App\Models\Amazon\ShippingService;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\NominalCode;
use App\Models\PaymentType;
use App\Models\Setting;
use App\Models\ShippingCarrier;
use App\Models\ShippingMethod;
use App\Models\Store;
use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Modules\Xero\Jobs\XeroDownloadAccountsJob;
use Modules\Xero\Jobs\XeroDownloadTaxRatesJob;
use Modules\Xero\Jobs\XeroSyncSuppliersJob;
use Modules\Xero\Managers\XeroManager;

/**
 * Class IntegrationInstanceObserver.
 */
class IntegrationInstanceObserver
{
    public function created(IntegrationInstance $integrationInstance): void
    {
        if ($integrationInstance->isSalesChannel()) {
            $this->finalizeSalesChannelSettings($integrationInstance);
        } elseif ($integrationInstance->isAccounting()) {
            $this->finalizeAccountingSettings($integrationInstance);

            if ($integrationInstance->isXero() && $integrationInstance->isFinalized()) {
                /** @var XeroManager $manager */
                $manager = app(XeroManager::class);
                dispatch(new XeroDownloadAccountsJob(manager: $manager));
                dispatch(new XeroDownloadTaxRatesJob(manager: $manager));
                dispatch(new XeroSyncSuppliersJob());
            }
        }
    }

    public function updated(IntegrationInstance $integrationInstance): void
    {
        //
    }

    /**
     * Finalizes settings for integration instance.
     */
    public function finalizeSalesChannelSettings(IntegrationInstance $integrationInstance)
    {
        $integrationSettings = $integrationInstance->integration_settings;

        if (! isset($integrationSettings['settings'])) {
            $integrationSettings['settings'] = [];
        }

        if (! isset($integrationSettings['settings']['isTaxIncluded'])) {
            $integrationSettings['settings']['isTaxIncluded'] = true;
        }
        if (! isset($integrationSettings['settings']['paymentType'])) {
            $integrationSettings['settings']['paymentType'] = PaymentType::PAYMENT_TYPE_CASH;
        }
        if (! isset($integrationSettings['settings']['feeNominalCodeId'])) {
            $integrationSettings['settings']['feeNominalCodeId'] = $this->getFeeNominalCodeId();
        }
        if (! isset($integrationSettings['settings']['store'])) {
            $store = $this->getDefaultStore();
            $integrationSettings['settings']['store'] = [
                'id' => $store->id,
                'name' => $store->name,
            ];
        }

        if (empty($integrationSettings['orders'])) {
            $integrationSettings['orders'] = [
                'condition' => IntegrationInstance::ORDER_CONDITION_CREATED_AFTER,
                'date' => Helpers::setting(Setting::KEY_INVENTORY_START_DATE, $integrationInstance->created_at),
                'order_statuses' => [],
                'orderSync' => [
                    'shipmentDetailUpdate' => false,
                    'lastUpdated' => null,
                    'orderCancellations' => false,
                    'orderRefunds' => false,
                    'captureGiftMessages' => false,
                ],
                'download' => [
                    'auto_download' => [
                        'enabled' => false,
                        'frequency_mins' => 15,
                    ],
                ],
            ];
        }

        if (
            is_null(@$integrationSettings['orders']['download']['open_start_date'])
            || empty($integrationSettings['orders']['download']['open_start_date'])
        ) {
            $integrationSettings['orders']['download']['open_start_date'] = Helpers::setting(Setting::KEY_INVENTORY_START_DATE, $integrationInstance->created_at);
        }

        if (
            is_null(@$integrationSettings['orders']['download']['closed_start_date'])
            || empty($integrationSettings['orders']['download']['closed_start_date'])
        ) {
            $integrationSettings['orders']['download']['closed_start_date'] = Helpers::setting(Setting::KEY_INVENTORY_START_DATE, $integrationInstance->created_at);
        }

        if (empty($integrationSettings['pricing'])) {
            $integrationSettings['pricing'] = [
                'masterOfPrice' => [
                    'name' => IntegrationInstance::SETTINGS_OPTION_NAME_NEITHER,
                    'id' => IntegrationInstance::SETTINGS_OPTION_ID_NEITHER,
                ],
            ];
        }

        if (empty($integrationSettings['pricing'])) {
            $integrationSettings['pricing'] = [
                'masterOfPrice' => [
                    'name' => IntegrationInstance::SETTINGS_OPTION_NAME_NEITHER,
                    'id' => IntegrationInstance::SETTINGS_OPTION_ID_NEITHER,
                ],
            ];
        }

        /* @see SKU-3451 & SKU-3454 */
        if (empty($integrationSettings['inventory']) && ! ($integrationInstance->isShopify() || $integrationInstance->isMagento())) {
            $integrationSettings['inventory'] = [
                'masterOfStock' => IntegrationInstance::SETTINGS_OPTION_NAME_NEITHER,
            ];

            $integrationSettings['inventory']['fulfillmentLatency'] = 1; // Default is 1.

            // We seed with all user (non-FBA) warehouses (SKU-2343)
            Warehouse::with([])
                ->where('type', '!=', Warehouse::TYPE_SUPPLIER)
                ->where('type', '!=', Warehouse::TYPE_AMAZON_FBA)
                ->each(function (Warehouse $warehouse) use (&$integrationSettings) {
                    $integrationSettings['inventory']['selectedWarehouses'][] = $warehouse->id;
                });
        }

        if (is_null(@$integrationSettings['auto_create_products'])) {
            $integrationSettings['auto_create_products'] = false;
        }

        if (is_null(@$integrationSettings['proforma_marketplace_cost_percentage'])) {
            $integrationSettings['proforma_marketplace_cost_percentage'] = null;
        }

        if (is_null(@$integrationSettings['proforma_payment_cost_percentage'])) {
            $integrationSettings['proforma_payment_cost_percentage'] = null;
        }

        $integrationInstance->integration_settings = $integrationSettings;

        $integrationInstance->save();

        if ($integrationInstance->isAmazonInstance()) {
            // If there isn't an Amazon payment type, we create it
            $integrationInstance->handleAmazonPaymentTypeSetup();

            /**
             * Amazon MCF.
             */
            /** @var ShippingCarrier $amazonCarrier */
            $amazonCarrier = ShippingCarrier::with([])->updateOrCreate(['name' => 'Amazon MCF']);
            $amazonCarrier->shippingMethods()->updateOrCreate(['name' => 'Standard'], [
                'type' => ShippingMethod::TYPE_BOTH,
            ]);
            $amazonCarrier->shippingMethods()->updateOrCreate(['name' => 'Expedited'], [
                'type' => ShippingMethod::TYPE_BOTH,
            ]);
            $amazonCarrier->shippingMethods()->updateOrCreate(['name' => 'SecondDay'], [
                'type' => ShippingMethod::TYPE_DOMESTIC,
            ]);

            //            ShippingService::with([])->updateOrCreate(
            //                [
            //                    'code'          => 'exp_us_d2d_dom',
            //                    'name'          => 'Exp US D2D Dom',
            //                    'domestic'      => true,
            //                    'international' => false,
            //                ]
            //            );
            //
            //            ShippingService::with([])->updateOrCreate(
            //                [
            //                    'code'          => 'second_us_d2d_dom',
            //                    'name'          => 'Second US D2D Dom',
            //                    'domestic'      => true,
            //                    'international' => false,
            //                ]
            //            );
            //
            //            ShippingService::with([])->updateOrCreate(
            //                [
            //                    'code'          => 'std_us_d2d_dom',
            //                    'name'          => 'Std US D2D Dom',
            //                    'domestic'      => true,
            //                    'international' => false,
            //                ]
            //            );
            //
            //            ShippingService::with([])->updateOrCreate(
            //                [
            //                    'code'          => 'standard',
            //                    'name'          => 'Standard',
            //                    'domestic'      => false,
            //                    'international' => true,
            //                ]
            //            );
            //
            //            ShippingService::with([])->updateOrCreate(
            //                [
            //                    'code'          => 'expedited',
            //                    'name'          => 'Expedited',
            //                    'domestic'      => false,
            //                    'international' => true,
            //                ]
            //            );
            //
            //            ShippingService::with([])->updateOrCreate(
            //                [
            //                    'code'          => 'nextday',
            //                    'name'          => 'NextDay',
            //                    'domestic'      => false,
            //                    'international' => true,
            //                ]
            //            );
            //
            //            ShippingService::with([])->updateOrCreate(
            //                [
            //                    'code'          => 'secondday',
            //                    'name'          => 'SecondDay',
            //                    'domestic'      => false,
            //                    'international' => true,
            //                ]
            //            );
        }

        if ($integrationInstance->isShopify()) {
            // If there isn't a Shopify Payment type, we create it
            $integrationInstance->handleShopifyPaymentTypeSetup();
        }
    }

    protected function isSalesChannel(IntegrationInstance $integrationInstance): bool
    {
        return $integrationInstance->integration->integration_type == Integration::TYPE_SALES_CHANNEL;
    }

    protected function getFeeNominalCodeId(): ?int
    {
        $code = NominalCode::with([])->where('code', 520)
            ->orWhere('name', 'Merchant Account Fees')
            ->first();

        return $code ? $code->id : null;
    }

    protected function getDefaultStore(): Model|Collection|Builder|array|null
    {
        return Store::with([])->findOrFail(Store::LOCAL_STORE_ID);
    }

    private function finalizeAccountingSettings(IntegrationInstance $integrationInstance)
    {
        $integrationSettings = $integrationInstance->integration_settings;

        $integrationSettings['remainingApiUsage'] = null;
        $integrationSettings['lastSyncContacts'] = null;
        $integrationSettings['settings']['sync_start_date'] = Carbon::now()->format('Y-m-d');

        $integrationInstance->integration_settings = $integrationSettings;
        $integrationInstance->update();
    }
}
