<?php

namespace App\Observers;

use App\Models\Address;
use App\Models\Attribute;
use App\Models\BackorderQueueRelease;
use App\Models\Customer;
use App\Models\PackingSlipQueue;
use App\Models\Product;
use App\Models\ProductImage;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\ShippingMethod;
use App\Models\Store;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Storage;

/**
 * @see SKU-3935
 */
class AddPackingSlipQueueObserver
{
    public function created(Model $model): void
    {
        if ($model instanceof SalesOrderFulfillment) {
            $model->packingSlipQueues()->create(['event' => PackingSlipQueue::EVENT_CREATE]);
        }

        if ($model instanceof BackorderQueueRelease) {
            static::addPackingSlipQueue($model->backorderQueue->salesOrderLine->salesOrder);
        }
    }

    public function updated(Model $model): void
    {
        if ($model instanceof Store && collect($model->getChanges())->except('archived_at')->isNotEmpty()) {
            $this->storeUpdated($model);
        }

        if ($model instanceof Address) {
            $this->addressUpdated($model);
        }

        if ($model instanceof ShippingMethod && isset($model->getChanges()['full_name'])) {
            $this->shippingMethodUpdated($model);
        }

        if ($model instanceof Customer && isset($model->getChanges()['name'])) {
            $this->CustomerUpdated($model);
        }

        if ($model instanceof Product && Arr::hasAny($model->getChanges(), ['sku', 'barcode', 'mpn'])) {
            static::productUpdated($model);
        }

        if ($model instanceof Attribute && Arr::hasAny($model->getChanges(), ['name', 'available_for_templates'])) {
            $this->attributeUpdated($model);
        }

        if ($model instanceof SalesOrder && $model->order_status != SalesOrder::STATUS_CLOSED && Arr::hasAny(
            $model->getChanges(),
            [
                'customer_id',
                'shipping_address_id',
                'billing_address_id',
                'currency_id',
                'shipping_method_id',
                'requested_shipping_method',
                'order_date',
                'store_id',
            ]
        )) {
            static::salesOrderUpdated($model);
        }
        if ($model instanceof SalesOrderLine && $model->salesOrder) {
            static::salesOrderUpdated($model->salesOrder);
        }
    }

    public function deleted(Model $model): void
    {
        // delete the packing slip file
        if ($model instanceof SalesOrder) {
            Storage::disk('order_packing_slips')->delete("{$model->id}.pdf");
        }
        if ($model instanceof SalesOrderFulfillment) {
            Storage::disk('fulfillment_packing_slips')->delete("{$model->id}.pdf");
        }

        if ($model instanceof BackorderQueueRelease) {
            static::addPackingSlipQueue($model->backorderQueue->salesOrderLine->salesOrder);
        }
    }

    private function storeUpdated(Store $store)
    {
        $store->salesOrders()->with(['salesOrderFulfillments'])
            ->where('order_status', SalesOrder::STATUS_OPEN)
            ->whereHas('salesOrderFulfillments')
            ->each(function (SalesOrder $order) {
                static::addPackingSlipQueue($order);
                $order->salesOrderFulfillments->each(fn ($sof) => static::addPackingSlipQueue($sof));
            });
    }

    private function addressUpdated(Address $address)
    {
        // reprint sales orders that have this address to shipping/billing address
        SalesOrder::with(['salesOrderFulfillments'])
            ->where('order_status', SalesOrder::STATUS_OPEN)
            ->whereHas('salesOrderFulfillments')
            ->where(function (Builder $builder) use ($address) {
                $builder->where('shipping_address_id', $address->id);
                $builder->orWhere('billing_address_id', $address->id);
            })
            ->each(function (SalesOrder $order) {
                static::addPackingSlipQueue($order);
                $order->salesOrderFulfillments->each(fn ($sof) => static::addPackingSlipQueue($sof));
            });
        // reprint sales orders' store that have this address
        SalesOrder::with(['salesOrderFulfillments'])
            ->where('order_status', SalesOrder::STATUS_OPEN)
            ->whereHas('salesOrderFulfillments')
            ->whereRelation('store', 'address_id', $address->id)
            ->each(function (SalesOrder $order) {
                static::addPackingSlipQueue($order);
                $order->salesOrderFulfillments->each(fn ($sof) => static::addPackingSlipQueue($sof));
            });
    }

    private function shippingMethodUpdated(ShippingMethod $shippingMethod)
    {
        SalesOrder::query()
            ->where('order_status', SalesOrder::STATUS_OPEN)
            ->where('shipping_method_id', $shippingMethod->id)
            ->each(function (SalesOrder $order) {
                static::addPackingSlipQueue($order);
            });

        SalesOrderFulfillment::query()
            ->whereRelation('salesOrder', 'order_status', SalesOrder::STATUS_OPEN)
            ->where('fulfilled_shipping_method_id', $shippingMethod->id)
            ->orWhere(function (Builder $builder) use ($shippingMethod) {
                $builder->whereNull('fulfilled_shipping_method_id');
                $builder->whereNull('fulfilled_shipping_method');
                $builder->where('requested_shipping_method_id', $shippingMethod->id);
            })
            ->each(fn ($sof) => static::addPackingSlipQueue($sof));
    }

    private function customerUpdated(Customer $customer)
    {
        SalesOrder::with(['salesOrderFulfillments'])
            ->where('order_status', SalesOrder::STATUS_OPEN)
            ->whereHas('salesOrderFulfillments')
            ->where('customer_id', $customer->id)
            ->each(function (SalesOrder $order) {
                static::addPackingSlipQueue($order);
                $order->salesOrderFulfillments->each(fn ($sof) => static::addPackingSlipQueue($sof));
            });
    }

    public static function salesOrderUpdated(SalesOrder $salesOrder)
    {
        static::addPackingSlipQueue($salesOrder);
        $salesOrder->salesOrderFulfillments->each(fn ($sof) => static::addPackingSlipQueue($sof));
    }

    public static function productPrimaryImageUpdated(ProductImage $productImage)
    {
        SalesOrder::with(['salesOrderFulfillments'])
            ->where('order_status', SalesOrder::STATUS_OPEN)
            ->whereHas('salesOrderFulfillments')
            ->whereRelation('salesOrderLines', 'product_id', $productImage->product_id)
            ->each(function (SalesOrder $order) {
                //                      static::addPackingSlipQueue($order);
                $order->salesOrderFulfillments->each(fn ($sof) => static::addPackingSlipQueue($sof));
            });
    }

    public static function productUpdated(Product $product)
    {
        SalesOrder::with(['salesOrderFulfillments'])
            ->where('order_status', SalesOrder::STATUS_OPEN)
            ->whereHas('salesOrderFulfillments')
            ->whereRelation('salesOrderLines', 'product_id', $product->id)
            ->each(function (SalesOrder $order) {
                static::addPackingSlipQueue($order);
                $order->salesOrderFulfillments->each(fn ($sof) => static::addPackingSlipQueue($sof));
            });
    }

    private function attributeUpdated(Attribute $attribute)
    {
        SalesOrder::with(['salesOrderFulfillments'])
            ->where('order_status', SalesOrder::STATUS_OPEN)
            ->whereHas('salesOrderFulfillments')
            ->whereRelation('productAttributes', 'attribute_id', $attribute->id)
            ->each(function (SalesOrder $order) {
                static::addPackingSlipQueue($order);
                $order->salesOrderFulfillments->each(fn ($sof) => static::addPackingSlipQueue($sof));
            });
    }

    public static function addPackingSlipQueue(SalesOrderFulfillment|SalesOrder $model, bool $updated = true): void
    {
        /** @var PackingSlipQueue $packingSlipQueue */
        $packingSlipQueue = PackingSlipQueue::query()->firstOrNew(['link_id' => $model->id, 'link_type' => $model->getMorphClass()]);

        $status = array_filter([PackingSlipQueue::STATUS_FAILED, $updated ? PackingSlipQueue::STATUS_IN_PROGRESS : null]);
        if (! $packingSlipQueue->exists || in_array($packingSlipQueue->status, $status)) {
            $model->packingSlipQueues()->create(['event' => PackingSlipQueue::EVENT_UPDATE]);
        }
    }
}
