<?php

namespace App\Notifications;

use App\Jobs\GenerateSalesCreditInvoice;
use App\Models\SalesCredit;
use App\Models\StoreEmailTemplate;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Storage;

class SalesCreditIssuedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * @var SalesCredit
     */
    public $salesCredit;

    /**
     * SalesCreditIssuedNotification constructor.
     */
    public function __construct(SalesCredit $salesCredit)
    {
        $this->salesCredit = $salesCredit;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     */
    public function via($notifiable): array
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return MailMessage|void
     */
    public function toMail($notifiable): MailMessage|bool
    {
        $this->salesCredit->load([
            'customer',
            'salesCreditLines',
        ]);

        $emailTemplate = StoreEmailTemplate::with([])
            ->where('type', StoreEmailTemplate::TYPE_SALES_CREDIT_ISSUED)
            ->first();
        if (! $emailTemplate || ! $emailTemplate->enabled) {
            return false;
        }

        $data = StoreEmailTemplate::fillTemplate($this->salesCredit->id, 'SalesCredit', $emailTemplate->subject, $emailTemplate->html_body);

        // Markdown Email
        $mailMessage = new MailMessage();
        $mailMessage->subject($data['subject']);
        $mailMessage->view('mail.template', ['body' => $data['body']]);

        // attach PO as a PDF attachment
        if ($emailTemplate->attach_pdf) {
            $path = dispatch_sync((new GenerateSalesCreditInvoice($this->salesCredit))->onConnection('sync'));
            $mailMessage->attach(Storage::disk('reports')->path($path));
        }

        return $mailMessage;
    }
}
