<?php

namespace App\Notifications;

use App\Helpers;
use App\Models\PurchaseOrder;
use App\Models\Setting;
use App\Models\StoreEmailTemplate;
use App\Models\Supplier;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use InvalidArgumentException;
use romanzipp\QueueMonitor\Traits\IsMonitored;

abstract class PurchaseOrderNotification extends Notification implements ShouldQueue
{
    use IsMonitored;
    use Queueable;

    public PurchaseOrder $purchaseOrder;

    /**
     * @var User
     */
    public $user;

    /**
     * Create a new notification instance.
     */
    public function __construct(PurchaseOrder $purchaseOrder)
    {
        $this->purchaseOrder = $purchaseOrder;
        $this->user = auth()->user();
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  Supplier  $notifiable
     */
    public function via($notifiable): array
    {
        $via = ['mail'];

        if ($this->hasAttachments()) {
            if (empty($notifiable->getPurchaseOrderEmail())) {
                throw new InvalidArgumentException('The selected supplier does not have email');
            }
        }

        return $via;
    }

    protected function hasAttachments(): bool
    {
        return in_array($this->purchaseOrder->submission_format, [
            PurchaseOrder::SUBMISSION_FORMAT_CSV_ATTACHMENT,
            PurchaseOrder::SUBMISSION_FORMAT_EMAIL_PDF_AND_CSV_ATTACHMENTS,
            PurchaseOrder::SUBMISSION_FORMAT_PDF_ATTACHMENT,
        ]);
    }

    protected function getAttachments(): array
    {
        $attachments = [];
        if ($this->needsPDFAttachment()) {
            $attachments[] = $this->purchaseOrder->exportToPDF();
        }

        if ($this->needsCSVAttachment()) {
            $attachments[] = $this->purchaseOrder->linesToCSV();
        }

        return $attachments;
    }

    protected function needsPDFAttachment(): bool
    {
        return in_array($this->purchaseOrder->submission_format, [
            PurchaseOrder::SUBMISSION_FORMAT_PDF_ATTACHMENT,
            PurchaseOrder::SUBMISSION_FORMAT_EMAIL_PDF_AND_CSV_ATTACHMENTS,
        ]);
    }

    protected function needsCSVAttachment(): bool
    {
        return in_array($this->purchaseOrder->submission_format, [
            PurchaseOrder::SUBMISSION_FORMAT_CSV_ATTACHMENT,
            PurchaseOrder::SUBMISSION_FORMAT_EMAIL_PDF_AND_CSV_ATTACHMENTS,
        ]);
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(mixed $notifiable): MailMessage|bool
    {
        customlog('backorderPurchasing', 'PO ' . $this->purchaseOrder->purchase_order_number . ' sending mail to supplier ' . $this->purchaseOrder->supplier->name, [], 7);
        $this->purchaseOrder->load([
            'store',
            'supplier',
            'purchaseOrderLines',
            'purchaseOrderLines.product',
            'purchaseOrderLines.product.supplierProducts',
        ]);

        $emailTemplate = StoreEmailTemplate::with([])
//                                       ->where( 'store_id', $this->purchaseOrder->store_id )
            ->where('type', $this->getTemplateType())
            ->first();
        if (! $emailTemplate) {
            return false;
        }

        $data = StoreEmailTemplate::fillTemplate($this->purchaseOrder->id, 'PurchaseOrder', $emailTemplate->subject, $emailTemplate->html_body);

        // Markdown Email
        $mailMessage = new MailMessage();
        /** @see SKU-3755 */
        $mailMessage->from($this->purchaseOrder->store->po_email ?: $this->purchaseOrder->store->email, $this->purchaseOrder->store->company_name ?: $this->purchaseOrder->store->name);
        if ($ccOutgoingEmail = Helpers::setting(Setting::KEY_PO_CC_OUTGOING)) {
            $mailMessage->cc($ccOutgoingEmail, $this->purchaseOrder->store->company_name ?: $this->purchaseOrder->store->name);
        }
        if ($this->user) {
            $mailMessage->cc($this->user->email, $this->user->name);
        }
        $mailMessage->subject($data['subject']);
        $mailMessage->view('mail.template', ['body' => $data['body']]);

        // attach PO attachments
        $attachments = $this->getAttachments();

        foreach ($attachments as $path) {
            $mailMessage->attach($path);
        }

        // Set the supplier's email
        $notifiable->email = $notifiable->getPurchaseOrderEmail();

        return $mailMessage;
    }

    abstract public function getTemplateType(): string;
}
