<?php

namespace App\Notifications;

use App\Data\FinancialAlertSummaryNotificationData;
use App\Enums\FinancialAlertNotificationConditionEnum;
use App\Exceptions\FinancialAlertNotificationConditionTypeException;
use App\Models\FinancialAlert;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\SlackMessage;
use Illuminate\Notifications\Notification;


class FinancialAlertSlackNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        private readonly ?FinancialAlert $financialAlert = null,
        private readonly ?FinancialAlertSummaryNotificationData $data = null,
        private readonly ?FinancialAlertNotificationConditionEnum $condition = null,
    )
    {
    }

    public function via($notifiable): array
    {
        return ['slack'];
    }

    /**
     * @throws FinancialAlertNotificationConditionTypeException
     */
    public function toSlack($notifiable): SlackMessage
    {
        if (isset($this->financialAlert)) {
            $salesOrder = $this->financialAlert->salesOrderLineFinancial->salesOrderLine->salesOrder;

            $message = "*Low Margin/Loss Alert for Sales Order #$salesOrder->sales_order_number*";
            $message .= "\n\n";
            $message .= "An alert has been triggered for Sales Order #$salesOrder->sales_order_number due to a low margin or loss detected on one of its items.";
            $message .= "\n\n\n";
            $message .= "*Sales Order Details:*\n\n";
            $message .= "*Sales Order Number:* " . $salesOrder->sales_order_number . "\n";
            $message .= "*Order date:* " . $salesOrder->order_date->format('m/d/Y h:m') . "\n";
            $message .= "*Customer:* " . $salesOrder->customer->name . "\n\n\n";

            $message .= "*Affected Item*\n\n";
            $message .= "*Product:* " . $this->financialAlert->salesOrderLineFinancial->salesOrderLine->product->name . "\n";
            $message .= "*Quantity:* " . $this->financialAlert->salesOrderLineFinancial->salesOrderLine->quantity . "\n";
            $message .= "*Revenue:* " . number_format($this->financialAlert->salesOrderLineFinancial->total_revenue, 2) . "\n";
            $message .= "*Cost:* " . number_format($this->financialAlert->salesOrderLineFinancial->total_cost, 2) . "\n";
            $message .= "Profit: " . number_format($this->financialAlert->salesOrderLineFinancial->profit, 2) . "\n";
            $message .= "*Profit Margin*: " . number_format($this->financialAlert->salesOrderLineFinancial->profit_margin, 2) . "\n\n";
            $url = url(config('app.url')."/orders?SKUTableOrders=a:0;d:expanded&layers=OrderDrawer&OrderDrawer=name:OrderDrawer;id:$salesOrder->id;mode:view;hideNavigator:false;m:view;i:$salesOrder->id&drawers=OrderDrawer");
            $message .= "Please review this sales order for further action by <$url|clicking here>.";
        }
        else {
            $alertCount = $this->data->new_alert_count;

            $subject = match($this->condition) {
                FinancialAlertNotificationConditionEnum::DAILY_SUMMARY=> "*Daily Summary: $alertCount Low Margin/Loss Alerts*",
                FinancialAlertNotificationConditionEnum::WEEKLY_SUMMARY => "*Weekly Summary: $alertCount Low Margin/Loss Alerts*",
                FinancialAlertNotificationConditionEnum::IMMEDIATELY => throw new FinancialAlertNotificationConditionTypeException,
            };

            $description = match($this->condition) {
                FinancialAlertNotificationConditionEnum::DAILY_SUMMARY=> "Here is your daily summary of unhandled low margin or loss alerts.\n\nToday’s New Alert Summary:",
                FinancialAlertNotificationConditionEnum::WEEKLY_SUMMARY => "Here is your weekly summary of unhandled low margin or loss alerts.\n\nPast Week Alert Summary:",
                FinancialAlertNotificationConditionEnum::IMMEDIATELY => throw new FinancialAlertNotificationConditionTypeException,
            };

            $message = $subject . "\n\n";
            $message .= $description . "\n\n";
            $message .= "*Total Alerts:* " . number_format($this->data->new_alert_count) . "\n";
            $message .= "Loss Alerts: " . number_format($this->data->new_loss_count) . "\n";
            $message .= "Low Margin Alerts: " . number_format($this->data->new_loss_margin_count) . "\n";
            $message .= "*Orders Affected:* " . number_format($this->data->new_order_count) . "\n\n\n";
            $message .= "Total Alert Summary:\n\n";
            $message .= "*Total Alerts:* " . number_format($this->data->total_alert_count) . "\n";
            $message .= "Loss Alerts: " . number_format($this->data->total_loss_count) . "\n";
            $message .= "Low Margin Alerts: " . number_format($this->data->total_loss_margin_count) . "\n";
            $message .= "*Orders Affected:* " . number_format($this->data->total_order_count) . "\n\n\n";
        }

        return (new SlackMessage)
            ->from('SKU.io')
            ->content($message);
    }
}
