<?php

namespace App\Notifications;

use App\Models\PurchaseOrder;
use App\Models\StoreEmailTemplate;
use App\Models\Supplier;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use InvalidArgumentException;

/**
 * Class BulkPurchaseOrderNotification.
 */
class BulkPurchaseOrderNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected array $attachments;

    protected Supplier $supplier;

    /**
     * BulkPurchaseOrderNotification constructor.
     */
    public function __construct(Supplier $supplier, array $attachments)
    {
        if (empty($attachments)) {
            throw new \InvalidArgumentException('Attachments cannot be empty.');
        }
        $this->attachments = $attachments;
        $this->supplier = $supplier;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(mixed $notifiable): array
    {
        $via = ['mail'];

        if (empty($notifiable->email)) {
            throw new InvalidArgumentException('The selected supplier does not have email');
        }

        return $via;
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(mixed $notifiable): MailMessage|bool
    {
        $emailTemplate = StoreEmailTemplate::with([])
//                                       ->where( 'store_id', $this->purchaseOrder->store_id )
            ->where('type', StoreEmailTemplate::TYPE_BULK_PURCHASE_ORDER)
            ->first();
        if (! $emailTemplate) {
            return false;
        }

        $data = StoreEmailTemplate::fillTemplate($this->supplier->id, 'Supplier', $emailTemplate->subject, $emailTemplate->html_body);

        // Markdown Email
        $mailMessage = new MailMessage();
        $mailMessage->subject($data['subject']);
        $mailMessage->view('mail.template', ['body' => $data['body']]);

        foreach ($this->attachments as $path) {
            $mailMessage->attach($path);
        }

        return $mailMessage;
    }
}
