<?php

namespace App\Models;

use App\Contracts\HasReference;
use App\Helpers;
use App\Managers\WarehouseTransferManager;
use App\Models\Concerns\HasAccountingTransactionLine;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\InventoryManagement\NegativeInventoryEvent;
use App\Services\InventoryManagement\PositiveInventoryEvent;
use Carbon\Carbon;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use InvalidArgumentException;
use Throwable;

/**
 * Class WarehouseTransferShipmentReceiptLine.
 *
 * @property-read int $id
 * @property int $warehouse_transfer_shipment_receipt_id
 * @property int $warehouse_transfer_shipment_line_id
 * @property int $quantity
 * @property Collection $inventoryMovements
 * @property WarehouseTransferShipmentLine $shipmentLine
 * @property WarehouseTransferShipmentReceipt $shipmentReceipt
 * @property FifoLayer $fifoLayers
 * @property-read AccountingTransactionLine $accountingTransactionLine
 * @property-read float $totalAmount
 */
class WarehouseTransferShipmentReceiptLine extends Model implements Filterable, HasReference, NegativeInventoryEvent, PositiveInventoryEvent
{
    use HasAccountingTransactionLine;
    use HasFilters;

    protected $guarded = [];

    public function accountingTransactionLine(): MorphOne
    {
        return $this->morphOne(AccountingTransactionLine::class, 'link');
    }

    public function shipmentReceipt(): BelongsTo
    {
        return $this->belongsTo(WarehouseTransferShipmentReceipt::class, 'warehouse_transfer_shipment_receipt_id');
    }

    public function shipmentLine(): BelongsTo
    {
        return $this->belongsTo(WarehouseTransferShipmentLine::class, 'warehouse_transfer_shipment_line_id');
    }

    public function getReference(): string
    {
        return $this->getDrawer()->getReference();
    }

    public function getDrawer(): Model|WarehouseTransfer
    {
        return $this->shipmentReceipt->shipment->warehouseTransfer;
    }

    public function inventoryMovements()
    {
        return $this->morphMany(InventoryMovement::class, 'link');
    }

    public function backorderQueueReleases()
    {
        return $this->morphMany(BackorderQueueRelease::class, 'link');
    }

    public function fifoLayers(): MorphMany
    {
        return $this->morphMany(FifoLayer::class, 'link');
    }

    /**
     * @param  array  $options
     * @return bool
     * @throws Throwable
     */
    public function save(array $options = []): bool
    {
        if(
            $this->exists &&
            $this->isDirty('quantity') &&
            $this->quantity < $this->getOriginal('quantity')
        ) {

            // Quantity is being reduced.

            // Check if there are any negative adjustments for the product at the destination warehouse.
            $negativeAdjustmentQty = abs($this->shipmentLine->warehouseTransferLine->adjustments()
                ->where('warehouse_id', $this->shipmentLine->warehouseTransferLine->warehouseTransfer->to_warehouse_id)
                ->where('product_id', $this->shipmentLine->warehouseTransferLine->product_id)
                ->where('quantity', '<', 0)
                ->sum('quantity'));

            // If there are any negative adjustments, we need to reverse them, processing blemished skus last.
            if($negativeAdjustmentQty > 0){
                $originalQty = $this->getOriginal('quantity');
                $unblemishedQty = max(0, $originalQty - $negativeAdjustmentQty);
                $changeInQty = $originalQty - $this->quantity;

                $blemishedToReverse = max(0, $changeInQty - $unblemishedQty);

                if($blemishedToReverse > 0) {
                    $this->reverseAdjustments(
                        quantity: $blemishedToReverse,
                        warehouseId: $this->shipmentLine->warehouseTransferLine->warehouseTransfer->to_warehouse_id,
                        productId: $this->shipmentLine->warehouseTransferLine->product_id
                    );
                }
            }

        }

        return parent::save($options);
    }

    /**
     * @throws Throwable
     */
    public function delete(): ?bool
    {
        /**
         * We attempt to remove all stock from
         * the receipt line and delete it.
         */
        $transferLine = $this->shipmentLine->warehouseTransferLine;
        // Reverse any adjustments for the receipt line, for instance due to blemished skus.
        $this->reverseAdjustments(
            quantity: $this->quantity,
            warehouseId: $transferLine->warehouseTransfer->to_warehouse_id,
            productId: $transferLine->product_id
        );

        InventoryManager::with(
            $transferLine->warehouseTransfer->to_warehouse_id,
            $transferLine->product
        )->removeAllStockFrom($this);

        /**
         * We need to delete any canceling in-transit movements
         * created for the receipt line.
         */
        $this->inventoryMovements()->delete();

        // This model triggers observers in Modules
        $result = parent::delete();

        app(WarehouseTransferManager::class)->setTransferReceiptStatus($this->shipmentReceipt->shipment->warehouseTransfer);

        if ($this->shipmentReceipt->receiptLines->count() == 0) {
            $this->shipmentReceipt->delete();
        }

        return $result;
    }

    /**
     * @throws Throwable
     */
    private function reverseAdjustments(int $quantity, int $warehouseId, int $productId): void
    {
        $negativeAdjustments = $this->shipmentLine->warehouseTransferLine->adjustments()
            ->where('warehouse_id', $warehouseId)
            ->where('product_id', $productId)
            ->where('quantity', '<', 0)
            ->orderByDesc('created_at')
            ->get();

        $inventoryManager = InventoryManager::with(
            $warehouseId,
            $this->shipmentLine->warehouseTransferLine->product
        );

        $usedQuantity = 0;
        /** @var InventoryAdjustment $adjustment */
        foreach ($negativeAdjustments as $adjustment) {
            $reductionQuantity = max(0, min(abs($adjustment->quantity), $quantity - $usedQuantity));
            if($reductionQuantity == 0) {
                break;
            }

            if (abs($adjustment->quantity) <= $reductionQuantity) {
                // We need to reverse the entire adjustment.
                $adjustment->delete();
            } else {
                $adjustment->quantity += $reductionQuantity;
                $adjustment->save();
                $inventoryManager->decreaseNegativeEventQty(
                    quantity: $reductionQuantity,
                    event: $adjustment,
                    autoApplyStock: false
                );
            }

            $usedQuantity += $reductionQuantity;
        }
        // Reverse any positive adjustments
        $positiveAdjustments = $this->shipmentLine->warehouseTransferLine->adjustments()
            ->with('product')
            ->where('warehouse_id', $warehouseId)
            ->where('quantity', '>', 0)
            ->orderByDesc('created_at')
            ->get();

        $usedQuantity = 0;
        /** @var InventoryAdjustment $adjustment */
        foreach ($positiveAdjustments as $adjustment) {

            $inventoryManager = InventoryManager::with(
                $warehouseId,
                $adjustment->product
            );

            $reductionQuantity = max(0, min(abs($adjustment->quantity), $quantity - $usedQuantity));
            if($reductionQuantity == 0) {
                break;
            }

            if ($adjustment->quantity <= $reductionQuantity) {
                // We need to reverse the entire adjustment.
                $adjustment->delete();
            } else {
                $adjustment->quantity -= $reductionQuantity;
                $adjustment->save();
                $inventoryManager->reducePositiveEventQty(
                    quantity: $reductionQuantity,
                    event: $adjustment
                );
            }

            if($adjustment->product->isBlemished()){
                $adjustment->product->delete();
            }

            $usedQuantity += $reductionQuantity;
        }

    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return ['quantity'];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->availableColumns();
    }

    public function createFifoLayer(int $quantity, ?float $unitCost = null, ?int $productId = null): FifoLayer
    {
        $transferLine = $this->shipmentLine->warehouseTransferLine;

        $fifoLayer = new FifoLayer;
        $fifoLayer->warehouse_id = $transferLine->warehouseTransfer->to_warehouse_id;
        $fifoLayer->product_id = $transferLine->product_id;
        $fifoLayer->fifo_layer_date = $this->getEventDate();
        $fifoLayer->original_quantity = $quantity;
        $fifoLayer->fulfilled_quantity = 0;
        $fifoLayer->total_cost = $quantity * $transferLine->product->getUnitCostAtWarehouse($transferLine->warehouseTransfer->from_warehouse_id);

        $this->fifoLayers()->save($fifoLayer);

        return $fifoLayer;
    }

    public function getUnitCost(): float
    {
        return $this->shipmentLine->warehouseTransferLine->product->getUnitCostAtWarehouse(
            $this->shipmentLine->warehouseTransferLine->warehouseTransfer->from_warehouse_id
        );
    }

    public function createInventoryMovements(int $quantity, FifoLayer $fifoLayer): void
    {
        /**
         * Destination:
         *  - in transit
         *  + active
         */
        $transferLine = $this->shipmentLine->warehouseTransferLine;

        /**
         * We need to attach the negative in-transit to the
         * original fifo layer at the origin warehouse. This
         * ensures that the in-transit stock at the time of
         * shipment at the original warehouse is duly canceled out.
         */
        /** @var FifoLayer $shipmentFifo */
        $shipmentFifo = $this->shipmentLine->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '<', 0)
            ->firstOrFail()
      ->fifo_layer;

        $movement = new InventoryMovement;
        $movement->quantity = -abs($quantity);
        $movement->fifo_layer = $shipmentFifo->id;
        $movement->warehouse_id = $transferLine->warehouseTransfer->to_warehouse_id;
        $movement->product_id = $transferLine->product_id;
        $movement->inventory_status = InventoryMovement::INVENTORY_STATUS_IN_TRANSIT;
        $movement->inventory_movement_date = $this->getEventDate();
        $movement->type = InventoryMovement::TYPE_TRANSFER;
        $this->inventoryMovements()->save($movement);

        $activeMovement = $movement->replicate();
        $activeMovement->fifo_layer = $fifoLayer->id;
        $activeMovement->quantity = abs($quantity);
        $activeMovement->warehouse_id = $transferLine->warehouseTransfer->to_warehouse_id;
        $activeMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $this->inventoryMovements()->save($activeMovement);
    }

    public function getOriginatingMovement(): ?InventoryMovement
    {
        return $this->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '>', 0)
            ->first();
    }

    public function getFifoLayer(): ?FifoLayer
    {
        return $this->fifoLayers()->firstOrFail();
    }

    public function getLinkType(): string
    {
        return self::class;
    }

    public function getId(): int
    {
        return $this->id;
    }

    public function createInventoryMovementsForLayers(array $layers, ?Carbon $dateOverride = null): void
    {
        foreach ($layers as $layerInfo) {
            if (! ($layerInfo['layer'] instanceof FifoLayer)) {
                throw new InvalidArgumentException("Inventory adjustment can only have Fifo Layers, {$layerInfo['layer_type']} given.");
            }

            $fifoLayer = $layerInfo['layer'];

            $movement = new InventoryMovement;
            $movement->inventory_movement_date = $this->getEventDate();
            $movement->quantity = -abs($layerInfo['quantity']);
            $movement->product_id = $this->shipmentLine->warehouseTransferLine->product_id;
            $movement->type = InventoryMovement::TYPE_TRANSFER;
            $movement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
            $movement->warehouse_id = $this->shipmentLine->warehouseTransferLine->warehouseTransfer->toWarehouse->id;

            $movement->link_id = $this->id;
            $movement->link_type = self::class;
            $fifoLayer->inventoryMovements()->save($movement);
        }
    }

    public function getReductionActiveMovements(): Arrayable
    {
        return $this->inventoryMovements()->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '<', 0)
            ->get();
    }

    public function clearMovements(): void
    {
        $this->inventoryMovements()->delete();
    }

    public function reduceQtyWithSiblings(int $quantity, InventoryMovement $representingMovement): void
    {
        $representingMovement->quantity += $quantity;
        if ($representingMovement->quantity == 0) {
            $representingMovement->delete();
        } else {
            $representingMovement->save();
        }
    }

    public function refreshFromFifoCostUpdate(float $amount = 0): void
    {
        $this->updated_at = now();
        $this->save();
    }

    public function getProductId(): int
    {
        return $this->shipmentLine->warehouseTransferLine->product_id;
    }

    public function getWarehouseId(): int
    {
        // TODO: Not sure if this is right
        return $this->shipmentReceipt->shipment->warehouseTransfer->to_warehouse_id;
    }

    public function getEventDate(): Carbon
    {
        return $this->shipmentReceipt->received_at->max(Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE), 'UTC')) ?? now()->max(Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE), 'UTC'));
    }

    public function getQuantity(): int
    {
        return $this->quantity;
    }

    public function getType(): string
    {
        return InventoryMovement::TYPE_TRANSFER;
    }

    public function getTotalAmountAttribute(): float
    {
        return $this->getFifoLayer()->avg_cost;
    }
}
