<?php

namespace App\Models;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Models\Concerns\HasAccountingTransaction;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionDataFromWarehouseTransferShipmentReceipt;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Throwable;

/**
 * @property int $id
 * @property int $warehouse_transfer_shipment_id
 * @property int $user_id
 * @property Carbon|string $received_at
 * @property WarehouseTransferShipment $shipment
 * @property Collection $receiptLines
 * @property-read User $user
 * @property AccountingTransaction $accountingTransaction
 */
class WarehouseTransferShipmentReceipt extends Model implements FinancialDocumentInterface
{
    use HasAccountingTransaction;

    /**
     * @var array
     */
    protected $fillable = ['warehouse_transfer_shipment_id', 'user_id', 'received_at'];

    protected $casts = [
        'received_at' => 'datetime',
    ];

    public function accountingTransaction(): MorphOne
    {
        return $this->morphOne(AccountingTransaction::class, 'link');
    }

    /**
     * @throws Throwable
     */
    public function getAccountingTransactionData(): AccountingTransactionData
    {
        return (new BuildAccountingTransactionDataFromWarehouseTransferShipmentReceipt($this))->handle();
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return $this->shipment->accountingTransaction;
    }

    public function scopeAccountingReady(Builder $builder): Builder
    {
        return $builder;
    }

    public function getAccountingDateFieldName(): string
    {
        return 'received_at';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function shipment(): BelongsTo
    {
        return $this->belongsTo(WarehouseTransferShipment::class, 'warehouse_transfer_shipment_id');
    }

    /**
     * @return MorphMany
     */
    public function receiptLines(): HasMany
    {
        return $this->hasMany(WarehouseTransferShipmentReceiptLine::class);
    }

    public function delete()
    {
        $this->load('receiptLines');

        if ($this->receiptLines) {
            $this->receiptLines->each(function (WarehouseTransferShipmentReceiptLine $line) {
                return $line->delete();
            });
        }

        return parent::delete();
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }
}
