<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 7/10/20
 * Time: 8:48 AM.
 */

namespace App\Models;

use App\Contracts\HasReference;
use App\Helpers;
use App\Managers\WarehouseTransferManager;
use App\Models\Concerns\HasAccountingTransactionLine;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\InventoryManagement\NegativeInventoryEvent;
use App\Services\StockTake\OpenStockTakeException;
use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\HigherOrderBuilderProxy;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Collection;
use InvalidArgumentException;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;

/**
 * Class WarehouseTransferShipmentLine.
 *
 * @property int $id
 * @property int $warehouse_transfer_shipment_id
 * @property int $warehouse_transfer_line_id
 * @property float $quantity
 * @property-read float $total_received
 * @property WarehouseTransferLine $warehouseTransferLine
 * @property WarehouseTransferShipment $warehouseTransferShipment
 * @property WarehouseTransferShipmentReceiptLine[] $receiptLines
 * @property Collection $inventoryMovements
 * @property-read Collection $fifoLayerUsage
 * @property-read float $totalAmount
 * @property-read AccountingTransactionLine $accountingTransactionLine
 * @property-read AmazonFbaReportInventoryLedger $amazonFbaLedger
 */
class WarehouseTransferShipmentLine extends Model implements HasReference, NegativeInventoryEvent
{
    use HasAccountingTransactionLine;
    use HasFactory;

    /**
     * @var array
     */
    protected $fillable = [
        'warehouse_transfer_shipment_id',
        'warehouse_transfer_line_id',
        'quantity',
    ];

    public function warehouseTransferShipment(): BelongsTo
    {
        return $this->belongsTo(WarehouseTransferShipment::class);
    }

    public function warehouseTransferLine(): BelongsTo
    {
        return $this->belongsTo(WarehouseTransferLine::class);
    }

    public function receiptLines(): HasMany
    {
        return $this->hasMany(WarehouseTransferShipmentReceiptLine::class);
    }

    public function getReference(): string
    {
        return $this->getDrawer()->getReference();
    }

    public function getDrawer(): Model|WarehouseTransfer
    {
        return $this->warehouseTransferShipment->warehouseTransfer;
    }

    public function inventoryMovements(): MorphMany
    {
        return $this->morphMany(InventoryMovement::class, 'link');
    }

    public function amazonFbaInitialInventory(): BelongsTo
    {
        return $this->belongsTo(AmazonFbaInitialInventory::class, 'amazon_fba_initial_inventory_id');
    }

    public function accountingTransactionLine(): MorphOne
    {
        return $this->morphOne(AccountingTransactionLine::class, 'link');
    }

    public function amazonFbaLedger(): MorphOne
    {
        return $this->morphOne(AmazonFbaReportInventoryLedger::class, 'sku_link');
    }

    public function getUnreceivedAttribute()
    {
        return $this->quantity - $this->receiptLines()->sum('quantity');
    }

    public function getTotalReceivedAttribute()
    {
        return $this->receiptLines()->sum('quantity');
    }

    /**
     * @return HigherOrderBuilderProxy|FifoLayer|mixed
     */
    public function getOriginatingFifoLayer()
    {
        $fromWarehouse = $this->warehouseTransferLine->warehouseTransfer->fromWarehouse;

        $movements = $this->inventoryMovements();
        $movement = $movements
            ->where('warehouse_id', $fromWarehouse->id)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('type', InventoryMovement::TYPE_TRANSFER)
            ->firstOrFail();

        return $movement->fifo_layer;
    }

    /**
     * @throws Exception
     */
    public function delete()
    {
        $this->load('inventoryMovements', 'receiptLines');
        $this->receiptLines->each(function (WarehouseTransferShipmentReceiptLine $receiptLine) {
            $receiptLine->delete();
        });

        $transferLine = $this->warehouseTransferLine;
        if ($transferLine->product) {
            InventoryManager::with(
                $transferLine->warehouseTransfer->to_warehouse_id,
                $transferLine->product
            )->reverseNegativeEvent($this);
        }

        app(WarehouseTransferManager::class)->setTransferShipmentStatus($this->warehouseTransferShipment->warehouseTransfer);

        // This model triggers observers in Modules

        return parent::delete();
    }

    /**
     * @throws OpenStockTakeException
     */
    public function createInventoryMovementsForLayers(array $layers, ?Carbon $dateOverride = null): void
    {
        foreach ($layers as $layerInfo) {
            if (! ($layerInfo['layer'] instanceof FifoLayer)) {
                throw new InvalidArgumentException('Warehouse transfers can only accept Fifo Layers. '
                                            .$layerInfo['layer_type'].' given.');
            }

            /**
             * There should be a total of 2 inventory movements
             * origin:
             *  - active
             * destination:
             *  + in transit
             */

            /** @var InventoryMovement $movement */
            $movement = $this->inventoryMovements()
                ->where('quantity', '<', 0)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('layer_type', $layerInfo['layer_type'])
                ->where('layer_id', $layerInfo['layer']->id)
                ->first();
            if ($movement) {
                $movement->quantity += -abs($layerInfo['quantity']);
                $movement->save();

                // Update shipment line quantity.
                $this->quantity += abs($layerInfo['quantity']);
                $this->save();
            } else {
                $movement = new InventoryMovement();
                $movement->quantity = -abs($layerInfo['quantity']);
                $movement->product_id = $this->warehouseTransferLine->product_id;
                $movement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
                $movement->warehouse_id = $this->warehouseTransferLine->warehouseTransfer->from_warehouse_id;
                $movement->inventory_movement_date = $this->getEventDate();
                $movement->type = InventoryMovement::TYPE_TRANSFER;
                $movement->layer_type = $layerInfo['layer_type'];
                $movement->layer_id = $layerInfo['layer']->id;
                $movement->link_id = $this->id;
                $movement->link_type = self::class;

                /** @var FifoLayer $fifoLayer */
                $fifoLayer = $layerInfo['layer'];
                $fifoLayer->inventoryMovements()->save($movement);
            }

            // What is the point of creating a positive and negative reservation?  Commenting out for now.
//            // + Reserved
//            /** @var InventoryMovement $reservation */
//            $reservation = $this->inventoryMovements()
//                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
//                ->where('quantity', '>', 0)
//                ->where('layer_type', $layerInfo['layer_type'])
//                ->where('layer_id', $layerInfo['layer']->id)
//                ->first();
//
//            if ($reservation) {
//                $reservation->quantity += abs($layerInfo['quantity']);
//                $reservation->save();
//            } else {
//                $reservation = $movement->replicate();
//                $reservation->quantity = abs($layerInfo['quantity']);
//                $reservation->inventory_status = InventoryMovement::INVENTORY_STATUS_RESERVED;
//                $this->inventoryMovements()->save($reservation);
//            }
//
//            // - Reserved
//            /** @var InventoryMovement $negReservation */
//            $negReservation = $this->inventoryMovements()
//                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
//                ->where('quantity', '<', 0)
//                ->where('layer_type', $layerInfo['layer_type'])
//                ->where('layer_id', $layerInfo['layer']->id)
//                ->first();
//            if ($negReservation) {
//                $negReservation->quantity += -abs($layerInfo['quantity']);
//                $negReservation->save();
//            } else {
//                $negReservation = $reservation->replicate();
//                $negReservation->quantity = -abs($layerInfo['quantity']);
//                $this->inventoryMovements()->save($negReservation);
//            }

            // + in transit at destination
            /** @var InventoryMovement $inTransit */
            $inTransit = $this->inventoryMovements()
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
                ->where('quantity', '>', 0)
                ->where('layer_type', $layerInfo['layer_type'])
                ->where('layer_id', $layerInfo['layer']->id)
                ->first();

            if ($inTransit) {
                $inTransit->quantity += abs($layerInfo['quantity']);
                $inTransit->save();
            } else {
                $inTransit = $movement->replicate();
                $inTransit->quantity = abs($layerInfo['quantity']);
                $inTransit->warehouse_id = $this->warehouseTransferLine->warehouseTransfer->to_warehouse_id;
                $inTransit->warehouse_location_id = $this->warehouseTransferLine->warehouseTransfer->toWarehouse->defaultLocation?->id;
                $inTransit->inventory_status = InventoryMovement::INVENTORY_STATUS_IN_TRANSIT;
                $this->inventoryMovements()->save($inTransit);
            }
        }
    }

    public function getReductionActiveMovements(): Arrayable
    {
        return $this->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '<', 0)
            ->get();
    }

    public function clearMovements(): void
    {
        $this->inventoryMovements()->delete();
    }

    public function reduceQtyWithSiblings(int $quantity, InventoryMovement $representingMovement): void
    {
        $this->inventoryMovements()->each(function (InventoryMovement $movement) use ($quantity) {
            if ($movement->quantity < 0) {
                $movement->quantity += $quantity;
                if ($movement->quantity >= 0) {
                    $movement->delete();
                } else {
                    $movement->save();
                }
            } else {
                $movement->quantity -= $quantity;
                if ($movement->quantity <= 0) {
                    $movement->delete();
                } else {
                    $movement->save();
                }
            }
        });
    }

    public function getLinkType(): string
    {
        return self::class;
    }

    public function getId(): int
    {
        return $this->id;
    }

    public function refreshFromFifoCostUpdate(float $amount = 0): void
    {
        $this->updated_at = now();
        $this->save();
    }

    public function getProductId(): int
    {
        return $this->warehouseTransferLine->product_id;
    }

    public function getWarehouseId(): int
    {
        return $this->warehouseTransferLine->warehouseTransfer->from_warehouse_id;
    }

    public function getEventDate(): Carbon
    {
        return $this->warehouseTransferShipment->shipped_at->max(Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE), 'UTC'));
    }

    public function getQuantity(): int
    {
        return $this->quantity;
    }

    public function getType(): string
    {
        return InventoryMovement::TYPE_TRANSFER;
    }

    public function getFifoLayerUsageAttribute(): Collection
    {
        $inventoryMovements =  $this->inventoryMovements()
            ->with('layer')
            ->where('type', InventoryMovement::TYPE_TRANSFER)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
            ->get();

        return $inventoryMovements->map(function (InventoryMovement $movement) {
            return [
                'layer' => $movement->fifoLayer,
                'amount' => $movement->fifoLayer->avg_cost,
                'quantity' => $movement->quantity,
            ];
        });
    }

    public function getTotalAmountAttribute(): float
    {
        return $this->fifoLayerUsage->sum('amount');
    }
}
