<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 7/8/20
 * Time: 7:05 PM.
 */

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Contracts\HasReference;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Abstractions\AmazonRemovalOrderDestinationInterface;

/**
 * Class WarehouseTransferLine.
 *
 *
 * @property int $id
 * @property int $product_id
 * @property int $quantity
 * @property string $description
 * @property-read int $warehouse_transfer_id
 * @property-read int $fifo_layer_id
 * @property-read int $from_warehouse_location_id
 * @property WarehouseTransferShipmentLine $shipmentLine
 * @property-read int $quantity_received
 * @property-read int $quantity_unreceived
 * @property WarehouseTransfer $warehouseTransfer
 * @property Collection $inventoryMovements
 * @property Collection|InventoryAdjustment[] $adjustments
 * @property Product $product
 */
class WarehouseTransferLine extends Model implements Filterable, HasReference, UniqueFieldsInterface, AmazonRemovalOrderDestinationInterface
{
    use HasFilters;
    use HasFactory;

    /**
     * @var array
     */
    protected $fillable = ['warehouse_transfer_id', 'product_id', 'quantity', 'description'];

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    public static function getUniqueFields(): array
    {
        return ['warehouse_transfer_id', 'product_id'];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function warehouseTransfer(): BelongsTo
    {
        return $this->belongsTo(WarehouseTransfer::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function inventoryMovements(): MorphMany
    {
        return $this->morphMany(InventoryMovement::class, 'link');
    }

    public function shipmentLine(): HasOne
    {
        return $this->hasOne(WarehouseTransferShipmentLine::class);
    }

    /**
     * @return int|mixed
     */
    public function getQuantityReceivedAttribute()
    {
        $shipmentLine = $this->shipmentLine;
        if ($shipmentLine) {
            return $this->shipmentLine->receiptLines()->pluck('quantity')->sum();
        }

        return 0;
    }

    public function getQuantityUnreceivedAttribute()
    {
        return max(0, $this->quantity - $this->quantity_received);
    }

    public function getFifoLayerIdAttribute()
    {
        return $this->inventoryMovements()
            ->where('link_type', self::class) // When the transfer was created.
            ->firstOrFail(['layer_id'])
      ->layer_id;
    }

    public function getFromWarehouseLocationIdAttribute()
    {
        return $this->inventoryMovements()
            ->where('link_type', self::class) // When the transfer was created.
            ->firstOrFail(['warehouse_location_id'])
      ->warehouse_location_id;
    }

    public function adjustments(): MorphMany
    {
        return $this->morphMany(InventoryAdjustment::class, 'link');
    }

    public function getReference(): ?string
    {
        return $this->warehouseTransfer->getReference();
    }

    /**
     * Deletes the warehouse transfer line.
     */
    public function delete()
    {
        // Remove inventory movements
        $this->load('inventoryMovements', 'shipmentLine');
        DB::transaction(function () {
            // Delete the shipment line
            if ($this->shipmentLine) {
                $this->shipmentLine->delete();
            }

            // This model triggers observers in Modules

            return parent::delete();
        });
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return ['quantity', 'sku'];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->availableColumns();
    }
}
