<?php

namespace App\Models;

use App\Exporters\MapsExportableFields;
use App\Helpers;
use App\Importers\DataImporter;
use App\Importers\DataImporters\StoreDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Response;
use App\SDKs\Starshipit\Model\SenderDetails;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

/**
 * Class Store.
 *
 *
 * @property int $id
 * @property string $name
 * @property string $company_name
 * @property string $email
 * @property int|null $address_id
 * @property string|null $logo_url
 * @property string|null $custom_jrxml_file
 * @property string|null $custom_message
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read Address|null $address
 */
class Store extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable
{
    use Archive, HasFilters, HasSort;
    use HasFactory;

    const LOCAL_STORE_ID = 1;

    const DEFAULT_SKUIO_STORE_NAME = 'SKU.io';

    protected $fillable = [
        'name',
        'company_name',
        'email',
        'po_email',
        'custom_jrxml_file',
        'custom_message',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function storeEmailTemplates()
    {
        return $this->hasMany(StoreEmailTemplate::class);
    }

    public function address()
    {
        return $this->belongsTo(Address::class);
    }

    public function salesChannels()
    {
        return $this->hasMany(SalesChannel::class);
    }

    public function salesOrders()
    {
        return $this->hasMany(SalesOrder::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getLogoUrlAttribute()
    {
        if (! empty($this->attributes['logo_url']) && ! $this->isExternalUrl($this->attributes['logo_url'])) {
            if (Str::startsWith($this->getRawOriginal()['logo_url'], '/images')) {
                // This is stored at a slightly different location
                return $this->attributes['logo_url'];
            }

            return Storage::disk('images')->url($this->attributes['logo_url']);
        }

        return $this->attributes['logo_url'] ?? null;
    }

    private function isExternalUrl(?string $url)
    {
        return isset(parse_url($url)['scheme']);
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * {@inheritDoc}
     */
    public function delete()
    {
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return parent::delete();
    }

    /**
     * Determine if the nominal code is used.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        $usage = [];

        if ($this->id == self::LOCAL_STORE_ID) {
            $usage['localStore'] = __('messages.store.delete_local_failed', ['id' => $this->name]);
        }

        // used in sales orders
        $this->loadCount('salesOrders');
        if ($this->sales_orders_count) {
            $usage['salesOrders'] = trans_choice('messages.currently_used', $this->sales_orders_count, [
                'resource' => 'sales order',
                'model' => 'store('.$this->name.')',
            ]);
        }

        // used in sales channels
        $this->loadCount('salesChannels');
        if ($this->sales_channels_count) {
            $usage['salesChannels'] = trans_choice('messages.currently_used', $this->sales_channels_count, [
                'resource' => 'sales channel',
                'model' => 'store('.$this->name.')',
            ]);
        }

        return count($usage) ? $usage : false;
    }

    /*
    |--------------------------------------------------------------------------
    | For Create/Update functions
    |--------------------------------------------------------------------------
    */

    public function setLogoUrl($logo, bool $download = false)
    {
        if ($logo === false) {
            return false;
        }

        if (! empty($logo)) {
            // store image
            $imageUrl = Helpers::getImageUrl($logo, $download);

            // add unresolvable image warning
            if ($imageUrl === false) {
                $message = __('messages.product.invalid_image_url', [
                    'url' => $logo,
                    'index' => null,
                ]);

                $data = ['logo_url' => $logo];

                Response::instance()->addWarning($message, Response::CODE_UNRESOLVABLE_IMAGE, 'logo_url', $data);
            }

            $this->logo_url = $imageUrl ?: $logo;

            // remove old image from filesystem
            $oldImage = $this->getRawOriginal('logo_url');
            if (! empty($oldImage) && ! $this->isExternalUrl($oldImage) && $oldImage != $this->attributes['logo_url']) {
                Helpers::deleteImage($oldImage);
            }
        } else {
            // remove image from filesystem
            if (! empty($this->logo_url) && ! $this->isExternalUrl($this->logo_url)) {
                Helpers::deleteImage($this->logo_url);
            }

            $this->logo_url = null;
        }

        return true;
    }

    public function toStarshipitSenderDetails()
    {
        // TODO: must check the required fields
        $senderDetails = new SenderDetails();
        $senderDetails->name = $this->name;
        $senderDetails->email = $this->address->email ?? null;
        $senderDetails->phone = $this->address->phone ?? null;
        $senderDetails->company = $this->address->company ?? null;
        $senderDetails->street = $this->address->address1 ?? null;
        $senderDetails->suburb = $this->address->city ?? null;
        $senderDetails->city = $this->address->city ?? null;
        $senderDetails->state = $this->address->province ?? null;
        $senderDetails->post_code = $this->address->zip ?? null;
        $senderDetails->country = $this->address->country ?? null;

        return $senderDetails;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.store.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name', 'company_name', 'email'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new StoreDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return StoreDataImporter::getExportableFields();
    }
}
