<?php

namespace App\Models;

use App\Abstractions\IsLineInterface;
use App\Abstractions\UniqueFieldsInterface;
use App\Contracts\HasReference;
use App\Helpers;
use App\Models\Concerns\HasAccountingTransactionLine;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\InventoryManagement\NegativeInventoryEvent;
use App\Services\InventoryManagement\PositiveInventoryEvent;
use App\Services\StockTake\OpenStockTakeException;
use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Collection;

/**
 * Class StockTakeItem.
 *
 * @property-read int $id
 * @property int $stock_take_id
 * @property int $product_id
 * @property float $qty_counted
 * @property float $unit_cost
 * @property float $snapshot_inventory
 * @property-read float $quantity_adjusted
 * @property Product $product
 * @property StockTake $stockTake
 * @property-read  Collection|ProductInventory[] $productInventories
 * @property-read  ProductInventory|null $product_inventory
 * @property-read  Collection|FifoLayer[] $fifoLayers
 * @property-read  FifoLayer $creationFifoLayer
 * @property-read  Collection|InventoryMovement[] $inventoryMovements
 */
class StockTakeItem extends Model implements HasReference, NegativeInventoryEvent, PositiveInventoryEvent, IsLineInterface, UniqueFieldsInterface
{
    use HasAccountingTransactionLine, HasFactory;

    /**
     * @var array
     */
    protected $fillable = [
        'id',
        'stock_take_id',
        'product_id',
        'snapshot_inventory',
        'qty_counted',
        'unit_cost',
        'created_at',
        'updated_at',
    ];

    protected $touches = ['stockTake'];

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    public function getLineParent(): Model
    {
        return $this->stockTake;
    }

    public static function getUniqueFields(): array
    {
        return ['stock_take_id', 'product_id'];
    }


    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function stockTake(): BelongsTo
    {
        return $this->belongsTo(StockTake::class);
    }

    public function inventoryMovements(): MorphMany
    {
        $relation = $this->morphMany(InventoryMovement::class, 'link');

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (InventoryMovement $inventoryMovement) {
                $inventoryMovement->delete();
            });
        });

        return $relation;
    }

    public function fifoLayers(): MorphMany
    {
        $relation = $this->morphMany(FifoLayer::class, 'link');

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (FifoLayer $fifoLayer) {
                $fifoLayer->delete();
            });
        });

        return $relation;
    }

    public function creationFifoLayer(): MorphOne
    {
        return $this->morphOne(FifoLayer::class, 'link');
    }

    public function getReference(): string
    {
        return $this->stockTake->getReference();
    }

    public function getDrawer(): Model
    {
        return $this->stockTake->getDrawer();
    }

    // TODO: Why use an accessor?  The data should be cached in the database.
//    public function getUnitCostAttribute()
//    {
//        // If there isn't an override, we fetch from the
//        // active fifo layer
//
//        $this->product->loadMissing('fifoLayers');
//        $activeFifoLayer = $this->product->getCurrentFifoLayerForWarehouse($this->stockTake->warehouse);
//        if ($activeFifoLayer) {
//            return $activeFifoLayer->avg_cost;
//        } else {
//            // Fifo layer is fully used, we default to
//            // the last one
//            /** @var FifoLayer $lastFifoLayer */
//            $lastFifoLayer = $this->product->fifoLayers()->latest('id')->first();
//            if ($lastFifoLayer) {
//                return $lastFifoLayer->avg_cost ?? null;
//            }
//        }
//
//        return null;
//    }

    /**
     * @throws Exception
     */
    public function delete()
    {
        if ($this->stockTake->status == StockTake::STOCK_TAKE_STATUS_CLOSED) {
            $manager = InventoryManager::with(
                $this->stockTake->warehouse_id,
                $this->product
            );

            if ($this->qty_counted > $this->snapshot_inventory) {
                /**
                 * We reverse the positive adjustment
                 * created for the stock take.
                 */
                $manager->removeAllStockFrom($this);
            } else {
                $manager->reverseNegativeEvent($this);
            }
        }

        return parent::delete();
    }

    public function productInventories()
    {
        return $this->hasMany(ProductInventory::class, 'product_id', 'product_id');
    }

    public function accountingTransactionLine()
    {
        return $this->morphOne(AccountingTransactionLine::class, 'link');
    }

    public function getProductInventoryAttribute()
    {
        return $this->productInventories->firstWhere('warehouse_id', $this->stockTake->warehouse_id);
    }

    public function createFifoLayer(int $quantity, ?float $unitCost = null, ?int $productId = null): FifoLayer
    {
        $unitCost = $unitCost ?? $this->product->getUnitCostAtWarehouse($this->stockTake->warehouse_id);

        // We create a fifo layer
        $fifoLayer = new FifoLayer;
        $fifoLayer->product_id = $this->product_id;
        $fifoLayer->original_quantity = abs($quantity);
        $fifoLayer->fulfilled_quantity = 0;
        $fifoLayer->total_cost = $unitCost * $quantity;
        $fifoLayer->fifo_layer_date = $this->getEventDate();
        $fifoLayer->warehouse_id = $this->stockTake->warehouse_id;
        $this->fifoLayers()->save($fifoLayer);

        // Update value changed
        $this->stockTake->addValueChanged($unitCost * $quantity);

        return $fifoLayer;
    }

    public function getUnitCost(): float
    {
        return $this->unit_cost ?? $this->product->getUnitCostAtWarehouse($this->stockTake->warehouse_id);
    }

    public function createInventoryMovements(int $quantity, FifoLayer $fifoLayer): void
    {
        $movement = new InventoryMovement;
        $movement->inventory_movement_date = $this->getEventDate();
        $movement->quantity = $quantity;
        $movement->product_id = $this->product_id;
        $movement->type = InventoryMovement::TYPE_STOCK_TAKE;
        $movement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $movement->fifo_layer = $fifoLayer->id;
        $movement->warehouse_id = $this->stockTake->warehouse_id;
        $this->inventoryMovements()->save($movement);
    }

    public function getOriginatingMovement(): ?InventoryMovement
    {
        return $this->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '>', 0)
            ->first();
    }

    public function getFifoLayer(): ?FifoLayer
    {
        return $this->fifoLayers()->firstOrFail();
    }

    public function createInventoryMovementsForLayers(array $layers, ?Carbon $dateOverride = null): void
    {
        $cost = 0;

        foreach ($layers as $layerInfo) {
            if (! ($layerInfo['layer'] instanceof FifoLayer)) {
                throw new \InvalidArgumentException('Only Fifo Layers are applicable for stock takes. '.$layerInfo['layer_type'].' encountered.');
            }

            /** @var FifoLayer $fifoLayer */
            $fifoLayer = $layerInfo['layer'];

            $movement = new InventoryMovement;
            $movement->inventory_movement_date = $this->getEventDate();
            $movement->quantity = -abs($layerInfo['quantity']);
            $movement->product_id = $this->product_id;
            $movement->type = InventoryMovement::TYPE_STOCK_TAKE;
            $movement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
            $movement->warehouse_id = $this->stockTake->warehouse_id;

            $cost += $layerInfo['quantity'] * $layerInfo['avg_cost'];
            $movement->link_id = $this->id;
            $movement->link_type = self::class;
            $fifoLayer->inventoryMovements()->save($movement);
        }

        // Update the value change to the stock take
        $this->stockTake->addValueChanged(-$cost);
    }

    public function getReductionActiveMovements(): Arrayable
    {
        return $this->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '<', 0)
            ->get();
    }

    public function clearMovements(): void
    {
        $this->inventoryMovements()->delete();
    }

    /**
     * @throws OpenStockTakeException
     */
    public function reduceQtyWithSiblings(int $quantity, InventoryMovement $representingMovement): void
    {
        if ($representingMovement->quantity > 0) {
            $representingMovement->quantity = max(0, $representingMovement->quantity - $quantity);
        } elseif ($representingMovement->quantity < 0) {
            $representingMovement->quantity = min(0, $representingMovement->quantity + $quantity);
        }

        if ($representingMovement->quantity == 0) {
            $representingMovement->delete();
        } else {
            $representingMovement->save();
        }
    }

    public function getQuantityAdjustedAttribute(): float
    {
        return $this->qty_counted - $this->snapshot_inventory;
    }

    public function getLinkType(): string
    {
        return self::class;
    }

    public function getId(): int
    {
        return $this->id;
    }

    public function refreshFromFifoCostUpdate(float $amount = 0): void
    {
        $this->updated_at = now();
        $this->save();
    }

    public function getProductId(): int
    {
        return $this->product_id;
    }

    public function getWarehouseId(): int
    {
        return $this->stockTake->warehouse_id;
    }

    public function getEventDate(): Carbon
    {
        return $this->stockTake->date_count->max(Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE), 'UTC'));
    }

    public function getQuantity(): int
    {
        return $this->quantity_adjusted;
    }

    public function getType(): string
    {
        return InventoryMovement::TYPE_STOCK_TAKE;
    }

    public function backorderQueueReleases()
    {
        return $this->morphMany(BackorderQueueRelease::class, 'link');
    }
}
