<?php

namespace App\Models\Shopify;

use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\IntegrationShippingMethod;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * Class RemovalShipment.
 *
 *
 * @property string $_id
 * @property int $integration_instance_id
 * @property string $shopify_id
 * @property string $name
 * @property bool $active
 * @property bool $service_discovery
 * @property string $carrier_service_type
 * @property string $admin_graphql_api_id
 * @property string $format
 * @property string $callback_url
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $deleted_at
 * @property string $code same shopify_id
 * @property bool $domestic always true
 * @property-read IntegrationShippingMethod|null $integration_shipping_method
 */
class ShopifyShippingService extends Model implements Filterable, Sortable
{
    use HasFilters, HasSort, SoftDeletes;

    protected $guarded = [];

    public $dataTableKey = 'shopify.shipping_service';

    public $dataTableClass = self::class;

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getIntegrationShippingMethodAttribute()
    {
        return IntegrationShippingMethod::with([])
            ->where('code', $this->code)
            ->where('integration_instance_id', $this->integration_instance_id)
            ->first();
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function availableColumns()
    {
        return config('data_table.shopify.shipping_service.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return ['name', 'code'];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }
}
