<?php

namespace App\Models\Shopify;

use App\Abstractions\Integrations\SalesChannels\ActiveSalesChannelProductInterface;
use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Exporters\MapsExportableFields;
use App\Helpers;
use App\Http\Resources\Shopify\ShopifyProductListingResource;
use App\Http\Resources\Shopify\ShopifyProductResource;
use App\Integrations\Listing;
use App\Integrations\Listings\ListingsHelper;
use App\Integrations\Listings\ProductMapper;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\IntegrationInstance;
use App\Models\Product as SkuProduct;
use App\Models\ProductListing;
use App\Models\ProductPricingTier;
use App\Models\Setting;
use App\Services\Shopify\ShopifyListingMapper;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Kirschbaum\PowerJoins\PowerJoins;

class ShopifyProduct extends Model implements ActiveSalesChannelProductInterface, Filterable, Listing, MapsExportableFields, Sortable
{
    use BulkImport, HandleDateTimeAttributes, HasFactory, HasFilters, HasSort, ListingsHelper, PowerJoins;

    protected $table = 'shopify_products';

    public const CREATED_AT = 'sku_created_at';

    public const UPDATED_AT = 'sku_updated_at';

    protected $fillable = [
        'variant_id',
        'integration_instance_id',
        'json_object',
        'removed_from_shopify',
        'has_orders',
        'created_at',
        'updated_at',
    ];

    protected $casts = [
        'variant_id' => 'integer',
        'integration_instance_id' => 'integer',
        'json_object' => 'array',
        'has_orders' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'images' => 'array',
    ];

    public $dataTableKey = 'shopify.product';

    const SELLER_SKU_COLUMN = 'sku';

    //Relations
    public function integrationInstance()
    {
        return $this->belongsTo(IntegrationInstance::class);
    }

    public function productListing()
    {
        return $this->hasOne(ProductListing::class, 'document_id')->where('document_type', self::class);
    }

    //Functions to replace
    public static function getProductMapper(IntegrationInstance $integrationInstance): ProductMapper
    {
        return new ShopifyListingMapper($integrationInstance);
    }

    public function isActive(): bool
    {
        return $this->json_object['status'] == 'active' && ! $this->removed_from_shopify;
    }

    private function mapToProduct($productId)
    {
        $mapping = [
            'variant_id' => $this['variant_id'],
            'product_id' => $productId,
        ];

        self::getProductMapper($this->integrationInstance)->mapListings([$mapping]);

        $this->mapped_at = now();
        $this->save();
    }

    public function createSkuProduct(array $fieldMapping = [])
    {
        $skuValue = empty($fieldMapping) ? $this->getValue('sku') : $this->getValue($this->getMappedField('sku', $fieldMapping));
        //SKU-4385 - Uncomment code if this is good.
        if (count($fieldMapping) > 0) {
            $skuParser = collect($fieldMapping)->where('listing_field', 'sku')->first();

            if ($skuParser && @$skuParser['parsers']) {
                $skuValue = $this->parseRules(
                    $this->getValue('sku'),
                    $skuParser['parsers'] ?? []
                );
            }
        }

        if (empty(trim($skuValue))) {
            return ['id' => $this['_id'], 'reason' => 'The listing has empty seller sku, skipping.'];
        }
        // Make sure the product doesn't exist already
        $product = SkuProduct::with([])->where('sku', $skuValue)->first();

        if ($product) {
            // We check if listing isn't mapped, we map it to the product
            if (! $this->productListing) {
                $this->mapToProduct($product->id);
            }

            return ['id' => $this['_id'], 'reason' => "Product with sku '{$skuValue}' already exists, skipping."];
        }

        if ($this->productListing) {
            // Skip if listing is already mapped.
            return [
                'id' => $this['_id'],
                'reason' => 'Listing already mapped to sku product, skipping.',
            ];
        }

        // We map the the fields to create the needed payload and
        // create the product
        $defaultDimension = Helpers::setting(Setting::KEY_DEFAULT_DIMENSION_UNIT, null, true);
        $defaultWeightUnit = Helpers::setting(Setting::KEY_DEFAULT_WEIGHT_UNIT, null, true);
        if (! empty($fieldMapping)) {
            // We create the product with mapping.
            $product = $this->createSkuProductWithMapping($fieldMapping, [
                'sku' => $this['sku'],
                'type' => SkuProduct::TYPE_STANDARD,
                'name' => $this['variant_title'],
            ]);
        } else {
            // We use the needed fields to create the product.
            $payload = [
                'sku' => $this['sku'],
                'name' => $this['variant_title'],
                'type' => SkuProduct::TYPE_STANDARD,
                'weight' => $this['weight'],
                'weight_unit' => $defaultWeightUnit,
                'dimension_unit' => $defaultDimension,
                'case_dimension_unit' => $defaultDimension,
            ];

            $product = new SkuProduct($payload);
            $product->save();

            //      $product->setBrandName( $this['matchingProduct']['AttributeSets'][0]['Brand'] ?? null );

            $defaultPricingTier = ProductPricingTier::with([])->where('is_default', 1)->first();
            $payload['pricing'] = $defaultPricingTier ? [
                [
                    'product_pricing_tier_name' => $defaultPricingTier->name,
                    'price' => $this['price'],
                ],
            ] : [];

            $product->setPricing($payload['pricing']);

            if ($this['image']) {
                $product->addImages([
                    [
                        'url' => $this['image'],
                    ],
                ]);
            }
        }

        if ($product) {
            $this->mapToProduct($product->id);
        }

        return true;
    }

    public function getListingUrl(): ?string
    {
        $shopUrl = $this->integrationInstance->connection_settings['shop_url'] ?? null;
        if (! $shopUrl || ! $this['handle']) {
            return null;
        }

        return "https://{$shopUrl}/products/{$this['handle']}";
    }

    public static function getResource()
    {
        return ShopifyProductResource::class;
    }

    public static function makeExportData($builder): array
    {
        $records = $builder->get()->map(function (self $listing) {
            return [
                (string) $listing['variant_id'],
                $listing->productListing ? $listing->productListing->product->sku : null,
                $listing['sku'],
                $listing['variant_title'],
                $listing['price'],
                $listing['vendor'],
                $listing['product_type'],
            ];
        })->toArray();

        return [
            'filename' => 'shopify_listings',
            'headers' => [
                'variant_id',
                'mapped_sku',
                'sku',
                'variant_title',
                'price',
                'vendor',
                'product_type',
            ],
            'records' => $records,
            'xlsxheaders' => [
                'variant_id' => 'string',
                'mapped_sku' => 'string',
                'sku' => 'string',
                'variant_title' => 'string',
                'price' => 'price',
                'vendor' => 'string',
                'product_type' => 'string',
            ],
        ];
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', false, 'ID'),
            'variant_id' => Exporter::makeExportableField('variant_id', false),
            'sku' => Exporter::makeExportableField('sku', false),
            'price' => Exporter::makeExportableField('price', false),
            'variant_title' => Exporter::makeExportableField('variant_title', false),
            'product_title' => Exporter::makeExportableField('product_title', false),
            'inventory_quantity' => Exporter::makeExportableField('inventory_quantity', false),
            'mapped_sku.id' => Exporter::makeExportableField('mapped_sku.id', false),
            'mapped_sku.name' => Exporter::makeExportableField('mapped_sku.name', false),
            'mapped_sku.sku' => Exporter::makeExportableField('mapped_sku.sku', false),
            'mapped_sku' => Exporter::makeExportableField('mapped_sku', false),
            'image' => Exporter::makeExportableField('image', false),
        ];
    }

    public function scopeFilterProduct(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $integrationInstance = request()->route('integration_instance');
        $integrationInstance = is_numeric($integrationInstance) ? IntegrationInstance::with([])->find($integrationInstance) : $integrationInstance;

        // without whereHas product
        if ($operator == 'isEmpty') {
            return $builder->where('product');
        } elseif ($operator == 'isNotEmpty') {
            return $builder->whereNotNull('product');
        } else {
            return $builder->whereHas('productListing.product', function ($query) use ($operator, $value) {
                $query->filterKey('sku', $operator, $value);
            });
        }
    }

    public function scopeFilterImage(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $key = (int) $relation['key'] - 1;
        if ($key < 0) {
            return $builder;
        }

        $builder = $builder->where('integration_instance_id', request()->route('integration_instance')->id);
        $key = "images.{$key}.src";

        switch ($operator) {
            case 'isNotEmpty':
                return $builder->whereNull($key); // For some reason, this and isEmpty are being switched. Requires more investigation.
            case 'isEmpty':
                return $builder->whereNotNull($key);
            case 'contains':
                return $builder->where($key, 'like', '%'.$value.'%', $conjunction);
            case 'doesNotContain':
                return $builder->where($key, 'not like', '%'.$value.'%', $conjunction);
            case 'startsWith':
                return $builder->where($key, 'like', $value.'%', $conjunction);
            case 'endsWith':
                return $builder->where($key, 'like', '%'.$value, $conjunction);
        }

        return $builder;
    }

    public function availableColumns()
    {
        return config("data_table.{$this->dataTableKey}.columns");
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        $columns = collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
        $withMostImages = self::with([])
            ->orderBy('image_count', 'DESC')
            ->first();
        if ($withMostImages) {
            $maxImages = $withMostImages->image_count;
            for ($i = 1; $i <= $maxImages; $i++) {
                $columns[] = "image$i";
            }
        }

        return $columns;
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['sku', 'product_id', 'variant_id', 'variant_title'];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public static function getProductListingResource()
    {
        return ShopifyProductListingResource::class;
    }

    public static function getWithMostImages(): ?self
    {
        return self::with([])
            ->orderBy('image_count', 'DESC')
            ->first();
    }

    public function isUsed(): bool
    {
        // always deletable, don't delete this function because it used by bulk operation trait
        return false;
    }

    public function scopeLatestProduct(Builder $builder, int $integrationInstanceId, ?string $downloadedBy = null, string $column = 'updated_at')
    {
        $builder->where('integration_instance_id', $integrationInstanceId)
            ->when($downloadedBy, function (Builder $builder) use ($downloadedBy) {
                $builder->where(function (Builder $builder) use ($downloadedBy) {
                    $builder->where('downloaded_by', $downloadedBy);
                    $builder->orWhere('updated_by', $downloadedBy);
                });
            })
            ->latest($column);
    }

    public static function getSkuField(): string
    {
        return 'sku';
    }

    public static function getUniqueField(): string
    {
        return 'variant_id';
    }

    public function scopeFilterProductListing(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'or' ? 'orWhere' : 'where';
        return $builder->$function(function ($query) use ($relation, $operator, $value, $conjunction) {
            if ($operator == 'isEmpty') {
                return $query->whereNull('products.id');
            } elseif ($operator == 'isNotEmpty') {
                return $query->whereNotNull('products.id');
            }else {
                return $query->where("products.{$relation['subkey']}", $operator, $value);
            }
        });
    }

    public function scopeJoinProductListings(Builder $builder): Builder{
        return $builder
            ->leftJoin('product_listings', 'product_listings.document_id', '=', 'shopify_products.id')
            ->leftJoin('products', 'product_listings.product_id', '=', 'products.id');
    }
}
