<?php

namespace App\Models;

use App\Models\Concerns\LogsActivity;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Spatie\Activitylog\LogOptions;

/**
 * Class Setting.
 *
 *
 * @property int $id
 * @property int|null $user_id
 * @property string $key
 * @property string|null $description
 * @property string $type
 * @property string|null $value
 * @property string|null $default_value
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 */
class Setting extends Model
{
    use LogsActivity;

    /**
     * Setting Types.
     */
    const TYPE_STRING = 'string';

    const TYPE_DATE = 'date';

    const TYPE_CHECKBOX = 'checkbox';

    const TYPE_INTEGER = 'integer';

    const TYPE_FLOAT = 'float';

    const TYPE_TIME = 'time';

    const TYPE_JSON = 'json';

    const TYPE_EMAIL = 'email';

    /*
    |--------------------------------------------------------------------------
    | Setting Keys
    |--------------------------------------------------------------------------
    */

    // inventory settings
    const KEY_ADJUST_FORECAST_OUT_OF_STOCK = 'adjust_forecast';

    const KEY_DAYS_SALES_HISTORY = 'days_of_sales_history';

    const KEY_MAX_PROJECTION_DAYS = 'max_projection_days';

    const KEYS_INVENTORY_FORECASTING
        = [
            self::KEY_ADJUST_FORECAST_OUT_OF_STOCK,
            self::KEY_DAYS_SALES_HISTORY,
            self::KEY_MAX_PROJECTION_DAYS,
        ];

    // nominal codes mapping
    const KEY_NC_MAPPING_SALES_ORDERS = 'nominal_code_mapping_sales_orders';

    const KEY_NC_MAPPING_CN_SALES_ORDERS = 'nominal_code_mapping_credit_notes_on_sales_orders';

    const KEY_NC_MAPPING_TAX_SALES_ORDERS = 'nominal_code_mapping_tax_on_sales_orders';

    const KEY_NC_MAPPING_SHIPPING_SALES_ORDERS = 'nominal_code_mapping_shipping_revenue_from_sales_orders';

    const KEY_NC_MAPPING_CN_PURCHASE_ORDERS = 'nominal_code_mapping_credit_notes_on_purchase_orders';

    const KEY_NC_MAPPING_CN_CUSTOMERS = 'nominal_code_mapping_credit_notes_for_customers';

    const KEY_NC_MAPPING_CN_SUPPLIERS = 'nominal_code_mapping_credit_notes_for_suppliers';

    const KEY_NC_MAPPING_COGS = 'nominal_code_mapping_cost_of_goods_sold';

    const KEY_NC_MAPPING_INVENTORY_CONTROL = 'nominal_code_mapping_inventory_control';

    const KEY_NC_MAPPING_ACCRUED_PURCHASES = 'nominal_code_mapping_accrued_purchases';

    const KEY_NC_MAPPING_INVENTORY_IN_TRANSIT = 'nominal_code_mapping_inventory_in_transit';

    const KEY_NC_MAPPING_ROUNDING = 'nominal_code_mapping_rounding';

    const KEY_NC_MAPPING_RECEIVING_DISCREPANCIES = 'nominal_code_mapping_receiving_discrepancies';

    const KEYS_NC_MAPPING
        = [
            self::KEY_NC_MAPPING_SALES_ORDERS,
            //self::KEY_NC_MAPPING_CN_SALES_ORDERS,
            //self::KEY_NC_MAPPING_TAX_SALES_ORDERS,
            self::KEY_NC_MAPPING_SHIPPING_SALES_ORDERS,
            //self::KEY_NC_MAPPING_CN_PURCHASE_ORDERS,
            //self::KEY_NC_MAPPING_CN_CUSTOMERS,
            //self::KEY_NC_MAPPING_CN_SUPPLIERS,
            self::KEY_NC_MAPPING_INVENTORY_CONTROL,
            self::KEY_NC_MAPPING_ACCRUED_PURCHASES,
            self::KEY_NC_MAPPING_COGS,
            self::KEY_NC_MAPPING_ROUNDING,
            self::KEY_NC_MAPPING_INVENTORY_IN_TRANSIT,
            self::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES,
        ];

    // sales order settings
    const KEY_SO_PREFIX = 'sales_order_prefix';

    const KEY_SO_NUM_OF_DIGITS = 'sales_order_num_of_digits';

    const KEY_SO_START_NUMBER = 'sales_order_start_number';

    const KEY_SO_DEFAULT_STORE = 'sales_order_default_store';

    const KEY_SO_DEFAULT_SM_DOMESTIC = 'sales_order_default_shipping_method_domestic';

    const KEY_SO_DEFAULT_SM_INTERNATIONAL = 'sales_order_default_shipping_method_international';

    const KEY_SO_DEFAULT_HANDLING_DAYS = 'sales_order_default_handling_days';

    const KEY_SO_SAME_DAY_SHIPPING_CUTOFF = 'sales_order_same_day_shipping_cutoff';

    const KEY_SO_DEFAULT_PAYMENT_TYPE = 'sales_order_default_payment_type';

    const KEY_SO_SHOW_TAX_RATE = 'sales_order_show_tax_rate';

    const KEY_SO_SHOW_TAX_AMOUNT = 'sales_order_show_tax_amount';

    const KEY_SO_UNPAID_ORDERS_AS_RESERVED = 'unpaid_orders_as_reserved';

    const KEY_SO_SHIPPING_PROVIDER_SPLIT_BUNDLES = 'shipping_provider_split_bundles';

    const KEYS_SO
        = [
            self::KEY_SO_PREFIX,
            self::KEY_SO_NUM_OF_DIGITS,
            self::KEY_SO_START_NUMBER,
            self::KEY_SO_DEFAULT_STORE,
            self::KEY_SO_DEFAULT_SM_DOMESTIC,
            self::KEY_SO_DEFAULT_SM_INTERNATIONAL,
            self::KEY_SO_DEFAULT_HANDLING_DAYS,
            self::KEY_SO_SAME_DAY_SHIPPING_CUTOFF,
            self::KEY_SO_DEFAULT_PAYMENT_TYPE,
            self::KEY_SO_SHOW_TAX_RATE,
            self::KEY_SO_SHOW_TAX_AMOUNT,
            self::KEY_SO_UNPAID_ORDERS_AS_RESERVED,
            self::KEY_SO_SHIPPING_PROVIDER_SPLIT_BUNDLES,
        ];

    // supplier settings
    const KEY_SUPPLIER_LEADTIME = 'supplier_leadtime';

    const KEY_SUPPLIER_MOQ = 'supplier_moq';

    // Purchase Orders
    const KEY_PO_PREFIX = 'purchase_order_prefix';

    const KEY_PO_NUM_DIGITS = 'purchase_order_num_of_digits';

    const KEY_PO_START_NUMBER = 'purchase_order_start_number';

    const KEY_PO_FORMAT = 'purchase_order_format';

    const KEY_PO_DEFAULT_WAREHOUSE = 'purchase_order_default_warehouse';

    const KEY_PO_DEFAULT_SHIPPING_METHOD = 'purchase_order_default_shipping_method';

    const KEY_PO_LEAD_TIME = 'purchase_order_lead_time';

    const KEY_PO_DEFAULT_PAYMENT_TYPE = 'purchase_order_default_payment_type';

    const KEY_PO_DEFAULT_STORE = 'purchase_order_default_store';

    const KEY_PO_CC_OUTGOING = 'purchase_order_cc_outgoing';

    const KEY_PO_BUILDER_DEFAULT_ROUNDING_METHOD = 'purchase_order_builder_default_rounding_method';

    const KEY_PO_BUILDER_DEFAULT_USE_MOQ = 'purchase_order_builder_default_use_moq';

    const KEY_PO_BUILDER_DEFAULT_USE_LEAD_TIME = 'purchase_order_builder_default_use_lead_time';

    const KEY_PO_BUILDER_DEFAULT_FORECASt_TYPE = 'purchase_order_builder_default_forecast_type';

    const KEY_PO_BUILDER_DEFAULT_TARGET_QUANTITY = 'purchase_order_builder_target_quantity';

    const KEY_PO_BUILDER_DEFAULT_SALES_HISTORY_DAYS = 'purchase_order_builder_sales_history_days';

    const KEY_PO_BUILDER_DEFAULT_TARGET_DAYS_OF_STOCK = 'purchase_order_builder_target_days_of_stock';

    const KEYS_PO
        = [
            self::KEY_PO_PREFIX,
            self::KEY_PO_NUM_DIGITS,
            self::KEY_PO_START_NUMBER,
            self::KEY_PO_FORMAT,
            self::KEY_PO_DEFAULT_WAREHOUSE,
            self::KEY_PO_DEFAULT_SHIPPING_METHOD,
            self::KEY_PO_LEAD_TIME,
            self::KEY_PO_DEFAULT_PAYMENT_TYPE,
            self::KEY_PO_DEFAULT_STORE,
            self::KEY_PO_CC_OUTGOING,
            self::KEY_PO_BUILDER_DEFAULT_FORECASt_TYPE,
            self::KEY_PO_BUILDER_DEFAULT_USE_LEAD_TIME,
            self::KEY_PO_BUILDER_DEFAULT_USE_MOQ,
            self::KEY_PO_BUILDER_DEFAULT_ROUNDING_METHOD,
            self::KEY_PO_BUILDER_DEFAULT_TARGET_QUANTITY,
            self::KEY_PO_BUILDER_DEFAULT_SALES_HISTORY_DAYS,
            self::KEY_PO_BUILDER_DEFAULT_TARGET_DAYS_OF_STOCK,

        ];

    const KEYS_LOCATIONS
        = [
            self::KEY_PRIORITIZE_CLOSEST_WAREHOUSE_TO_CUSTOMER,
            self::KEY_WAREHOUSE_IGNORE_SUPPLIER_TYPE,
            self::KEY_WAREHOUSE_IGNORE_AMAZON_FBA_TYPE,
            self::KEY_AUTO_SPLIT_SALES_ORDER_LINE_ACROSS_WAREHOUSES,
            self::KEY_NEVER_SPLIT_SALES_ORDERS_ACROSS_WAREHOUSES,
        ];

    // Warehouse Transfers
    const KEY_WH_TRANSFER_PREFIX = 'warehouse_transfer_prefix';

    const KEY_WH_TRANSFER_NUM_OF_DIGITS = 'warehouse_transfer_num_of_digits';

    const KEY_WH_TRANSFER_SHIPPING_METHOD = 'warehouse_transfer_shipping_method';

    // Sales Credit
    const KEY_SC_PREFIX = 'sales_credit_prefix';

    const KEY_SC_NUM_DIGITS = 'sales_credit_num_of_digits';

    const KEY_SC_START_NUMBER = 'sales_credit_start_number';

    const KEY_SC_SHOW_BARCODE_IN_FRONTEND = 'sales_credit_show_barcode_in_frontend';

    const KEY_SC_DEFAULT_WAREHOUSE = 'sales_credit_default_warehouse';

    const KEYS_SC
        = [
            self::KEY_SC_PREFIX,
            self::KEY_SC_NUM_DIGITS,
            self::KEY_SC_START_NUMBER,
            self::KEY_SC_SHOW_BARCODE_IN_FRONTEND,
            self::KEY_SC_DEFAULT_WAREHOUSE,
        ];

    // Seller country
    const KEY_SELLER_COUNTRY = 'seller_country';

    // Warehouse priority
    const KEY_WAREHOUSE_PRIORITY = 'warehouse_priority';

    const KEY_PRIORITIZE_CLOSEST_WAREHOUSE_TO_CUSTOMER = 'prioritize_closest_warehouse_to_customer';

    const KEY_WAREHOUSE_IGNORE_SUPPLIER_TYPE = 'warehouse_ignore_supplier_type';

    const KEY_WAREHOUSE_IGNORE_AMAZON_FBA_TYPE = 'warehouse_ignore_amazon_fba_type';

    const KEY_AUTO_SPLIT_SALES_ORDER_LINE_ACROSS_WAREHOUSES = 'auto_split_sales_order_line_across_warehouses';

    const KEY_NEVER_SPLIT_SALES_ORDERS_ACROSS_WAREHOUSES = 'never_split_sales_orders_across_warehouses';

    // Inventory Start Date
    const KEY_INVENTORY_START_DATE = 'inventory_start_date';

    const KEY_DEFAULT_TIMEZONE = 'default_timezone';

    // Payment types mapping

    const KEY_PAYMENT_TYPE_MAPPING_CASH = 'payment_type_mapping_cash';

    const KEY_PAYMENT_TYPE_MAPPING_WIRE = 'payment_type_mapping_wire';

    const KEY_PAYMENT_TYPE_MAPPING_CREDIT = 'payment_type_mapping_credit';

    const KEY_PAYMENT_TYPE_MAPPING_AMAZON_PAYMENT = 'payment_type_mapping_amazon_payment';

    const KEY_PAYMENT_TYPE_MAPPING_AMAZON = 'payment_type_mapping_amazon';

    const KEY_PAYMENT_TYPE_MAPPING_SHOPIFY_PAYMENT = 'payment_type_mapping_shopify_payment';

    const KEYS_PAYMENT_TYPES_MAPPING
        = [
            self::KEY_PAYMENT_TYPE_MAPPING_CASH,
            self::KEY_PAYMENT_TYPE_MAPPING_WIRE,
            self::KEY_PAYMENT_TYPE_MAPPING_CREDIT,
            self::KEY_PAYMENT_TYPE_MAPPING_AMAZON_PAYMENT,
            self::KEY_PAYMENT_TYPE_MAPPING_AMAZON,
            self::KEY_PAYMENT_TYPE_MAPPING_SHOPIFY_PAYMENT,
        ];

    const KEY_DEFAULT_DATE_FORMAT = 'default_date_format';

    // Dimensions
    const KEY_DEFAULT_WEIGHT_UNIT = 'default_weight_unit';

    const KEY_DEFAULT_DIMENSION_UNIT = 'default_dimension_unit';

    const KEY_FULFILL_ON_MAPPED_SHIPPING_METHOD = 'fulfill_on_mapped_shipping_method';

    const KEY_AUTO_FULFILLMENT_EMAIL = 'auto_fulfillment_email';

    const KEY_AUTO_FULFILLMENT_FAILURE_NOTIFICATIONS = 'auto_fulfillment_failure_notifications';

    const KEY_BLEMISHED_SKU_PATTERN = 'blemished_sku_pattern';

    const KEY_USE_BLEMISHED_SKU_PATTERN = 'use_blemished_sku_pattern';

    const KEY_ASN_CC_EMAIL = 'asn_cc_email';

    const KEY_ASN_FILE_FORMAT = 'asn_file_format';

    const KEY_FINANCIAL_ALERTS_ENABLED = 'financial_alerts_enabled';
    const KEY_FINANCIAL_ALERTS_LOW_MARGIN_THRESHOLD = 'financial_alerts_low_margin_threshold';
    const KEY_FINANCIAL_ALERTS_IGNORE_FILTERS = 'financial_alerts_ignore_filters';
    const KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE = 'financial_alerts_notification_type';
    const KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE_EMAIL = 'email';
    const KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE_SLACK = 'slack';
    const KEY_FINANCIAL_ALERTS_NOTIFICATION_DESTINATION = 'financial_alerts_notification_destination';
    const KEY_FINANCIAL_ALERTS_NOTIFICATION_CONDITIONS = 'financial_alerts_notification_conditions';
    const KEY_FINANCIAL_ALERTS_LAST_NOTIFICATION_DATE = 'financial_alerts_last_notification_date';
    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [];
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->id;
    }

    /*
    |--------------------------------------------------------------------------
    | Accounting
    |--------------------------------------------------------------------------
    */

    const KEY_ACCOUNTING_ENABLED = 'accounting_enabled';

    const KEY_ACCOUNTING_START_DATE = 'accounting_start_date';

    const KEY_ACCOUNTING_LOCK_DATE = 'accounting_lock_date';

    // fetched settings, to avoid fetch it again
    public static $fetchedSettings = [];

    protected $fillable = ['key', 'description', 'type', 'value'];

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    /**
     * Set "type" attribute.
     */
    public function setTypeAttribute($type): static
    {
        $this->attributes['type'] = $type;

        $this->setValueCasting($type);

        return $this;
    }

    /**
     * Get the casts array.
     */
    public function getCasts(): array
    {
        $this->setValueCasting($this->attributes['type'] ?? null);

        return parent::getCasts();
    }

    /**
     * Set casting to setting value.
     */
    private function setValueCasting(?string $type)
    {
        if ($type) {
            switch ($type) {
                case self::TYPE_CHECKBOX:
                    $type = 'boolean';
                    break;
                case self::TYPE_TIME:
                    $type = 'datetime:Y-m-dTH:i:s';
                    break;
                case self::TYPE_EMAIL:
                    $type = 'string';
                    break;
            }

            $this->casts = ['value' => $type, 'default_value' => $type];
        }
    }

    public function isNullable(): bool
    {
        return in_array($this->key, [
            Setting::KEY_ASN_CC_EMAIL
        ]);
    }

    /**
     * Get a setting value by key.
     *
     *
     * @return mixed|static
     */
    public static function getValueByKey($key)
    {
        $setting = self::with([])->where('key', $key)->first();

        return $setting ? $setting->value : null;
    }

    public static function defaultWarehousePriority(): string
    {
        $priority = self::getValueByKey(self::KEY_WAREHOUSE_PRIORITY);

        if (is_null($priority)) {
            $priority = json_encode([
                Warehouse::query()
                    ->where('name', Warehouse::DEFAULT_WAREHOUSE_NAME)
                    ->where('type', Warehouse::TYPE_DIRECT)
                    ->where('order_fulfillment', 'manual')
                    ->firstOrFail()
                    ->id,
            ]);
        }

        $priority = self::filterPriorityBasedOnWarehouseType(
            $priority,
            self::KEY_WAREHOUSE_IGNORE_AMAZON_FBA_TYPE,
            Warehouse::TYPE_AMAZON_FBA
        );

        $priority = self::filterPriorityBasedOnWarehouseType(
            $priority,
            self::KEY_WAREHOUSE_IGNORE_SUPPLIER_TYPE,
            Warehouse::TYPE_SUPPLIER
        );

        return $priority;
    }

    private static function filterPriorityBasedOnWarehouseType($priority, $key, $type)
    {
        if (self::getValueByKey($key)) {
            $priority = collect(is_string($priority) ? json_decode($priority, true) : $priority)
                ->reject(function ($id) use ($type) {
                    return Warehouse::query()->where('id', $id)->where('type', $type)->exists();
                })
                ->toJson();
        }

        return $priority;
    }

    public static function getBlemishedSkuPattern(): string
    {
        return self::where('key', self::KEY_BLEMISHED_SKU_PATTERN)->value('value') ?? '';
    }
}
