<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Exporters\MapsExportableFields;
use App\Importers\DataImporters\SalesOrderLineFinancialDataImporter;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

/**
 * @property int $id
 * @property int $sales_order_line_id
 * @property float $revenue
 * @property float $revenue_allocated
 * @property float $credits
 * @property-read float $total_revenue
 * @property float $cogs
 * @property float $cogs_returned
 * @property float $cost_allocated
 * @property-read float $total_cost
 * @property-read float $profit
 * @property-read float $profit_margin
 * @property ?Carbon $created_at
 * @property ?Carbon $updated_at
 * @property-read SalesOrderLine $salesOrderLine
 * @property-read FinancialAlert $financialAlert
 */
class SalesOrderLineFinancial extends Model implements Filterable, MapsExportableFields, Sortable, UniqueFieldsInterface
{
    use BulkImport;
    use HasFactory;
    use HasFilters;
    use HasSort;

    const BULK_THRESHOLD = 100;

    protected $guarded = [];

    public static function getUniqueFields(): array
    {
        return ['sales_order_line_id'];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesOrderLine(): BelongsTo
    {
        return $this->belongsTo(SalesOrderLine::class);
    }

    public function financialAlert(): HasOne
    {
        return $this->hasOne(FinancialAlert::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function perUnit(string $attribute): float
    {
        $value = $this->{$attribute} ?? 0;

        if ($this->salesOrderLine->quantity != 0) {
            $value /= $this->salesOrderLine->quantity;
        }

        return $value;
    }

    public static function getExportableFields(): array
    {
        return SalesOrderLineFinancialDataImporter::getExportableFields();
    }

    public function availableColumns()
    {
        return config('data_table.reporting.sales_order_line_financial.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return [];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function delete(): int
    {
        if ($this->financialAlert) {
            $this->financialAlert->delete();
        }

        return parent::delete();
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeStartDate(Builder $query, $date): Builder
    {
        return $query->where('sales_orders.order_date', '>=', Carbon::parse($date));
    }

    public function scopeEndDate(Builder $query, $date): Builder
    {
        return $query->where('sales_orders.order_date', '<', Carbon::parse($date));
    }

    public function scopeSortSalesOrderLine(Builder $builder, array $relation, bool $ascending)
    {

        if($relation['subkey'] == 'defaultSupplierProduct') {
            return $builder->leftJoin('sales_order_lines', 'sales_order_line_financials.sales_order_line_id', '=',
                'sales_order_lines.id')
                ->leftJoin('supplier_products', 'sales_order_lines.product_id', '=', 'supplier_products.product_id')
                ->leftJoin('suppliers', 'supplier_products.supplier_id', '=', 'suppliers.id')
                ->where('supplier_products.is_default', '=', 1)
                ->orderBy('suppliers.name', $ascending ? 'asc' : 'desc');
        }

        $relationInstance = $builder->getModel()->{$relation['name']}();
        if ($relationInstance instanceof BelongsTo) {
            return $this->sortBelongsToRelations($builder, $relation, $ascending);
        }

        $realKey = $relation['combined_key'];

        if (($key['from_json'] ?? false)) {
            return $builder->orderBy($realKey, $ascending ? 'asc' : 'desc');
        }

        return $builder->orderByRaw($this->getSortingSQLQuery($this, $realKey, $ascending));
    }


}
