<?php

namespace App\Models;

use App\Abstractions\FinancialDocumentLineInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrder;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrderLine;
use App\Abstractions\UniqueFieldsInterface;
use App\Contracts\HasReference;
use App\Helpers;
use App\Http\Resources\Magento\SalesOrderLineListingMagentoResource;
use App\Http\Resources\Shopify\SalesOrderLineListingShopifyResource;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\Amazon\FulfilledShipment;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasAccountingTransactionLine;
use App\Models\Concerns\HasCurrencyAttributes;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Concerns\LogsActivity;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Repositories\FifoLayerRepository;
use App\SDKs\ShipStation\Model\OrderItem;
use App\SDKs\ShipStation\Model\Weight;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\InventoryManagement\NegativeInventoryEvent;
use App\Services\SalesOrder\WarehouseRoutingMethod;
use App\Services\StockTake\OpenStockTakeException;
use Awobaz\Compoships\Compoships;
use Carbon\Carbon;
use Carbon\CarbonImmutable;
use Exception;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Kirschbaum\PowerJoins\PowerJoins;
use Modules\Amazon\Http\Resources\SalesOrderLineListingAmazonResource;
use Modules\Ebay\Http\Resources\SalesOrderLineListingEbayResource;
use Modules\WooCommerce\Http\Resources\SalesOrderLineListingWooCommerceResource;
use Spatie\Activitylog\LogOptions;
use Throwable;

/**
 * Class SalesOrderLine.
 *
 *
 * @property int $id
 * @property int $sales_order_id
 * @property int|null $split_from_line_id
 * @property int|null $warehouse_id
 * @property WarehouseRoutingMethod|null $warehouse_routing_method
 * @property string $sales_channel_line_id
 * @property int $product_listing_id
 * @property string $description
 * @property int|null $product_id
 * @property float $amount
 * @property float $quantity
 * @property int $nominal_code_id
 * @property int $tax_rate_id
 * @property float $tax_rate
 * @property bool $is_product
 * @property ?int $bundle_id
 * @property ?int $bundle_quantity_cache
 * @property ?int $bundle_component_quantity_cache
 * @property bool $has_backorder
 * @property bool $has_active_backorder
 * @property bool $no_audit_trail
 * @property bool $is_taxable
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 *
 * Financials
 * @property float $discount_allocation
 * @property float $tax_allocation
 * @property float $tax_allocation_in_tenant_currency
 * @property float $weight_extended
 * @property float $volume_extended
 * @property float $creditedCost
 * @property-read bool $fulfilled
 * @property float $fulfilled_quantity
 * @property-read float $unfulfilled_quantity
 * @property-read bool $is_fully_fulfillable
 * @property-read float $proration_fulfilled
 * @property-read float $amount_in_tenant_currency
 * @property-read int $reserved_quantity
 * @property-read int $unreserved_quantity
 * @property-read int $dropshipped_quantity
 * @property-read Product $product
 * @property-read ProductListing $productListing
 * @property-read BackorderLayer $backorderLayer
 * @property-read Collection $inventoryMovements
 * @property-read Warehouse|null $warehouse
 * @property-read bool $is_dropship
 * @property-read bool $fully_returned
 * @property-read int $returned_quantity
 * @property-read int $unreturned_quantity
 * @property int $canceled_quantity
 * @property int $externally_fulfilled_quantity
 * @property int $processableQuantity
 * @property-read int $fulfillable_quantity
 * @property-read PurchaseOrder|null $purchase_order
 * @property-read SalesOrder $salesOrder
 * @property-read float $subtotal
 * @property-read float $subtotal_cache
 * @property-read float $reporting_subtotal
 * @property-read float $total
 * @property-read float $fulfillment_revenue
 * @property-read float $prorated_revenue
 * @property-read float $prorated_fulfillment_revenue
 * @property-read Collection|BackorderLayer[] $backorderLayers
 * @property-read float $backordered_quantity
 * @property-read float $active_backordered_quantity
 * @property-read Collection|SalesOrderFulfillmentLine[] $salesOrderFulfillmentLines
 * @property-read Collection|SalesCreditLine[] $salesCreditLines
 * @property-read BackorderQueue|null|Model $backorderQueue
 * @property-read BackorderQueue|null|Model $activeBackorderQueue
 * @property-read NominalCode|null $nominalCode
 * @property-read Collection|SalesOrderLineLayer[] $layers
 * @property-read SalesOrderLineFinancial $salesOrderLineFinancial
 * @property-read SalesOrderLineLayer[]|Collection $salesOrderLineLayers
 * @property-read Collection|FinancialAllocatable[] $financialAllocatables
 * @property string $sku
 * @property-write string $nominal_code
 * @property-write string $nominal_code_name
 * @property-write string $warehouse_name
 */
class SalesOrderLine extends Model implements Filterable, FinancialDocumentLineInterface, HasReference, NegativeInventoryEvent, Sortable, UniqueFieldsInterface
{
    use BulkImport,
        Compoships,
        HasAccountingTransactionLine,
        HasCurrencyAttributes,
        HasFactory,
        HasFilters,
        HasSort,
        LogsActivity,
        PowerJoins;

    const INVALID_FINANCIALS_KEY = 'invalid_sales_order_line_financials_ids';

    protected $fillable = [
        'sales_order_id',
        'sales_channel_line_id',
        'warehouse_id',
        'warehouse_name',
        'description',
        'product_id',
        'sku',
        'amount',
        'quantity',
        'fulfilled_quantity',
        'canceled_quantity',
        'tax_allocation',
        'tax_rate_id',
        'tax_rate',
        'discount_allocation',
        'nominal_code_id',
        'nominal_code_name',
        'nominal_code',
        'is_product',
        'has_backorder',
        'product_listing_id',
        'bundle_id',
        'bundle_quantity_cache',
        'bundle_component_quantity_cache',
        'subtotal_cache',
        'canceled_quantity',
        'externally_fulfilled_quantity',
        'warehouse_routing_method',
        'no_audit_trail',
        'is_taxable',
        'updated_at',
        'cogs',
    ];

    /**
     * Casting.
     */
    protected $casts = [
        'amount' => 'float',
        'quantity' => 'float',
        'fulfilled_quantity' => 'float',
        'canceled_quantity' => 'float',
        'externally_fulfilled_quantity' => 'float',
        'subtotal_cache' => 'float',
        'discount_allocation' => 'float',
        'tax_allocation' => 'float',
        'is_product' => 'boolean',
        'is_taxable' => 'boolean',
        'warehouse_routing_method' => WarehouseRoutingMethod::class,
    ];

    private bool $check_backorder = false;

    public static function getUniqueFields(): array
    {
        return [
            'sales_order_id',
            'sales_channel_line_id',
            'product_listing_id',
            'product_id',
            'description',
            'quantity',
            'amount',
            'bundle_id',
            'warehouse_id'
        ];
    }

    public function availableColumns()
    {
        return [
            'sales_order_id',
            'sales_orders.order_date',
            'sales_channel_line_id',
            'product_listing_id',
            'warehouse_id',
            'product_id',
            'quantity',
            'discount_allocation',
            'tax_allocation',
            'revenue_extended',
            'proforma_total_revenue',
            'cogs',
            'unit_cost_extended',
            'total_cost',
            'profit',
            'customer',
        ];
    }

    public function filterableColumns(): array
    {
        $availableColumns = config('data_table.reporting.sales_order_line_financial.columns');
        $availableColumns = collect($availableColumns)->where('filterable', 1)->pluck('data_name')->all();

        return array_merge($availableColumns, $this->availableColumns());
    }

    public function sortableColumns()
    {
        return $this->availableColumns();
    }

    public function generalFilterableColumns(): array
    {
        return ['order_date'];
    }

    protected $attributes = ['amount' => 0, 'quantity' => 1];

    protected $currencyAttributes = ['amount'];

    public function getReference(): string
    {
        return $this->salesOrder->getReference();
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->salesOrder->id;
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [
            'id' => $this->id,
            'sku' => $this->product?->sku,
            'description' => $this->description,
            'warehouse_id' => $this->warehouse_id,
            'warehouse_name' => $this->warehouse?->name ?? 'Not set',
        ];
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at', 'hash'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }


    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesOrder()
    {
        return $this->belongsTo(SalesOrder::class);
    }

    public function salesOrderLineLayers(): HasMany
    {
        return $this->hasMany(SalesOrderLineLayer::class);
    }

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function bundle()
    {
        return $this->belongsTo(Product::class, 'bundle_id', 'id');
    }

    public function productListing()
    {
        return $this->belongsTo(ProductListing::class);
    }

    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function nominalCode()
    {
        return $this->belongsTo(NominalCode::class);
    }

    public function salesOrderFulfillmentLines()
    {
        return $this->hasMany(SalesOrderFulfillmentLine::class);
    }

    public function inventoryMovements()
    {
        return $this->morphMany(InventoryMovement::class, 'link');
    }

    public function salesCreditLines()
    {
        return $this->hasMany(SalesCreditLine::class);
    }

    public function fifoLayers()
    {
        return $this->morphedByMany(FifoLayer::class, 'layer', 'sales_order_line_layers')
            ->using(SalesOrderLineLayer::class)
            ->withPivot('quantity')
            ->withTimestamps();
    }

    public function backorderQueue(): HasOne
    {
        return $this->hasOne(BackorderQueue::class);
    }

    public function inventoryAdjustments(): MorphMany
    {
        return $this->morphMany(InventoryAdjustment::class, 'link');
    }

    public function activeBackorderQueue()
    {
        return $this->hasOne(BackorderQueue::class)->active();
    }

    public function layers()
    {
        return $this->hasMany(SalesOrderLineLayer::class);
    }

    public function brothers()
    {
        return $this->hasMany(self::class, 'sales_order_id', 'sales_order_id');
    }

    public function splitsInto()
    {
        return $this->hasMany(self::class, 'split_from_line_id', 'id');
    }

    public function amazonFulfilledShipments()
    {
        return $this->hasMany(FulfilledShipment::class, 'amazon-order-item-id', 'sales_channel_line_id');
    }

    public function productsInventory()
    {
        return $this->hasMany(ProductInventory::class, 'product_id', 'product_id');
    }

    public function fulfilledLines()
    {
        return $this->salesOrderFulfillmentLines()->whereRelation('salesOrderFulfillment', 'status', SalesOrderFulfillment::STATUS_FULFILLED);
    }

    public function adjustments(): MorphMany
    {
        return $this->morphMany(InventoryAdjustment::class, 'link');
    }

    public function taxRate()
    {
        return $this->belongsTo(TaxRate::class);
    }

    public function accountingTransactionLine()
    {
        return $this->morphOne(AccountingTransactionLine::class, 'link');
    }

    public function reportingDailyFinancials(): HasMany
    {
        return $this->hasMany(ReportingDailyFinancial::class, 'product_id', 'id');
    }

    public function salesOrderLineFinancial(): HasOne
    {
        return $this->hasOne(SalesOrderLineFinancial::class);
    }

    public function financialLinesRevenueAllocatedFrom(): BelongsToMany
    {
        return $this->belongsToMany(self::class, 'financial_allocatables', 'allocatable_to_id', 'allocatable_from_id')
            ->wherePivot('allocatable_to_type', self::class)
            ->wherePivot('allocatable_from_type', FinancialLine::class);
    }

    public function financialAllocatables(): BelongsToMany
    {
        return $this->belongsToMany(
            FinancialLine::class, // The related model
            'financial_allocatables', // The pivot table
            'allocatable_to_id', // Foreign key on the pivot table that is related to the current model
            'allocatable_from_id', // Foreign key on the pivot table that is related to the related model
            'id', // Local key on the current model
            'id' // Local key on the related model
        )
            ->wherePivot('allocatable_to_type', self::class)
            ->wherePivot('allocatable_from_type', FinancialLine::class)
            ->withPivot([
                'allocatable_from_type',
                'allocatable_to_type',
                'amount'
            ]);
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function amountInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->amount * $this->salesOrder->currency_rate);
    }

    public function taxAllocationInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->tax_allocation * $this->salesOrder->currency_rate);
    }

    public function getFulfilledAttribute()
    {
        if (! $this->is_product) {
            return true;
        }

        return ($this->fulfilled_quantity + $this->externally_fulfilled_quantity) == $this->quantity;
    }

    public function getUnfulfilledQuantityAttribute()
    {
        if (! $this->is_product) {
            return 0;
        }

        return $this->quantity - $this->fulfilled_quantity;
    }

    public function getReportingSubtotalAttribute(): float
    {
        return $this->quantity * $this->amount;
    }

    public function getSubtotalAttribute(): float
    {
        return $this->quantity * $this->amount;
    }

    /*
    |--------------------------------------------------------------------------
    | Financial Proforma Accessors
    |--------------------------------------------------------------------------
    */

    public function getWeightExtendedAttribute(): float
    {
        return $this->quantity * $this->product?->weight ?? 0.00;
    }

    public function getVolumeExtendedAttribute(): float
    {
        $volume = $this->product?->length * $this->product?->width * $this->product?->height ?? 0.00;

        return $this->quantity * $volume;
    }

    /*
    |--------------------------------------------------------------------------
    | Other Accessors
    |--------------------------------------------------------------------------
    */

    public function getDiscountValueAttribute()
    {
        return $this->quantity * $this->amount * $this->discount_allocation / 100;
    }

    public function getTotalAttribute()
    {
        return $this->subtotal;
    }

    public function getFulfillmentRevenueAttribute()
    {
        return $this->amount * $this->salesOrderFulfillmentLines()->sum('quantity');
    }

    public function getProratedFulfillmentRevenueAttribute()
    {
        if ((float) $this->salesOrder->fulfillment_revenue == 0) {
            return 0;
        }

        return $this->fulfillment_revenue / $this->salesOrder->fulfillment_revenue;
    }

    public function getProratedRevenueAttribute()
    {
        $this->salesOrder->loadMissing('salesOrderLines');

        if ((float) $this->salesOrder->product_subtotal == 0) {
            return 0;
        }

        return $this->subtotal / $this->salesOrder->product_subtotal;
    }

    public function getShippingCostAttribute()
    {
        return $this->salesOrderFulfillmentLines->sum('shipping_cost');
    }

    public function setDescriptionAttribute($value)
    {
        if (empty($this->sales_channel_line_id)) {
            $this->sales_channel_line_id = $value;
        }

        $this->attributes['description'] = $value;
    }

    public function setSkuAttribute($value)
    {
        if (empty($value)) {
            return;
        }

        if (empty($this->sales_channel_line_id) || $this->sales_channel_line_id == $this->description) {
            $this->sales_channel_line_id = $value;
        }

        $this->product_id = Product::with([])->where('sku', $value)->value('id');
    }

    public function setNominalCodeAttribute($value)
    {
        $this->nominal_code_id = empty($value) ? null : NominalCode::with([])->where('code', $value)->value('id');
    }

    public function setNominalCodeNameAttribute($value)
    {
        $this->nominal_code_id = empty($value) ? null : NominalCode::with([])->where('name', $value)->value('id');
    }

    public function setWarehouseNameAttribute($value)
    {
        $this->warehouse_id = empty($value) ? null : Warehouse::with([])->where('name', $value)->value('id');
    }

    public function getIsDropshipAttribute()
    {
        return (bool) ($this->warehouse->supplier_id ?? null);
    }

    public function getFullyReturnedAttribute()
    {
        return $this->fulfilled_quantity == $this->returned_quantity;
    }

    public function getReturnedQuantityAttribute()
    {
        return $this->salesCreditLines->sum('quantity');
    }

    public function creditedCost(): Attribute
    {
        return Attribute::get(
            fn () => $this->salesCreditLines->sum('unitCostExtended')
        );
    }

    public function getUnreturnedQuantityAttribute()
    {
        if (! $this->is_product) {
            return 0;
        }

        return $this->fulfilled_quantity - $this->returned_quantity;
    }

    public function getPurchaseOrderAttribute()
    {
        if ($this->is_dropship) {
            return PurchaseOrder::with('purchaseOrderLines')
                ->where([
                    'sales_order_id' => $this->sales_order_id,
                    'supplier_warehouse_id' => $this->warehouse_id,
                ])
                ->first();
        }

        return null;
    }

    public function getActiveBackorderedQuantityAttribute()
    {
        return $this->activeBackorderQueue ? $this->activeBackorderQueue->unreleased_quantity : 0;
    }

    public function getBackorderedQuantityAttribute()
    {
        return $this->backorderQueue->unreleased_quantity ?? 0;
    }

    public function getFulfillableQuantityAttribute()
    {
        if ($this->unfulfilled_quantity == 0) {
            return 0;
        }

        return max($this->unfulfilled_quantity - $this->backordered_quantity, 0);
    }

    public function hasActiveBackorder(): Attribute{
        return Attribute::get(fn() => $this->backorderQueue?->shortage_quantity > 0);
    }

    public function setTaxAttribute(float $value)
    {
        $this->setTaxTotalAttribute($value);
    }

    public function getIsFullyFulfillableAttribute(): bool
    {
        if ($this->fulfilled) {
            return false;
        }

        customlog('SKU-6135', 'Checking if sales order line '.$this->id.' for product '.$this->product?->sku.' for order '.$this->salesOrder->sales_order_number.' is fully fulfillable: '.($this->fulfilled_quantity > 0).'.', [
            'backordered_quantity' => $this->backordered_quantity,
            'available_quantity' => $this->product?->availableQuantity?->quantity,
            'fulfillable_quantity' => $this->fulfillable_quantity,
            'quantity' => $this->quantity,
        ]);

        return ($this->fulfillable_quantity > 0) && ($this->backordered_quantity == 0);
    }

    public function setBackorderStatus(bool $hasBackorder)
    {
        $this->has_backorder = $hasBackorder;
        $this->check_backorder = true;
    }

    public function getProrationFulfilledAttribute()
    {
        if ($this->subtotal == 0) {
            return 0;
        }

        return $this->fulfilled_quantity * $this->amount / $this->subtotal;
    }

    public function isWarehousedProduct(): bool
    {
        return ! is_null($this->product_id) && ! is_null($this->warehouse_id);
    }

    public function getSalesNominalCodeId(): ?int
    {
        return $this->salesOrder->salesChannel->integrationInstance->integration_settings['sales_nominal_code_id'] ?? null;
    }

    // TODO: Needs updating
    public function getLandedCost(): float
    {
        if (! $this->is_product) {
            return 0;
        }

        if ($this->proforma_landed_cost) {
            return $this->proforma_landed_cost;
        }

        if ($this->product?->proforma_landed_cost_percentage) {
            return $this->subtotal * $this->product->proforma_landed_cost_percentage / 100;
        }

        return 0;
    }

    // TODO: Needs updating
    public function getShippingCost()
    {
        if (! $this->is_product) {
            return 0;
        }
        // TODO: this will be correct after merging SKU-4936 branch
        $fulfillmentShippingCost = $this->salesOrderFulfillmentLines->sum('shipping_cost');

        if ($fulfillmentShippingCost > 0) {
            return $fulfillmentShippingCost;
        }

        if ($this->proforma_shipping_cost) {
            return $this->proforma_shipping_cost;
        }

        return 0;
    }

    // Return nominal code id based on specific to general logic for settings
    public function getNominalCodeId()
    {
        if ($this->product_id && $this->product->sales_nominal_code_id) {
            return $this->product->sales_nominal_code_id;
        }

        $salesNominalCodeId = $this->getSalesNominalCodeId();
        if ($salesNominalCodeId) {
            return $salesNominalCodeId;
        }

        if (! $this->product_id && $this->description == 'Shipping') {
            return Helpers::setting(Setting::KEY_NC_MAPPING_SHIPPING_SALES_ORDERS);
        }

        return Helpers::setting(Setting::KEY_NC_MAPPING_SALES_ORDERS);
    }

    /**
     * @throws Exception
     */
    public function save(array $options = []): bool
    {
        if ($this->isDirty('warehouse_id') && $this->getOriginal('warehouse_id') != null) {
            customlog('SKU-6240', 'Warehouse changing for sales order line '.$this->id.' (SKU: '.$this->product?->sku.') for sales order '.$this->salesOrder->sales_order_number, [
                'warehouse_id_from' => $this->getOriginal('warehouse_id'),
                'warehouse_id_to' => $this->warehouse_id,
                'debug' => debug_pretty_string(),
                'payload' => request()?->all(),
            ]);
        }
        if (empty($this->is_product)) {
            $this->is_product = (bool) $this->product_id;
        }

        if ($this->is_product === false) {
            $this->warehouse_id = null;
        }

        if (empty($this->nominal_code_id)) {
            $this->nominal_code_id = $this->getNominalCodeId();
        }

        // Set has backorder layer field
        if (! $this->check_backorder) {
            if ($this->backorderQueue && $this->backorderQueue->backordered_quantity > $this->backorderQueue->released_quantity) {
                $this->has_backorder = true;
            } else {
                $this->has_backorder = false;
            }
        }

        $this->subtotal_cache = $this->subtotal;

        if ($this->quantity < 0) {
            throw new Exception('Quantity must be non-negative.');
        }

        return parent::save($options);
    }

    /**
     * @throws Exception
     */
    public function delete(): ?bool
    {
        $this->resetMovements();
        // Update products inventory
        if ($this->product_id) {
            dispatch(new UpdateProductsInventoryAndAvgCost([$this->product_id]));
        }

        if ($this->salesOrderLineFinancial) {
            $this->salesOrderLineFinancial->delete();
        }
        $this->salesOrderFulfillmentLines()->delete();
        $this->salesCreditLines()->each(function (SalesCreditLine $creditLine) {
            $creditLine->delete();
        });

        // Remove any sales order line layers
        SalesOrderLineLayer::with([])
            ->where('sales_order_line_id', $this->id)->delete();

        foreach ($this->splitsInto as $splitsIntoLine) {
            $splitsIntoLine->delete();
        }

        return parent::delete();
    }

    /**
     * @throws Throwable
     */
    public function updateCanceledQuantity(int $newTotalCanceledQty, int $maxCancelable, int $existingCanceledQuantity = 0): void
    {
        $canceledQuantity = min($newTotalCanceledQty, $maxCancelable);
        $updatedQuantity = max(0, $this->quantity - $canceledQuantity);
        $this->canceled_quantity = $canceledQuantity;
        customlog('cancelQty', $this->product?->sku.' for '.$this->salesOrder->sales_order_number.' qty: '.$this->quantity.' canceled: '.$canceledQuantity.' updated: '.$updatedQuantity);
        $this->quantity = $updatedQuantity;
        $this->save();
        $this->refresh();
        customlog('cancelQty', 'after save qty: '.$this->quantity.' canceled: '.$this->canceled_quantity);
        $qtyNewlyCanceled = max($newTotalCanceledQty - $existingCanceledQuantity, 0);
        if ($this->product && $this->warehouse && $qtyNewlyCanceled > 0) {
            DB::transaction(function () use ($qtyNewlyCanceled) {
                InventoryManager::with($this->warehouse_id, $this->product)
                    ->decreaseNegativeEventQty($qtyNewlyCanceled, $this);
            });
            // We re-calculate backorder queue coverages.
            dispatch(new SyncBackorderQueueCoveragesJob(null, null, [$this->product_id], $this->warehouse_id));
        }
    }

    /**
     * Reset inventory movements and backorders.
     *
     * @throws Exception
     */
    public function resetMovements()
    {
        if (! $this->isWarehousedProduct()) {
            return;
        }
        InventoryManager::with($this->warehouse_id, $this->product)
            ->reverseNegativeEvent($this);
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeWhereDetails(Builder $builder, $lineDetails)
    {
        if (! empty($lineDetails['sales_channel_line_id'])) {
            return $builder->where('sales_channel_line_id', $lineDetails['sales_channel_line_id']);
        } elseif (! empty($lineDetails['sku'])) {
            return $builder->where('sales_channel_line_id', $lineDetails['sku']);
        } elseif (! empty($lineDetails['description'])) {
            return $builder->where('sales_channel_line_id', $lineDetails['description']);
        }
    }

    public function scopeTrailingDays(Builder $query, int $trailing_days): Builder
    {
        return $query->where('sales_orders.order_date', '>=', Carbon::now()->subDays($trailing_days));
    }

    public function scopeStartDate(Builder $query, $date): Builder
    {
        return $query->where('sales_orders.order_date', '>=', Helpers::dateLocalToUtc($date));
    }

    public function scopeEndDate(Builder $query, $date): Builder
    {
        return $query->where('sales_orders.order_date', '<', Helpers::dateLocalToUtc(CarbonImmutable::parse($date)->addDay()));
    }

    public function scopeNonFba(Builder $query): Builder
    {
        return $query->whereHas('warehouse', function ($query) {
            $query->where('type', '!=', Warehouse::TYPE_AMAZON_FBA);
        });
    }

    public function toShipStationOrderItem()
    {
        $orderItem = new OrderItem();
        $orderItem->lineItemKey = $this->id;
        $orderItem->sku = $this->product ? $this->product->sku : null;
        $orderItem->upc = $this->product ? $this->product->barcode : null;
        $orderItem->name = $this->description;
        $orderItem->imageUrl = $this->product ? $this->product->getExternalImageUrl() : null;
        $orderItem->weight = $this->product ? new Weight([
            'value' => $this->product->weight,
            'units' => $this->mapWeightUnits($this->product->weight_unit),
        ]) : null;
        $orderItem->quantity = $this->quantity;
        $orderItem->unitPrice = $this->amount;
        $orderItem->taxAmount = $this->tax_allocation;
        $orderItem->shippingAmount = 0;
        $orderItem->adjustment = ! $this->is_product;

        if ($this->product) {
            $shipstation = IntegrationInstance::shipstation()->first();
            $orderItem->fulfillmentSku = $this->getFulfillmentSku($shipstation->integration_settings['settings']['productFieldToPass']['name'] ?? null);
        }

        return $orderItem;
    }

    public function toStarshipitOrderItem()
    {
        $orderItem = new \App\SDKs\Starshipit\Model\OrderItem();
        $orderItem->description = $this->description;
        $orderItem->sku = $this->product ? $this->product->sku : null;
        $orderItem->quantity = $this->quantity;
        $orderItem->weight = $this->product ? Helpers::weightConverter($this->product->weight, $this->product->weight_unit, Product::WEIGHT_UNIT_KG) : null;
        $orderItem->value = $this->amount;

        if ($this->product) {
            $shipstation = IntegrationInstance::starshipit()->first();
            $orderItem->tariff_code = $this->getFulfillmentSku($shipstation->integration_settings['settings']['productFieldToPass']['name'] ?? null);
        }

        return $orderItem;
    }

    private function getFulfillmentSku($productFieldToPass)
    {
        if (! $productFieldToPass) {
            return null;
        }

        if (strtolower($productFieldToPass) == 'mpn') {
            return $this->product->mpn;
        } elseif (strtolower($productFieldToPass) == strtolower('Default Supplier SKU')) {
            return $this->product->supplierProducts()->where('is_default', true)->value('supplier_sku') ?? $this->product->supplierProducts()->value('supplier_sku');
        } else {
            $attribute = $this->product->productAttributes()->where('name', $productFieldToPass)->first();
            if ($attribute) {
                return $attribute->pivot->value;
            }
        }
    }

    public function getListing()
    {
        if ($this->is_product && $this->product_id == null) {
            $integration = $this->salesOrder->salesChannel->integrationInstance->integration;

            if (strtolower($integration->name) == strtolower(Integration::NAME_SKU_IO)) {
                return null;
            }

            if (! in_array($integration->name, [
                Integration::NAME_AMAZON_US,
                Integration::NAME_SHOPIFY,
                Integration::NAME_WOOCOMMERCE,
                Integration::NAME_EBAY,
                Integration::NAME_MAGENTO,
            ])) {
                return null;
            }
            /** @var Model $product */
            $product = $integration->getProductsModelPath();
            $order = $this->salesOrder->order_document;

            if ($order)
            {
                if ($order instanceof AbstractSalesChannelOrder)
                {
                    $itemClass = $order::getItemClassName();
                    $item = $order->orderItems->where(app($itemClass)->getUniqueId(), $this->sales_channel_line_id)->first();
                }
                else {
                    $item = collect(Arr::get($order,
                        $order::LINE_ITEMS_QUERY))->firstWhere($order::LINE_ITEMS_QUERY_ID,
                        $this->sales_channel_line_id);
                }

                if ($item)
                {
                    $product = ($item instanceof AbstractSalesChannelOrderLine) ?
                        $item->product :
                        $product::query()
                            ->where('integration_instance_id', $this->salesOrder->salesChannel->integrationInstance->id)
                            ->where($order::PRODUCT_LOCAL_QUERY_IDENTIFER, $item[$order::PRODUCT_LOCAL_FORIEN_IDENTIFER])
                            ->first();

                    if ($product)
                    {
                        $item['product'] = $product;
                    }

                    switch ($integration->name) {
                        case Integration::NAME_AMAZON_US:
                            return SalesOrderLineListingAmazonResource::make($item);
                        case Integration::NAME_SHOPIFY:
                            return SalesOrderLineListingShopifyResource::make($item);
                        case Integration::NAME_MAGENTO:
                            return SalesOrderLineListingMagentoResource::make($item);
                        case Integration::NAME_WOOCOMMERCE:
                            return SalesOrderLineListingWooCommerceResource::make($item);
                        case Integration::NAME_EBAY:
                            return SalesOrderLineListingEbayResource::make($item);
                    }
                }
            }
        }
    }

    private function mapWeightUnits($unit)
    {
        switch ($unit) {
            case Product::WEIGHT_UNIT_LB:
                return Weight::UNIT_POUNDS;
            case Product::WEIGHT_UNIT_OZ:
                return Weight::UNIT_OUNCES;
            case Product::WEIGHT_UNIT_KG:
                return Weight::UNIT_GRAMS;
        }
    }

    /**
     * {@inheritDoc}
     */
    public function getTransactionCurrency(): Currency
    {
        return $this->salesOrder->currency;
    }

    /**
     * Creates inventory movements for the sales order line
     * which results in negative inventory.
     */
    public function createInventoryMovementsForLayers(array $layers, ?Carbon $dateOverride = null): void
    {
        foreach ($layers as $layerInfo) {
            /** @var InventoryMovement $inventoryMovement */
            $inventoryMovement = $this->inventoryMovements()
                ->where('layer_type', $layerInfo['layer_type'])
                ->where('layer_id', $layerInfo['layer']->id)
                ->where('quantity', '<', 0)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->first();

            $quantity = $layerInfo['quantity'];

            if (! $inventoryMovement) {
                $inventoryMovement = new InventoryMovement();
                $inventoryMovement->inventory_movement_date = $dateOverride ?? $this->getEventDate();
                $inventoryMovement->product_id = $this->product_id;
                $inventoryMovement->warehouse_id = $this->warehouse_id;
                $inventoryMovement->warehouse_location_id = $this->warehouse->defaultLocation->id ?? null;
                $inventoryMovement->reference = $this->salesOrder->sales_order_number; // THIS IS IMPORTANT, IT'S USED TO MATCH FULFILLMENT MOVEMENTS.
            } else {
                $quantity += abs($inventoryMovement->quantity);
            }

            $inventoryMovement->quantity = -$quantity;

            if ($layerInfo['layer'] instanceof FifoLayer) {
                $inventoryMovement->fifo_layer = $layerInfo['layer']->id;
                $inventoryMovement->save();
                // Associate layer in sales order line layers
                $this->fifoLayers()->attach($layerInfo['layer']->id, ['quantity' => $quantity]);
            } else {
                $inventoryMovement->backorder_queue = $layerInfo['layer']->id;
            }

            // save movement with inventory status Active
            $inventoryMovement->type = InventoryMovement::TYPE_SALE;
            $inventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
            $this->inventoryMovements()->save($inventoryMovement);

            // save another movement with inventory status Reserved
            if ($inventoryMovement->wasRecentlyCreated) {
                $reservedInventoryMovement = $inventoryMovement->replicate();
            } else {
                // Fetch prior reservation
                $reservedInventoryMovement = $this->inventoryMovements()
                    ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                    ->where('layer_type', $layerInfo['layer_type'])
                    ->where('layer_id', $layerInfo['layer']->id)
                    ->where('quantity', '>', 0)
                    ->firstOrFail();
            }
            $reservedInventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_RESERVED;
            $reservedInventoryMovement->quantity = $quantity;
            $this->inventoryMovements()->save($reservedInventoryMovement);
        }
    }

    public function getReductionActiveMovements(): Arrayable
    {
        $movements = $this->inventoryMovements()->with('layer')
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '<', 0)
            ->orderBy('layer_type') // Prioritize backorders first
            ->get();

        $fulfillmentLines = $this->salesOrderFulfillmentLines()->with(['inventoryMovements', 'salesOrderFulfillment'])->get();

        // sort movements by fulfillments
        return $movements->each(function (InventoryMovement $movement) use ($fulfillmentLines) {
            // backorders first
            if ($movement->layer instanceof BackorderQueue) {
                $movement->priority = 0;
            } else {
                // fulfillment lines that used the same fifo layer
                $fulfillmentLinesUsedFifo = $fulfillmentLines->filter(fn (SalesOrderFulfillmentLine $fulfillmentLine) => $fulfillmentLine->inventoryMovements->where('layer_id', $movement->layer_id)->isNotEmpty());
                if ($fulfillmentLinesUsedFifo->isEmpty()) {
                    // then movements without fulfillment lines
                    $movement->priority = 1;
                } elseif ($fulfillmentLinesUsedFifo->where('salesOrderFulfillment.status', SalesOrderFulfillment::STATUS_FULFILLED)->isEmpty()) {
                    // then movements that have fulfillment lines but the fulfillment status is not fulfilled
                    $movement->priority = 2;
                } else {
                    // finally, the movements that have fulfillment lines and the fulfillment status is fulfilled
                    $movement->priority = 3;
                }
            }
        })->sortBy('priority')->each(function ($m) {
            unset($m->priority);
        })->values();
    }

    public function clearMovements(): void
    {
        $this->inventoryMovements()->delete();
    }

    /**
     * @throws OpenStockTakeException
     */
    public function reduceQtyWithSiblings(int $quantity, InventoryMovement $representingMovement): void
    {
        // Update positive reserved quantity
        /** @var InventoryMovement $reservationMovement */
        $reservationMovement = $this->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('quantity', abs($representingMovement->quantity))
            ->where('layer_type', $representingMovement->layer_type)
            ->where('layer_id', $representingMovement->layer_id)
            ->first();
        $representingMovement->quantity += $quantity; // Note that movement->qty is negative

        if ($representingMovement->quantity >= 0) {
            $representingMovement->delete();
            $this->layers()->where('layer_id', $representingMovement->layer_id)
                ->where('layer_type', $representingMovement->layer_type)
                ->delete();
        } else {
            $representingMovement->save();
            $this->layers()->where('layer_id', $representingMovement->layer_id)
                ->where('layer_type', $representingMovement->layer_type)
                ->update(['quantity' => abs($representingMovement->quantity)]);
        }
        if ($reservationMovement) {
            $reservationMovement->quantity -= abs($quantity);
            if ($reservationMovement->quantity <= 0) {
                $reservationMovement->delete();
                $this->clearFulfillmentLines($representingMovement);
            } else {
                $reservationMovement->save();
                // Next, we reduce fulfillment lines if applicable.
                // This handles the edge case where the sales order line
                // quantity is reduced even though there are active fulfillments.
                if ($this->unfulfilled_quantity < 0) {
                    // SKU-5752, delete fulfillment even if there are remaining lines.
                    //$this->reduceFulfillmentQuantityBy(abs($this->unfulfilled_quantity));
                    $this->clearFulfillmentLines($representingMovement);
                }
            }
        } else {
            // Delete any orphan fulfillment line.
            $this->clearFulfillmentLines($representingMovement);
        }
    }

    private function reduceFulfillmentQuantityBy(int $quantity): void
    {
        $fulfillmentLines = $this->salesOrderFulfillmentLines()->orderBy('created_at', 'DESC')->get();
        $total = 0;
        /** @var SalesOrderFulfillmentLine $fulfillmentLine */
        foreach ($fulfillmentLines as $fulfillmentLine) {
            if ($total >= $quantity) {
                break;
            }
            $remaining = $quantity - $total;
            if ($fulfillmentLine->quantity > $remaining) {
                $fulfillmentLine->quantity -= $remaining;
                $fulfillmentLine->save();
                $fulfillmentLine->inventoryMovements()->update(['quantity' => -$fulfillmentLine->quantity]);
                break;
            } else {
                $total += $fulfillmentLine->quantity;
                // The fulfillment line's quantity is less than what we need to reverse,
                // we delete the fulfillment line and attempt other fulfillment lines.
                $fulfillmentLine->deleteWithFulfillmentIfLast(true);
            }
        }
    }

    private function clearFulfillmentLines(InventoryMovement $reservationMovement): void
    {
        $this->salesOrderFulfillmentLines()
            ->whereHas('inventoryMovements', function (Builder $builder) use ($reservationMovement) {
                $builder->where('layer_type', $reservationMovement->layer_type);
                $builder->where('layer_id', $reservationMovement->layer_id);
            })
            ->each(function (SalesOrderFulfillmentLine $fulfillmentLine) {
                //$fulfillmentLine->deleteWithFulfillmentIfLast(true);
                // SKU-5752, delete fulfillment even if there are remaining lines.
                $fulfillmentLine->deleteWithFulfillment(true);
            });
    }

    public function getLinkType(): string
    {
        return self::class;
    }

    public function getId(): int
    {
        return $this->id;
    }

    /**
     * @throws Exception
     */
    public function splitExternallyFulfilled(): self|bool
    {
        if ($this->quantity - $this->canceled_quantity <= $this->externally_fulfilled_quantity) {
            return false;
        }
        // We split externally fulfilled quantity
        // into its own line.
        // For instance, if 2 of 5 is externally fulfilled,
        // we move those 2 units to a new line that's fully
        // externally fulfilled.
        $split = $this->replicate();
        $split->split_from_line_id = $this->id;
        $split->quantity = $this->externally_fulfilled_quantity;
        $split->canceled_quantity = 0;
        $split->warehouse_id = null;
        $split->save();
        $this->update([
            'quantity' => max($this->quantity - $split->quantity, 0),
            'externally_fulfilled_quantity' => 0,
        ]);

        return $split;
    }

    /**
     * @throws Exception
     */
    public function splitSalesOrderLineByWarehouses($warehousesWithInventory): self|bool
    {
        $notComplete = $this->processableQuantity;
        $warehouses = collect($warehousesWithInventory);

        foreach ($warehouses as $key => $warehouse) {
            $quantityToAllocate = min($notComplete, $warehouse['quantity']);
            $notComplete -= $quantityToAllocate;

            if ($key === $warehouses->count() - 1 && $notComplete > 0) {
                $quantityToAllocate += $notComplete;
            }

            if ($key === 0) {
                $this->quantity = $quantityToAllocate;
                $this->warehouse_id = $warehouse['warehouse_id'];
                $this->save();

                continue;
            }

            $split = $this->replicate();
            $split->quantity = $quantityToAllocate;
            $split->warehouse_id = $warehouse['warehouse_id'];
            $split->split_from_line_id = $this->id;
            $split->save();

            if ($notComplete <= 0) {
                break;
            }
        }

        return true;
    }

    public function replicate(?array $except = null): self
    {
        return parent::replicate(
            except: [
                'is_proforma_financials_cache_valid',
                'is_proforma_cost_cache_valid',
            ]
        );
    }

    public function removeWarehouse(): self
    {
        $this->warehouse_id = null;
        $this->save();

        return $this;
    }

    public function processableQuantity(): Attribute
    {
        return Attribute::get(fn () => max(0, $this->quantity - $this->externally_fulfilled_quantity));
    }

    public function reservedQuantity(): Attribute
    {
        return Attribute::get(fn () => $this->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('quantity', '>', 0)
            ->sum('quantity'));
    }

    public function unreservedQuantity(): Attribute
    {
        return Attribute::get(fn () => max(0, $this->quantity - $this->reserved_quantity));
    }

    public function dropshippedQuantity(): Attribute
    {
        $dropshippedQuantity = ! $this->salesOrder?->purchaseOrders ? 0 :
            $this->salesOrder->purchaseOrders->sum(
                fn (PurchaseOrder $purchaseOrder) => $purchaseOrder->purchaseOrderLines->sum(
                    fn (PurchaseOrderLine $purchaseOrderLine) => $purchaseOrderLine->product_id === $this->product_id ?
                        $purchaseOrderLine->quantity : 0
                )
            );

        return Attribute::get(fn () => $dropshippedQuantity);
    }

    /**
     * @throws Exception
     */
    public function setWarehouseId(?int $warehouseId): self
    {
        if ($warehouseId) {
            $this->warehouse_id = $warehouseId;
            $this->save();
        }

        return $this;
    }

    public function updateTaxAllocation()
    {
        $this->tax_allocation = $this->getTaxAmount($this->salesOrder->is_tax_included, $this->subtotal,
            $this->taxRate?->rate ?? 0);
        $this->update();
    }

    public function getTaxAmount($is_tax_included, $total, $tax_rate)
    {
        return $is_tax_included ? $total - ($total / (1 + ($tax_rate / 100))) : ($total * ($tax_rate / 100));
    }

    /**
     * @throws Exception
     */
    public function incrementFulfilledQuantity(float $quantity): self
    {
        $this->fulfilled_quantity = min($this->quantity, abs($quantity) + ($this->fulfilled_quantity ?? 0));
        $this->save();

        return $this;
    }

    public function reduceExternallyFulfilledQuantity(int $quantity): self
    {
        $this->update([
            'externally_fulfilled_quantity' => max(
                0,
                $this->externally_fulfilled_quantity - abs($quantity)
            ),
            'quantity' => $this->quantity + abs($quantity),
        ]);

        return $this;
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function refreshFromFifoCostUpdate(float $amount = 0): void
    {
        DB::transaction(function () use ($amount) {
            $this->setCogsIfFinancialLineExists($amount);
            $this->salesCreditLines->each(function (SalesCreditLine $salesCreditLine) use ($amount) {
                $salesCreditLine->unit_cost = $amount;
                $salesCreditLine->save();
                $salesCreditReturnFifoLayerIdsForRecalculation = [];
                $salesCreditLine->salesCreditReturnLines->each(function (SalesCreditReturnLine $salesCreditReturnLine) use (&$salesCreditReturnFifoLayerIdsForRecalculation) {
                    $salesCreditReturnLine->salesCreditReturn->updated_at = now(); // To trigger accounting transaction update
                    $salesCreditReturnLine->salesCreditReturn->save();
                    $salesCreditReturnFifoLayerIdsForRecalculation[] = $salesCreditReturnLine->getOriginatingMovement()->fifo_layer->id;
                });
                if (!empty($salesCreditReturnFifoLayerIdsForRecalculation)) {
                    app(FifoLayerRepository::class)->recalculateTotalCosts($salesCreditReturnFifoLayerIdsForRecalculation);
                }
            });
        });
    }

    public function getProductId(): int
    {
        return $this->product_id;
    }

    public function getWarehouseId(): int
    {
        return $this->warehouse_id;
    }

    public function getEventDate(): Carbon
    {
        return $this->salesOrder->order_date->max(Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE), 'UTC'));
    }

    public function getQuantity(): int
    {
        return $this->quantity;
    }

    public function getType(): string
    {
        return InventoryMovement::TYPE_SALE;
    }

    public function setCogsIfFinancialLineExists(float $amount): void
    {
        if ($this->salesOrderLineFinancial) {
            $this->salesOrderLineFinancial->cogs = $amount;
            $this->salesOrderLineFinancial->save();
            $this->touch();
            // Updating COGS means usages, which derive from COGS, also need updating.
            // So we touch the timestamp here to trigger an update.

            // getTransactionsNeedingUpdate checks sales order fulfillment timestamp, but not line, so we update the fulfillment here
            $this->salesOrder->salesOrderFulfillments->each->touch();
        }
    }

    /**
     * Get the total credited quantity (in sales credit)
     *
     * @return int
     */
    public function getCreditedQuantityAttribute(): int
    {
        return $this->salesCreditLines()->sum('quantity');
    }

    public function getActiveInventoryMovement(?FifoLayer $fifoLayer = null): ?InventoryMovement
    {
        $query = $this->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '<', 0);

        if ($fifoLayer) {
            $query->where('layer_type', FifoLayer::class)
                ->where('layer_id', $fifoLayer->id);
        }
        try {
            $movement = $query->sole();
        } catch (ModelNotFoundException) {
            return null;
        }

        return $movement;
    }

    public function getReservedInventoryMovement(?FifoLayer $fifoLayer = null): ?InventoryMovement
    {
        $query = $this->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('quantity', '>', 0);

        if ($fifoLayer) {
            $query->where('layer_type', FifoLayer::class)
                ->where('layer_id', $fifoLayer->id);
        }
        try {
            $movement = $query->sole();
        } catch (ModelNotFoundException) {
            return null;
        }

        return $movement;
    }
}
