<?php

namespace App\Models;

use App\Abstractions\FinancialDocumentInterface;
use App\Abstractions\UniqueFieldsInterface;
use App\Data\AccountingTransactionData;
use App\Exporters\BaseExporter;
use App\Exporters\MapsExportableFields;
use App\Exporters\TransformsExportData;
use App\Helpers;
use App\Importers\DataImporters\SalesOrderFulfillmentDataImporter;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\Amazon\FulfilledShipment;
use App\Models\Amazon\SubmittedFulfillment;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasAccountingTransaction;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Concerns\LogsActivity;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\ShipStation\ShipstationOrder;
use App\Models\ShipStation\ShipstationWarehouse;
use App\Models\Starshipit\StarshipitOrder;
use App\Repositories\Shopify\ShopifyOrderMappingRepository;
use App\SDKs\ShipStation\Model\AdvancedOptions;
use App\SDKs\ShipStation\Model\Dimensions;
use App\SDKs\Starshipit\Model\OrderPackage;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionDataFromSalesOrderFulfillment;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionLineDataFromSalesOrderFulfillmentLine;
use App\Services\SalesOrder\FulfillSalesOrderService;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute as Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Entities\AmazonFulfillmentOrder;
use Modules\Veracore\Entities\VeracoreOrder;
use Modules\Veracore\Services\VeracoreManager;
use Spatie\Activitylog\LogOptions;
use Throwable;

/**
 * Class SalesOrderFulfillment.
 *
 *
 * @property int $id
 * @property int $sales_order_id
 * @property int $fulfillment_sequence
 * @property int $warehouse_id
 * @property int|null $requested_shipping_method_id
 * @property string|null $requested_shipping_method
 * @property int|null $fulfilled_shipping_method_id
 * @property string|null $fulfilled_shipping_method
 * @property float|null $cost
 * @property string $status
 * @property string $fulfillment_type
 * @property string|null $tracking_number
 * @property Carbon|null $fulfilled_at
 * @property Carbon|null $packing_slip_printed_at
 * @property Carbon|null $submitted_to_sales_channel_at
 * @property int|null $user_id
 * @property array $metadata
 * @property string|null $sales_channel_link_type
 * @property int|null $sales_channel_link_id
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read int $totalQuantity
 * @property-read Collection|SalesOrderFulfillmentLine[] $salesOrderFulfillmentLines
 * @property-read Collection|SalesOrderFulfillmentLine[] $nonDropshipFulfillmentLines
 * @property-read Warehouse $warehouse
 * @property-read ShippingMethod $requestedShippingMethod
 * @property-read ShippingMethod $fulfilledShippingMethod
 * @property-read SalesOrder $salesOrder
 * @property-read User|null $user
 * @property-read string $fba_shipping_method
 * @property-read string $fulfillment_number
 * @property-read string|null $tracking_link
 * @property-read StarshipitOrder $starshipitOrder
 * @property-read ShipstationOrder $shipstationOrder
 * @property-read Model $salesChannelLink
 * @property-read AmazonFulfillmentOrder $amazonFulfillmentOrder
 * @property bool $is_manual
 */
class SalesOrderFulfillment extends Model implements Filterable, FinancialDocumentInterface, MapsExportableFields, Sortable, TransformsExportData, UniqueFieldsInterface
{
    use Archive, HandleDateTimeAttributes, HasAccountingTransaction, HasFactory, HasFilters, HasSort, LogsActivity;

    const STATUS_SUBMITTED = 'submitted';

    const STATUS_ACKNOWLEDGED = 'acknowledged';

    const STATUS_FULFILLED = 'fulfilled';

    const STATUS_FULFILLED_AWAITING_INFO = 'fulfilled_awaiting_info';

    const STATUS_PICKED = 'picked'; // for Direct Warehouse

    const STATUS_PACKED = 'packed'; // for Direct Warehouse

    const STATUS_CANCELED = 'canceled';

    const STATUS = [
        self::STATUS_SUBMITTED,
        self::STATUS_ACKNOWLEDGED,
        self::STATUS_PICKED,
        self::STATUS_PACKED,
        self::STATUS_FULFILLED,
        self::STATUS_CANCELED,
    ];

    const TYPE_MANUAL = 'manual';

    const TYPE_SHIPSTATION = 'shipstation';

    const TYPE_STARSHIPIT = 'starshipit';

    const TYPE_SHIPMYORDERS = 'shipmyorders';
    const TYPE_VERACORE = 'veracore';

    const TYPE_FBA = 'fba';

    const TYPES = [
        self::TYPE_MANUAL,
        self::TYPE_SHIPSTATION,
        self::TYPE_STARSHIPIT,
        self::TYPE_SHIPMYORDERS,
        self::TYPE_FBA,
        self::TYPE_VERACORE,
    ];

    protected $casts = [
        'fulfilled_at' => 'datetime',
        'packing_slip_printed_at' => 'datetime',
        'submitted_to_sales_channel_at' => 'datetime',
        'cost' => 'float',
        'metadata' => 'array',
    ];

    protected $fillable = [
        'sales_order_id',
        'warehouse_id',
        'shipping_method_id',
        'requested_shipping_method_id',
        'requested_shipping_method',
        'fulfilled_shipping_method_id',
        'fulfilled_shipping_method',
        'cost',
        'fulfillment_type',
        'tracking_number',
        'fulfilled_at',
        'submitted_to_sales_channel_at',
        'metadata',
        'fulfillment_sequence',
        'status',
        'packing_slip_printed_at',
        'updated_at',
        'status',
    ];

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->salesOrder->id;
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [
            'id' => $this->id,
            'fulfillment_number' => $this->fulfillment_number,
        ];
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->dontSubmitEmptyLogs();
    }

    public function getAccountingTransactionData(): AccountingTransactionData
    {
        return (new BuildAccountingTransactionDataFromSalesOrderFulfillment($this))->handle();
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return $this->salesOrder->accountingTransaction;
    }

    public function getAccountingDateFieldName(): string
    {
        return 'fulfilled_at';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesOrder()
    {
        return $this->belongsTo(SalesOrder::class, 'sales_order_id');
    }

    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function shippingProvider()
    {
        return $this->belongsTo(ShippingProvider::class);
    }

    public function requestedShippingMethod()
    {
        return $this->belongsTo(ShippingMethod::class, 'requested_shipping_method_id');
    }

    public function fulfilledShippingMethod()
    {
        return $this->belongsTo(ShippingMethod::class, 'fulfilled_shipping_method_id');
    }

    public function salesOrderFulfillmentLines(): HasMany
    {
        $relation = $this->hasMany(SalesOrderFulfillmentLine::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (SalesOrderFulfillmentLine $fulfillmentLine) {
                $fulfillmentLine->delete();
            });
        });

        return $relation;
    }

    public function nonDropshipFulfillmentLines(): HasMany
    {
        return $this->salesOrderFulfillmentLines()
            ->whereHas('salesOrderLine', function ($query) {
                $query->whereHas('product');
                $query->whereHas('warehouse', fn ($query) => $query->whereNull('supplier_id'));
            });
    }

    public function accountingTransaction()
    {
        return $this->morphOne(AccountingTransaction::class, 'link');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function packingSlipQueues()
    {
        return $this->morphMany(PackingSlipQueue::class, 'link');
    }

    public function starshipitOrder()
    {
        return $this->hasOne(Starshipit\StarshipitOrder::class, 'sku_fulfillment_id');
    }

    public function shipstationOrder()
    {
        return $this->hasOne(ShipstationOrder::class, 'sku_fulfillment_id');
    }

    public function salesChannelLink(): MorphTo
    {
        return $this->morphTo();
    }

    public function amazonFulfillmentOrder(): HasOne
    {
        return $this->hasOne(AmazonFulfillmentOrder::class, 'sales_order_fulfillment_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getFbaShippingMethodAttribute()
    {
        if (isset($this->metadata['shipment-id'])) {
            return FulfilledShipment::with([])->where('shipment-id', $this->metadata['shipment-id'])->value('carrier') ?? 'Amazon FBA';
        }

        return 'Amazon FBA';
    }

    public function setShippingMethodIdAttribute($value)
    {
        $this->requested_shipping_method_id = $value;
    }

    /**
     * Retrieves a value from the `metadata` attribute.
     * To find values, use dot notation.
     *
     * @example $this->getMetadataValue('foo.bar') will return $this->metadata['foo']['bar']
     *
     * @return mixed
     */
    public function getMetadataValue(string $value)
    {
        return Arr::get($this['metadata'], $value);
    }

    public function hasMetadataValue(string $value): bool
    {
        return Arr::has($this['metadata'], $value);
    }


    public function getFulfillmentNumberAttribute(): string
    {
        return $this->salesOrder->sales_order_number.'.'.$this->fulfillment_sequence;
    }

    public function getTrackingLinkAttribute()
    {
        if (empty($this->tracking_number)) {
            return null;
        }
        // tracking link by Starshipit
        $starshipitOrder = $this->getShippingProviderOrder();
        if ($this->fulfillment_type === self::TYPE_STARSHIPIT && $starshipitOrder) {
            if (! empty($starshipitTrackingUrl = $starshipitOrder->packages[0]['tracking_url'])) {
                return $starshipitTrackingUrl;
            }
        }

        // tracking link by shopify
        if ($this->salesOrder->salesChannel->integrationInstance->isShopify()) {
            $shopifyOrder = $this->salesOrder->order_document;
            $shopifyFulfillmentId = app(ShopifyOrderMappingRepository::class)->getShopifyFulfillmentIdForSalesOrderFulfillment($this);
            if ($shopifyFulfillmentId) {
                $shopifyFulfillment = collect($shopifyOrder->fulfillments)->firstWhere('id', $shopifyFulfillmentId);
                if ($shopifyFulfillment['tracking_url'] ?? null) {
                    return $shopifyFulfillment['tracking_url'];
                }
            }
        }

        if (empty($this->fulfilled_shipping_method_id)) {
            return null;
        }

        // tracking link by carrier
        $carrier = $this->fulfilledShippingMethod->shippingCarrier;
        if (! empty($carrier->tracking_link)) {
            return $carrier->tracking_link.$this->tracking_number;
        }

        return null;
    }

    public function totalQuantity(): Attribute
    {
        return Attribute::get(fn () => $this->salesOrderFulfillmentLines->sum('quantity'));
    }

    public function isManual(): Attribute
    {
        return Attribute::get(fn () => $this->fulfillment_type === self::TYPE_MANUAL);
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function save(array $options = [])
    {
        if (empty($this->user_id)) {
            $this->user_id = auth()->id();
        }

        if (! $this->exists && empty($this->status)) {
            $this->status = self::getDefaultStatusByFulfillmentType($this->fulfillment_type);
        }
        if (empty($this->fulfillment_sequence)) {
            $this->fulfillment_sequence = static::query()->where('sales_order_id', $this->sales_order_id)->latest('fulfillment_sequence')->value('fulfillment_sequence') + 1;
        }

        return parent::save($options);
    }

    /**
     * @param  string|null  $fulfillmentType
     * @return string
     */
    public static function getDefaultStatusByFulfillmentType(?string $fulfillmentType = null): string{
        if(empty($fulfillmentType)) return self::STATUS_SUBMITTED;
        if ($fulfillmentType == self::TYPE_MANUAL) {
            return self::STATUS_FULFILLED;
        } elseif ($fulfillmentType == self::TYPE_FBA) {
            return self::STATUS_FULFILLED_AWAITING_INFO;
        } else {
            return self::STATUS_SUBMITTED;
        }
    }

    /**
     * @throws Throwable
     */
    public function delete(bool $ignoreShippingProviderExceptions = false, bool $fromDropshipShipmentDeletion = false)
    {
        customlog('SKU-6191', 'Sales order fulfillment '.$this->fulfillment_number.' for '.$this->salesOrder->sales_order_number.' being deleted', [
            'debug' => debug_pretty_string(),
        ], 7);
        if ($this->status === self::STATUS_FULFILLED) {
            customlog('SKU-6919', 'Sales order fulfillment '.$this->fulfillment_number.' for '.$this->salesOrder->sales_order_number.' is a fulfilled fulfillment being deleted', [
                'debug' => debug_pretty_string(),
            ], 7);
        }
        $this->loadMissing('salesOrder');

        $productIds = [];

        // Attempt to delete submitted fulfillment from the shipping provider.
        // If the fulfillment is already fulfilled, we don't want to delete it.
        if ($this->status === self::STATUS_SUBMITTED) {
            try {
                if ($this->fulfillment_type == self::TYPE_SHIPSTATION) {
                    dispatch_sync(( new \App\Jobs\ShipStation\DeleteOrder($this) )->onConnection('sync'));
                } elseif ($this->fulfillment_type == self::TYPE_STARSHIPIT) {
                    dispatch_sync(( new \App\Jobs\Starshipit\DeleteOrder($this) )->onConnection('sync'));
                } elseif ($this->fulfillment_type == self::TYPE_VERACORE) {
                    /** @var VeracoreManager $manager */
                    $manager = app(VeracoreManager::class);
                    $manager->cancelOrderForFulfillment($this);
                } elseif ($this->fulfillment_type == self::TYPE_FBA) {
                    $this->getShippingProviderOrder()?->update(['sales_order_fulfillment_id' => null]);
                }

                $this->getShippingProviderOrder()?->delete();
            } catch (Throwable $exception) {
                if (! $ignoreShippingProviderExceptions) {
                    throw $exception;
                } else {
                    // We break the link to the shipping provider, so we can continue
                    // with the sku fulfillment deletion.
                    if ($this->fulfillment_type == self::TYPE_SHIPSTATION || $this->fulfillment_type == self::TYPE_SHIPMYORDERS) {
                        $this->getShippingProviderOrder()?->delete();
                    } elseif ($this->fulfillment_type == self::TYPE_STARSHIPIT) {
                        $this->getShippingProviderOrder()?->update(['sku_fulfillment_id' => null]);
                    }
                }
            }
        } else {
            if ($this->fulfillment_type == self::TYPE_FBA) {
                $this->getShippingProviderOrder()?->update(['sales_order_fulfillment_id' => null]);
            } else {
                $this->getShippingProviderOrder()?->delete();
            }

        }

        // Shipping provider deletion succeeded. We can now delete the fulfillment.
        $deleted = DB::transaction(function () use ($ignoreShippingProviderExceptions, &$productIds, $fromDropshipShipmentDeletion) {
            $this->salesOrderFulfillmentLines()->each(function ($salesOrderFulfillmentLine) use (&$productIds) {
                $salesOrderFulfillmentLine->salesOrderLine->fulfilled_quantity -= $salesOrderFulfillmentLine->quantity;
                $salesOrderFulfillmentLine->salesOrderLine->update();
                $productIds[] = $salesOrderFulfillmentLine->salesOrderLine->product_id;
            });
            $this->salesOrderFulfillmentLines()->delete();

            // If is dropship, delete any shipments
            // linked to the fulfillment. Note that dropship
            // shipments/receipts don't create inventory movements,
            // so we don't need to worry about those.
            if (!$fromDropshipShipmentDeletion) {
                PurchaseOrderShipment::with([])->where('sales_order_fulfillment_id', $this->id)
                    ->get()->each(function (PurchaseOrderShipment $shipment) {
                        $shipment->delete();
                        $shipment->purchaseOrder->refreshReceiptStatus(now());
                    });
            }

            if ($deleted = parent::delete()) {
                // Update the fulfillment status of the sales order
                FulfillSalesOrderService::make($this->salesOrder)->updateFulfillmentStatus();

                // we don't want to update the fulfillment sequence because the fulfillment number will be different on the SKU and the Shipping Provider,
                // and we don't need to update the order_number on the Shipping Provider because it reserves the two order_numbers (the old and the new),
                // but this maybe hide "Order Exists" issue if the order had more than one fulfillment at the same shipping provider (I think it's an extreme case)
                /** @see SKU-4051 */

                // clear the fulfillments map for the fulfillment that is deleted in the sales channel record
                if ($this->salesOrder->salesChannel->integrationInstance->isShopify()) {
                    $shopifyOrder = $this->salesOrder->shopifyOrder;
                    if ($shopifyOrder) {
                        // Delete any shopify order mappings
                        app(ShopifyOrderMappingRepository::class)->deleteMappingsForSalesOrderFulfillment($this);
                    }

                    if ($shopifyOrder->hasSalesChannelFulfillmentsOutOfSync()) {
                        customlog('out-of-sync', $this->salesOrder->sales_order_number.' had fulfillments that were deleted, and the shopify order was determined to be out of sync, so marking out of sync');
                        $this->salesOrder->notes()->create(['note' => 'Fulfillments were deleted, and the shopify order was determined to be out of sync, so marking out of sync']);
                        $this->salesOrder->update(['fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC]);
                    }

                    // TODO: should cancel the Shopify fulfillment?
                }
                // Magento
                if ($this->salesOrder->salesChannel->integrationInstance->isMagento()) {
                    $magentoOrder = $this->salesOrder->magentoOrder;
                    if ($magentoOrder) {
                        $fulfillmentsMap = collect($magentoOrder->fulfillments_map);
                        $magentoOrder->fulfillments_map = $fulfillmentsMap->where('sku_fulfillment_id', '!=', $this->id)->values()->toArray();
                        $magentoOrder->save();
                    }

                    // TODO: should cancel the Shopify fulfillment?
                }
            }

            return $deleted;
        });
        // Update the inventory cache
        (new UpdateProductsInventoryAndAvgCost($productIds))->handle();

        return $deleted;
    }

    /**
     * Get Shipping Provider's Order.
     *
     * @return SubmittedFulfillment|ShipstationOrder|StarshipitOrder|Model|null
     */
    public function getShippingProviderOrder(): SubmittedFulfillment|ShipstationOrder|Starshipit\StarshipitOrder|Model|null
    {
        if ($this->fulfillment_type == self::TYPE_SHIPSTATION) {
            return ShipstationOrder::with([])->where('sku_fulfillment_id', $this->id)->first();
        }

        if ($this->fulfillment_type == self::TYPE_STARSHIPIT) {
            return Starshipit\StarshipitOrder::with([])->where('sku_fulfillment_id', $this->id)->first();
        }

        if($this->fulfillment_type == self::TYPE_VERACORE){
            return VeracoreOrder::with([])->where('sku_fulfillment_id', $this->id)->first();
        }

        if ($this->fulfillment_type == self::TYPE_FBA) {
            return $this->amazonFulfillmentOrder;
        }

        return null;
    }

    /**
     * Get Shipping Provider's Order Status.
     */
    public function getMappedShippingProviderOrderStatus(): ?string
    {
        $shippingProviderOrder = $this->getShippingProviderOrder();

        if (! $shippingProviderOrder) {
            return null;
        }

        if ($this->fulfillment_type == self::TYPE_SHIPSTATION) {
            if ($shippingProviderOrder->orderStatus == 'cancelled') {
                return self::STATUS_CANCELED;
            }

            if ($shippingProviderOrder->orderStatus == 'shipped') {
                return self::STATUS_FULFILLED;
            }

            return self::STATUS_SUBMITTED;
        }

        if ($this->fulfillment_type == self::TYPE_STARSHIPIT) {
            if (in_array($shippingProviderOrder->status, ['Printed', 'Delivered'])) {
                return self::STATUS_FULFILLED;
            }

            return self::STATUS_SUBMITTED;
        }

        if ($this->fulfillment_type == self::TYPE_FBA) {
            //TODO: map FBA fulfillment order status
        }

        return null;
    }

    /**
     * Determine if the sales order fulfillment is used (deletable).
     *
     * @return array|bool
     */
    public function isUsed()
    {
        $usage = [];
        $shippingProviderOrderStatus = $this->getMappedShippingProviderOrderStatus();

        if ($shippingProviderOrderStatus && $shippingProviderOrderStatus == self::STATUS_FULFILLED) {
            $usage['status'] = __('messages.sales_order.can_not_delete_shipped_fulfillment', ['shipping_provider' => $this->fulfillment_type]);
        }

        return count($usage) ? $usage : false;
    }

    public function fulfilled(): bool
    {
        return $this->status === self::STATUS_FULFILLED;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.sales_order_fulfillment.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->add('backordered')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['fulfillment_number', 'customer_reference', 'tracking_number', 'item_sku'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function toShipStationOrder()
    {
        $ssOrderPrefix = config('shipstation.order_prefix');
        $shipStationIntegrationInstance = IntegrationInstance::shipstation()->firstOrFail();

        $order = $this->salesOrder->toShipStationOrder(false);
        $order->items = $this->salesOrderFulfillmentLines->map->toShipStationOrderItem()->all();
        $order->orderKey = $ssOrderPrefix.'_'.$this->id.'_'.$this->fulfillment_number;
        $order->orderNumber = $this->fulfillment_number;
        $customFieldId = @$shipStationIntegrationInstance->integration_settings['settings']['gift_card_note_sales_order_custom_field_id'];
        if ($customFieldId) {
            $giftMessage = $this->salesOrder->customFieldValues->where('custom_field_id', $customFieldId)->first()?->value ?? '';
            $order->gift = ! empty($giftMessage);
            $order->giftMessage = $giftMessage;
        }

        if (count($order->items) === 1) {
            $line = $this->salesOrderFulfillmentLines->first();
            if ($line->salesOrderLine->product) {
                $order->dimensions = new Dimensions();
                $order->dimensions->height = $line->salesOrderLine->product->height;
                $order->dimensions->width = $line->salesOrderLine->product->width;
                $order->dimensions->length = $line->salesOrderLine->product->length;
                $order->dimensions->units = $line->salesOrderLine->product->dimension_unit == 'cm' ? Dimensions::UNIT_CENTIMETERS : Dimensions::UNIT_INCHES;
            }
        }

        if ($this->requested_shipping_method_id) {
            $starshipitMethodMapping = ShippingMethodMappingsSkuToShippingProviderMethod::with([])
                ->where(['shipping_provider_id' => $shipStationIntegrationInstance->id, 'shipping_method_id' => $this->requested_shipping_method_id])
                ->first();

            if ($starshipitMethodMapping) {
                $order->carrierCode = $starshipitMethodMapping->shipping_provider_carrier;
                $order->serviceCode = $starshipitMethodMapping->shipping_provider_method;
            } else {
                $order->requestedShippingService = $this->requestedShippingMethod->full_name;
            }
        } else {
            $order->requestedShippingService = $this->requested_shipping_method;
        }

        // If the fulfillment warehouse is mapped to a shipstation warehouse,
        // we provide the warehouse in the request.
        /** @var ShippingProviderWarehouseMapping $mapping */
        $mapping = ShippingProviderWarehouseMapping::query()
            ->where('warehouse_id', $this->warehouse_id)
            ->where('provider_type', ShipstationWarehouse::class)
            ->first();

        if($mapping){
            if (!isset($this->advancedOptions))
            {
                $order->advancedOptions = new AdvancedOptions();
            }
            $order->advancedOptions->warehouseId = $mapping->provider->shipstation_id;
        }

        unset($order->orderTotal);

        return $order;
    }

    /**
     * @throws Throwable
     */
    public function toStarshipitOrder(?Starshipit\StarshipitOrder $starshipOrder = null)
    {
        $ssOrderPrefix = config('shipstation.order_prefix');

        $order = $this->salesOrder->toStarshipitOrder(false);

        throw_if(! $this->warehouse->address, 'Warehouse "'.$this->warehouse->name.'" has no address');

        $order->sender_details = $this->warehouse->address->toStarshipitAddress();
        $order->sender_details->name = $this->salesOrder->store->company_name ?: $this->salesOrder->store->name;
        $order->sender_details->company = null;

        $order->items = $this->salesOrderFulfillmentLines->map->toStarshipitOrderItem($starshipOrder)->toArray();
        // when updating, we should send old order items that not found on order to delete them
        if ($starshipOrder) {
            foreach ($starshipOrder->items as $item) {
                if (! collect($order->items)->firstWhere('item_id', $item['item_id'])) {
                    $order->items[] = ['item_id' => $item['item_id'], 'delete' => true];
                }
            }
        }

        $order->reference = $ssOrderPrefix.$this->id;
        $order->order_number = $this->fulfillment_number;
        $order->signature_required = $this->metadata['signature_required'] ?? false;

        if ($starshipOrder && $starshipOrder->order_id) {
            $order->order_id = $starshipOrder->order_id;
        }

        $package = new OrderPackage();

        $firstLine = $this->salesOrderFulfillmentLines->first();

        /*
         * Set dimensions only if order has one line and qty 1
         */
        if (count($order->items) === 1 && $firstLine->quantity === 1) {
            $package->height = Helpers::dimensionConverter($firstLine->salesOrderLine->product->height, $firstLine->salesOrderLine->product->dimension_unit, 'm');
            $package->width = Helpers::dimensionConverter($firstLine->salesOrderLine->product->width, $firstLine->salesOrderLine->product->dimension_unit, 'm');
            $package->length = Helpers::dimensionConverter($firstLine->salesOrderLine->product->length, $firstLine->salesOrderLine->product->dimension_unit, 'm');
        }

        $weightTally = 0;
        $this->salesOrderFulfillmentLines->each(function (SalesOrderFulfillmentLine $salesOrderFulfillmentLine) use (&$weightTally) {
            $weightTally += $salesOrderFulfillmentLine->quantity *
                Helpers::weightConverter(
                    $salesOrderFulfillmentLine->salesOrderLine->product->weight,
                    $salesOrderFulfillmentLine->salesOrderLine->product->weight_unit,
                    Product::WEIGHT_UNIT_KG
                );
        });

        $package->weight = $weightTally;

        if ($starshipOrder && ! empty($starshipOrder->packages)) {
            $package->package_id = $starshipOrder->packages[0]['package_id'];
        }
        $order->packages = [$package];

        // when updating, remove unused packages
        if ($starshipOrder && ! empty($starshipOrder->packages)) {
            foreach ($starshipOrder->packages as $index => $p) {
                if (count($order->items) === 1 && $index === 0) {
                    continue;
                }
                $order->packages[] = ['package_id' => $p['package_id'], 'delete' => true];
            }
        }

        /*
         * We must only use the shipping method id if there is no requested shipping method override.
         */
        // set shipping method
        if ($this->requested_shipping_method_id && empty($this->requested_shipping_method)) {
            customlog('starshipit', $this->salesOrder->sales_order_number . ': Requested shipping method is specified', days: 7);
            $starshipitMethodMapping = ShippingMethodMappingsSkuToShippingProviderMethod::with([])
                ->where(['shipping_provider_id' => IntegrationInstance::starshipit()->firstOrFail()->id, 'shipping_method_id' => $this->requested_shipping_method_id])
                ->first();

            if ($starshipitMethodMapping) {
                $order->shipping_method = $starshipitMethodMapping->full_name;
                customlog('starshipit', $this->salesOrder->sales_order_number . ': mapping found ' . $starshipitMethodMapping->full_name, days: 7);
            } else {
                $order->shipping_method = $this->requestedShippingMethod->full_name;
            }
        } else {
            $order->shipping_method = $this->requested_shipping_method;
            customlog('starshipit', $this->salesOrder->sales_order_number . ': Requested shipping method is empty, using requested_shipping_method instead: ' . $this->requested_shipping_method, days: 7);
        }

        return $order;
    }

    /**
     * Mark the fulfillment as submitted to the sales channel
     */
    public function markAsSubmittedToSalesChannel(Carbon $date): void
    {
        $this->submitted_to_sales_channel_at = $date;
        $this->save();
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeAccountingReady(Builder $builder): Builder
    {
        return $builder
            ->whereDoesntHave('warehouse', function (Builder $builder) {
                $builder->where('type', Warehouse::TYPE_SUPPLIER);
            })
            ->whereNotNull('fulfilled_at');
    }

    public function scopeFilterSalesOrder(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'whereHas' : 'orWhereHas';
        if ($relation['key'] === 'fulfillment_number') {
            return $builder->{$function}('salesOrder', function (Builder $builder) use ($value, $operator) {
                $builder->filterKey(['is_relation' => false, 'key' => DB::raw('CONCAT_WS(".", `sales_order_number`, `sales_order_fulfillments`.`fulfillment_sequence`)')], $operator, $value);
            });
        } elseif ($relation['key'] === 'shippingMethod') {
            $builder->{$function}('salesOrder', function (Builder $builder) use ($value, $operator) {
                $builder->whereHas('shippingMethod', function (Builder $builder) use ($value, $operator) {
                    $builder->filterKey(['is_relation' => false, 'key' => $builder->qualifyColumn('full_name')], $operator, $value);
                })->orWhere(function (Builder $builder) use ($value, $operator) {
                    $builder->whereNull('shipping_method_id')
                        ->filterKey(['is_relation' => false, 'key' => 'requested_shipping_method'], $operator, $value);
                });
            });
        }

        return $builder->filterWhereNestedRelation($relation, $operator, $value, $conjunction);
    }

    public function scopeFilterRequestedShippingMethod(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        $builder->{$function}(function (Builder $builder) use ($value, $operator) {
            $builder->whereHas('requestedShippingMethod', function (Builder $builder) use ($value, $operator) {
                $builder->filterKey(['is_relation' => false, 'key' => $builder->qualifyColumn('full_name')], $operator, $value);
            })->orWhere(function (Builder $builder) use ($value, $operator) {
                $builder->whereNull('requested_shipping_method_id')
                    ->filterKey(['is_relation' => false, 'key' => 'requested_shipping_method'], $operator, $value);
            });
        });
    }

    public function scopeFilterFulfilledShippingMethod(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        $builder->{$function}(function (Builder $builder) use ($value, $operator) {
            $builder->whereHas('fulfilledShippingMethod', function (Builder $builder) use ($value, $operator) {
                $builder->filterKey(['is_relation' => false, 'key' => $builder->qualifyColumn('full_name')], $operator, $value);
            })->orWhere(function (Builder $builder) use ($value, $operator) {
                $builder->whereNull('fulfilled_shipping_method_id')
                    ->filterKey(['is_relation' => false, 'key' => 'fulfilled_shipping_method'], $operator, $value);
            });
        });
    }

    public function scopeFilterBackordered(Builder $builder, array $relation, string $operator, $value = null, $conjunction = 'and'): Builder
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        return $builder->{$function}(function (Builder $builder) use ($operator, $value) {
            if ($operator === 'yes') {
                // There should be queues
                return $builder->whereHas('salesOrderFulfillmentLines.salesOrderLine.backorderQueue');
            } elseif ($operator === 'no') {
                // Doesn't have backorder queues
                return $builder->whereDoesntHave('salesOrderFulfillmentLines.salesOrderLine.backorderQueue');
            } elseif ($operator === 'released_by_po') {
                // the backorderQueue should be released by a purchase order
                return $builder->whereHas('salesOrderFulfillmentLines.salesOrderLine.backorderQueue', function (Builder $builder) use ($value) {
                    return $builder->releasedByPurchaseOrder($value ?: null);
                });
            }
        });
    }

    public function scopeSortSalesOrder(Builder $builder, array $relation, bool $ascending)
    {
        // If the table is already joined, we abort
        if (! $this->isTableJoined($builder, 'sales_orders')) {
            $builder->leftJoin('sales_orders', 'sales_order_fulfillments.sales_order_id', '=', 'sales_orders.id');
        }

        $column = $relation['key'];
        if ($column == 'fulfillment_number') {
            $column = DB::raw("CONCAT_WS('.', `sales_orders`.`sales_order_number`, `sales_order_fulfillments`.`fulfillment_sequence`)");
        }

        return $builder->orderBy($column, $ascending ? 'asc' : 'desc');
    }

    public function scopeSortWarehouse(Builder $builder, array $relation, bool $ascending)
    {
        // If the table is already joined, we abort
        if (! $this->isTableJoined($builder, 'warehouses')) {
            $builder->leftJoin('warehouses', 'sales_order_fulfillments.warehouse_id', '=', 'warehouses.id');
        }

        return $builder->orderBy($relation['key'], $ascending ? 'asc' : 'desc');
    }

    /**
     * {@inheritDoc}
     */
    public static function transformExportData(array $data): array
    {
        return BaseExporter::groupByLines($data, 'fulfillment_number');
    }

    /**
     * {@inheritDoc}
     */
    public static function getExportableFields(): array
    {
        return SalesOrderFulfillmentDataImporter::getExportableFields();
    }

    public function removeLinesNotIn(array $retainedFulfillmentLineIds)
    {
        $this->salesOrderFulfillmentLines()->whereNotIn('id', $retainedFulfillmentLineIds)->delete();
    }

    public function getReference(): ?string
    {
        return $this->fulfillment_number;
    }

    public static function getUniqueFields(): array
    {
        return ['sales_order_id', 'fulfillment_sequence', 'warehouse_id'];
    }
}
