<?php

namespace App\Models;

use App\Abstractions\CustomFieldsInterface;
use App\Abstractions\FinancialDocumentInterface;
use App\Abstractions\HasNotesInterface;
use App\Abstractions\SendEmailContextInterface;
use App\Abstractions\UniqueFieldsInterface;
use App\Contracts\HasReference;
use App\Data\AccountingTransactionData;
use App\DTO\SalesOrderTotalsDto;
use App\Enums\FinancialLineClassificationEnum;
use App\Exceptions\SalesOrder\LineFulfilledAtWarehouseException;
use App\Exporters\BaseExporter;
use App\Exporters\MapsExportableFields;
use App\Exporters\TransformsExportData;
use App\Helpers;
use App\Importers\DataImporter;
use App\Importers\DataImporters\SalesOrderDataImporter;
use App\Importers\ImportableInterface;
use App\Integrations\Amazon;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Models\Amazon\Order as AmazonOrder;
use App\Models\Amazon\ShippingService;
use App\Models\Concerns\Archive;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\CachesOrderCurrency;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasAccountingTransaction;
use App\Models\Concerns\HasCustomFields;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasNotesTrait;
use App\Models\Concerns\HasSort;
use App\Models\Concerns\LogsActivity;
use App\Models\Concerns\SalesOrder\HasDropship;
use App\Models\Concerns\TaxRateTrait;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\Ebay\Order as EbayOrder;
use App\Models\Magento\Order as MagentoOrder;
use App\Models\Shopify\ShopifyOrder as ShopifyOrder;
use App\Models\WooCommerce\Order as WooCommerceOrder;
use App\Repositories\Shopify\ShopifyOrderMappingRepository;
use App\Repositories\WarehouseRepository;
use App\Response;
use App\SDKs\ShipStation\Model\XML\BillTo;
use App\SDKs\ShipStation\Model\XML\Item;
use App\SDKs\ShipStation\Model\XML\Order;
use App\SDKs\ShipStation\Model\XML\ShipTo;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionDataFromSalesOrder;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\Payments\Payable;
use App\Services\SalesOrder\ApproveSalesOrderService;
use App\Services\SalesOrder\FulfillSalesOrderService;
use App\Services\SalesOrder\SalesOrderManager;
use App\Services\SalesOrder\WarehouseRoutingMethod;
use Carbon\Carbon;
use Closure;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use InvalidArgumentException;
use Kirschbaum\PowerJoins\PowerJoins;
use Picqer\Barcode\BarcodeGeneratorPNG;
use Spatie\Activitylog\LogOptions;
use Spatie\Tags\HasTags;
use Throwable;

/**
 * Class SalesOrder.
 *
 *
 * @property int $id
 * @property string $sales_order_number
 * @property int $sales_channel_id
 * @property string $order_status
 * @property string $fulfillment_status
 * @property string $fulfillment_channel
 * @property int|null $customer_id
 * @property int|null $shipping_address_id
 * @property int|null $billing_address_id
 * // TODO: Analyze these order level costs
 * @property int $currency_id
 * @property float $currency_rate
 * @property int $currency_id_tenant_snapshot
 * @property int|null $shipping_method_id
 * @property string|null $requested_shipping_method
 * @property string|null $mapped_shipping_method
 * @property string|null $payment_status
 * @property Carbon $order_date
 * @property Carbon|null $ship_by_date
 * @property Carbon|null $deliver_by_date
 * @property int|null $store_id
 * @property Carbon|null $approved_at
 * @property Carbon|null $reserved_at
 * @property Carbon|null $fulfilled_at
 * @property Carbon|null $fully_paid_at
 * @property Carbon|null $canceled_at
 * @property bool $is_fba
 * @property bool $is_replacement_order
 * @property float $tax_total
 * @property bool $is_tax_included
 * @property int|null $tax_rate_id
 * @property float|null $tax_rate
 * @property float $discount_total
 * @property float $proration_fulfilled
 * @property int $sales_channel_order_id
 * @property string $sales_channel_order_type
 * @property SalesOrderLine[]|Collection $productLines
 * @property SalesOrderLine[]|Collection $backorderedLines
 * @property SalesOrderLine[]|Collection $warehousedProductLines
 * @property SalesOrderLine[]|Collection $nonDropshipProductLines
 * @property SalesOrderLine[]|Collection $dropshipLines
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $packing_slip_printed_at
 * @property Carbon|null $last_synced_from_sales_channel_at
 *
 * Calculated Properties
 * @property-read float $product_subtotal
 * @property-read float $total_discount
 * @property-read float $calculated_discount_total
 * @property-read float $shipping_cost
 * @property-read float $fulfillment_revenue
 * @property-read float $calculated_total
 * @property-read Store|null $active_store
 * @property-read int|null $active_store_id
 * @property-read Address|null $shippingAddress
 * @property-read Address|null $billingAddress
 * @property-read Collection|SalesOrderLine[] $salesOrderLines
 * @property-read Collection|SalesOrderLineFinancial[] $salesOrderLineFinancials
 * @property-read Collection|Payment[] $payments
 * @property-read ShopifyOrder $shopifyOrder
 * @property-read WooCommerceOrder $wooCommerceOrder
 * @property-read EbayOrder $ebayOrder
 * @property-read MagentoOrder|null $magentoOrder
 * @property-read Customer $customer
 * @property-read string $customer_reference
 * @property-read bool $is_fully_fulfilled
 * @property-read bool $is_awaiting_tracking
 * @property-read bool $is_partially_fulfilled
 * @property-read bool $is_fully_fulfillable
 * @property-read bool $is_fulfillable
 * @property-read bool $closed
 * @property-read float $total_paid
 * @property-read float $total
 * @property-read bool $is_fully_paid
 * @property-read bool $has_processable_quantity
 * @property-read SalesChannel $salesChannel
 * @property-read AmazonOrder $salesChannelOrder // Will add more or conditions of type once using this relationship
 * @property-read Collection|SalesCredit[] $sales_credits
 * @property-read Collection|SalesCreditReturn[] $salesCreditReturns
 * @property-read Collection|AmazonOrder|MagentoOrder|ShopifyOrder|WooCommerceOrder|EbayOrder $order_document
 * @property-read string $sales_channel_shipping_service
 * @property-read Collection|SalesOrderFulfillment[] $salesOrderFulfillments
 * @property-read Collection|SalesCredit[] $salesCredits
 * @property-read Currency $currency
 * @property-read Currency $currencyTenantSnapshot
 * @property-read ShippingMethod $shippingMethod
 * @property-read Store $store
 * @property-read Collection|Note[] $notes
 * @property-read Collection|PurchaseOrder[] $purchaseOrders
 * @property-read AccountingTransaction $accountingTransaction
 * @property-read Collection|FinancialLine[] $financialLines
 * @property-read Collection|CustomFieldValue[] $customFieldValues
 * @property-write string $store_name
 * @property-write string $currency_code
 * @property-write string $shipping_method_name
 * @property-read  string $integration_instance_name
 */
class SalesOrder extends Model
    implements
    CustomFieldsInterface,
    Filterable,
    FinancialDocumentInterface,
    HasNotesInterface,
    HasReference,
    ImportableInterface,
    MapsExportableFields,
    Payable,
    SendEmailContextInterface,
    Sortable,
    TransformsExportData,
    UniqueFieldsInterface
{
    use Archive;
    use LogsActivity;
    use BulkImport;
    use CachesOrderCurrency;
    use HandleDateTimeAttributes;
    use HasAccountingTransaction;
    use HasCustomFields;
    use HasDropship;
    use HasFactory;
    use HasFilters;
    use HasNotesTrait;
    use HasSort;
    use HasTags;
    use PowerJoins;
    use TaxRateTrait;
    use Upsert;

    const BULK_THRESHOLD = 2;

    /**
     * Order Status.
     */
    const STATUS_DRAFT = 'draft';

    const STATUS_RESERVED = 'reserved';

    const STATUS_OPEN = 'open';

    const STATUS_CLOSED = 'closed';

    const STATUS_CANCELED = 'cancelled';

    const STATUSES = [
        self::STATUS_DRAFT,
        self::STATUS_RESERVED,
        self::STATUS_OPEN,
        self::STATUS_CLOSED,
    ];

    const COST_UNIT = 'unit_cost';

    const COST_SHIPPING = 'shipping_cost';

    const COST_MARKETPLACE = 'marketplace_cost';

    const COST_PAYMENT = 'payment_cost';

    const COST_LANDED = 'landed_cost';

    const COSTS = [
        self::COST_UNIT,
        self::COST_SHIPPING,
        self::COST_MARKETPLACE,
        self::COST_PAYMENT,
        self::COST_LANDED,
    ];

    /**
     * Fulfillment Statues.
     */
    const FULFILLMENT_STATUS_UNFULFILLED = 'unfulfilled';

    const FULFILLMENT_STATUS_OUT_OF_SYNC = 'out_of_sync';

    const FULFILLMENT_STATUS_PARTIALLY_FULFILLED = 'partially_fulfilled';

    const FULFILLMENT_STATUS_FULFILLED = 'fulfilled';

    const FULFILLMENT_STATUS_OVER_FULFILLED = 'over_fulfilled';

    const FULFILLMENT_STATUS_AWAITING_TRACKING = 'awaiting_tracking';

    const FULFILLMENT_STATUES = [
        self::FULFILLMENT_STATUS_UNFULFILLED,
        self::FULFILLMENT_STATUS_OUT_OF_SYNC,
        self::FULFILLMENT_STATUS_PARTIALLY_FULFILLED,
        self::FULFILLMENT_STATUS_FULFILLED,
        self::FULFILLMENT_STATUS_AWAITING_TRACKING,
    ];

    const FULFILLABLE_STATUS_NONE = 'none';

    const FULFILLABLE_STATUS_SOME = 'some';

    const FULFILLABLE_STATUS_ALL = 'all';

    const FULFILLABLE_STATUSES = [
        self::FULFILLABLE_STATUS_ALL,
        self::FULFILLABLE_STATUS_SOME,
        self::FULFILLABLE_STATUS_NONE,
    ];

    /**
     * Payment Statuses.
     */
    const PAYMENT_STATUS_UNPAID = 'unpaid';

    const PAYMENT_STATUS_PARTIALLY_PAID = 'partially_paid';

    const PAYMENT_STATUS_PAID = 'paid';

    const PAYMENT_STATUS_REFUNDED = 'refunded';

    const PAYMENT_STATUS_PARTIALLY_REFUNDED = 'partially_refunded';

    const PAYMENT_STATUSES = [
        self::PAYMENT_STATUS_UNPAID,
        self::PAYMENT_STATUS_PARTIALLY_PAID,
        self::PAYMENT_STATUS_PAID,
        self::PAYMENT_STATUS_REFUNDED,
        self::PAYMENT_STATUS_PARTIALLY_REFUNDED,
    ];

    /**
     * Casting.
     */
    protected $casts = [
        'ship_by_date' => 'datetime',
        'deliver_by_date' => 'datetime',
        'order_date' => 'datetime',
        'archived_at' => 'datetime',
        'approved_at' => 'datetime',
        'reserved_at' => 'datetime',
        'fulfilled_at' => 'datetime',
        'fully_paid_at' => 'datetime',
        'tax_revenue' => 'float',
        'product_total' => 'float',
        'is_fba' => 'boolean',
        'is_replacement_order' => 'boolean',
        'discount_lines' => 'array',
        'tax_lines' => 'array',
        'packing_slip_printed_at' => 'datetime',
        'last_synced_from_sales_channel_at' => 'datetime',
        'canceled_at' => 'datetime',
        'is_tax_included' => 'boolean',
        'sales_channel_id' => 'integer',
        'fulfillment_channel' => 'string',
    ];

    protected $fillable = [
        'sales_channel_id',
        'fulfillment_status',
        'currency_code',
        'order_date',
        'customer_id',
        'shipping_method_id',
        'shipping_method_name',
        'requested_shipping_method',
        'payment_status',
        'ship_by_date',
        'deliver_by_date',
        'fulfilled_at',
        'store_id',
        'store_name',
        'currency_id',
        'currency_code',
        'shipping_address_id',
        'billing_address_id',
        'sales_order_number',
        'discount_lines',
        'tax_lines',
        'packing_slip_printed_at',
        'last_synced_from_sales_channel_at',
        'is_tax_included',
        'tax_rate_id',
        'updated_at',
        'sales_channel_order_id',
        'sales_channel_order_type',
        'sales_channel_id',
        'fulfillment_channel',
        'order_status',
        'is_fba',
        'is_replacement_order',
        'canceled_at',
    ];

    protected $attributes = [
        'sales_channel_id' => SalesChannel::LOCAL_CHANNEL_ID,
        'order_status' => self::STATUS_DRAFT,
    ];

    protected $touches = ['salesCreditReturns'];

    public function getReference(): ?string
    {
        return $this->sales_order_number;
    }

    public static function getUniqueFields(): array
    {
        return ['sales_order_number', 'sales_channel_id'];
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->id;
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [
            'id' => $this->id,
            'sales_order_number' => $this->sales_order_number,
        ];
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    /**
     * @throws Throwable
     */
    public function getAccountingTransactionData(): AccountingTransactionData
    {
        return (new BuildAccountingTransactionDataFromSalesOrder($this))->handle();
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return null;
    }

    public function getAccountingDateFieldName(): string
    {
        return 'order_date';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesChannel()
    {
        return $this->belongsTo(SalesChannel::class);
    }

    public function shippingAddress()
    {
        return $this->belongsTo(Address::class);
    }

    public function billingAddress()
    {
        return $this->belongsTo(Address::class);
    }

    public function salesOrderLines(): HasMany
    {
        $relation = $this->hasMany(SalesOrderLine::class);

        $relation->onDelete(function (Builder $builder) {
            $builder->with('inventoryMovements', 'inventoryMovements.layer');

            return $builder->each(function (SalesOrderLine $salesOrderLine) {
                $salesOrderLine->delete();
            });
        });

        return $relation;
    }

    public function salesOrderLineFinancials(): HasManyThrough
    {
        return $this->hasManyThrough(SalesOrderLineFinancial::class, SalesOrderLine::class);
    }

    public function warehousedProductLines()
    {
        return $this->salesOrderLines()
            ->where('is_product', true)
            ->whereNotNull('product_id')
            ->whereNotNull('warehouse_id');
    }

    public function productLines(): HasMany
    {
        return $this->salesOrderLines()
            ->where('is_product', true)
            ->whereNotNull('product_id');
    }

    public function nonDropshipProductLines(): HasMany
    {
        return $this->warehousedProductLines()
            ->where(function (Builder $builder) {
                $builder->where('warehouse_routing_method', '!=', WarehouseRoutingMethod::DROPSHIP);
                $builder->orWhereNull('warehouse_routing_method');
            });
    }

    public function backorderedLines()
    {
        return $this->productLines()
            ->whereHas('backorderQueue', function (Builder $builder) {
                return $builder->active();
            });
    }

    public function orderLines()
    {
        return $this->salesOrderLines();
    }

    public function products()
    {
        return $this->belongsToMany(Product::class, 'sales_order_lines');
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    public function shippingMethod()
    {
        return $this->belongsTo(ShippingMethod::class);
    }

    public function salesOrderFulfillments()
    {
        $relation = $this->hasMany(SalesOrderFulfillment::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (SalesOrderFulfillment $salesOrderFulfillment) {
                $salesOrderFulfillment->delete();
            });
        });

        return $relation;
    }

    public function tags()
    {
        return $this->morphToMany(Tag::class, 'taggable')->withTimestamps();
    }

    public function adjustments()
    {
        return $this->morphMany(InventoryAdjustment::class, 'link');
    }

    public function store()
    {
        return $this->belongsTo(Store::class);
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    public function currencyTenantSnapshot(): BelongsTo
    {
        return $this->belongsTo(Currency::class, 'currency_id_tenant_snapshot');
    }

    public function parentLinks()
    {
        return $this->morphMany(OrderLink::class, 'child');
    }

    public function accountingTransaction()
    {
        return $this->morphOne(AccountingTransaction::class, 'link');
    }

    /**
     * Sets the status of the order.
     */
    public function setStatus(string $status): self
    {
        $this->order_status = $status;
        $this->save();

        return $this;
    }

    /**
     * Sets the payment status of the Sales Order.
     */
    public function setPaymentStatus($date = null): void
    {
        if ($this->is_fully_paid) {
            $this->fully_paid_at = $date ?: now();
            $this->payment_status = static::PAYMENT_STATUS_PAID;
        } else {
            $this->fully_paid_at = null;

            if ($this->payment_paid > 0 && $this->payment_refunded < 0) {
                if ($this->payment_paid === -$this->payment_refunded) {
                    $this->payment_status = static::PAYMENT_STATUS_REFUNDED;
                    $this->order_status = static::STATUS_CLOSED;
                } else {
                    $this->payment_status = static::PAYMENT_STATUS_PARTIALLY_REFUNDED;
                }
            } elseif ($this->total_paid == 0) {
                $this->payment_status = static::PAYMENT_STATUS_UNPAID;
            } else {
                $this->payment_status = static::PAYMENT_STATUS_PARTIALLY_PAID;
            }
        }

        $this->save();
    }

    public function childLinks()
    {
        $relation = $this->morphMany(OrderLink::class, 'parent');

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (OrderLink $orderLink) {
                $orderLink->delete();
                if ($orderLink->child) {
                    $orderLink->child->delete();
                }
            });
        });

        return $relation;
    }

    public function salesCredits()
    {
        $relation = $this->hasMany(SalesCredit::class);
        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (SalesCredit $salesCredit) {
                $salesCredit->delete();
            });
        });

        return $relation;
    }

    public function salesCreditReturns(): HasManyThrough
    {
        return $this->hasManyThrough(
            SalesCreditReturn::class,
            SalesCredit::class,
            'sales_order_id',
            'sales_credit_id',
            'id',
            'id'
        );
    }

    public function resendLinks()
    {
        return $this->childLinks()->where('child_type', self::class)->where('link_type', 'resend');
    }

    public function resendParentsLinks()
    {
        return $this->parentLinks()->where('parent_type', self::class)->where('link_type', 'resend');
    }

    public function purchaseOrdersLinks()
    {
        return $this->childLinks()->where('child_type', PurchaseOrder::class);
    }

    public function purchaseOrders()
    {
        $relation = $this->hasMany(PurchaseOrder::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (PurchaseOrder $purchaseOrder) {
                $purchaseOrder->delete();
            });
        });

        return $relation;
    }

    public function payments(): MorphMany
    {
        return $this->morphMany(Payment::class, 'link');
    }

    public function packingSlipQueues()
    {
        return $this->morphMany(PackingSlipQueue::class, 'link');
    }

    public function productAttributes()
    {
        return $this->hasManyThrough(ProductAttribute::class, SalesOrderLine::class, 'sales_order_id', 'product_id', 'id', 'product_id');
    }

    public function taxRate()
    {
        return $this->belongsTo(TaxRate::class);
    }

    public function salesCreditsParentsLinks()
    {
        return $this->parentLinks()->where('parent_type', SalesCredit::class);
    }

    public function financialLines(): HasMany
    {
        return $this->hasMany(FinancialLine::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Relationships with Sales Channel.
    |--------------------------------------------------------------------------
    |
    | Relationship with every Sales Channel type collection,
    | then will collect together with "raw_order" attribute.
    */
    public function salesChannelOrder(): MorphTo
    {
        return $this->morphTo('sales_channel_order');
    }

    public function ebayOrder()
    {
        return $this->hasOne(EbayOrder::class, 'sku_sales_order_id');
    }

    public function shopifyOrder()
    {
        return $this->hasOne(ShopifyOrder::class, 'sku_sales_order_id');
    }

    public function wooCommerceOrder()
    {
        return $this->hasOne(WooCommerceOrder::class, 'sku_sales_order_id');
    }

    public function magentoOrder()
    {
        return $this->hasOne(MagentoOrder::class);
    }

    public function getRawOrderAttribute()
    {
        if ($this->relationLoaded('salesChannel')) {
            $type = SalesChannelType::getTypeByID($this->salesChannel->sales_channel_type_id);

            switch ($type) {
                case SalesChannelType::TYPE_AMAZON:
                    return $this->amazonOrder->raw_order ?? null;
                case SalesChannelType::TYPE_EBAY:
                    return $this->ebayOrder->raw_order ?? null;
                case SalesChannelType::TYPE_SHOPIFY:
                    return $this->shopifyOrder->raw_order ?? null;
                case SalesChannelType::TYPE_WOOCOMMERCE:
                    return $this->wooCommerceOrder->raw_order ?? null;
            }
        }
    }

    public function getOrderDocumentAttribute()
    {
        $type = $this->sales_channel_id != SalesChannel::LOCAL_CHANNEL_ID ?
            $this->salesChannel->integrationInstance->integration->name :
            null;

        return match ($type) {
            Integration::NAME_SHOPIFY => $this->shopifyOrder,
            Integration::NAME_MAGENTO => $this->magentoOrder,
            default => $this->salesChannelOrder,
        };
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function mappedShippingMethod(): Attribute{
        return Attribute::get(function(){
            if(empty($this->requested_shipping_method)){
                return null;
            }
            /** @var ShippingMethodMappingsSalesChannelToSku|null $mapping */
            $mapping = ShippingMethodMappingsSalesChannelToSku::query()
                ->where('sales_channel_id', $this->sales_channel_id)
                ->where('sales_channel_method', $this->requested_shipping_method)
                ->first();
            if($mapping && $mapping->shippingMethod){
                $shippingMethod = $mapping->shippingMethod;
                return $shippingMethod?->full_name ?? $shippingMethod->shippingCarrier->name.' '.$shippingMethod->name;
            }
            return null;
        });
    }

    public function getNeedMappingAttribute()
    {
        if ($this->relationLoaded('salesOrderLines')) {
            return $this->salesOrderLines->where('product_id', null)->count() > 0;
        }

        return null;
    }

    public function getIntegrationInstanceNameAttribute()
    {
        if ($this->salesChannel->integrationInstance) {
            return $this->salesChannel->integrationInstance->name;
        }
    }

    public function getSalesChannelShippingServiceAttribute()
    {
        $type = $this->sales_channel_id != SalesChannel::LOCAL_CHANNEL_ID ?
            $this->salesChannel->integrationInstance->integration->name :
            null;
        switch ($type) {
            case Integration::NAME_AMAZON_US:
                if ($this->amazonOrder) {
                    return ShippingService::with([])->where('name', $this->amazonOrder->ShipServiceLevel)->first();
                }

                return null;
            default:
                return null;
        }
    }

    public function setStoreNameAttribute($value)
    {
        $this->store_id = Store::with([])->where('name', $value)->value('id');
    }

    public function getProductSubtotalAttribute()
    {
        $subtotal = 0;

        foreach ($this->salesOrderLines as $salesOrderLine) {
            $subtotal += $salesOrderLine->isWarehousedProduct() ? $salesOrderLine->subtotal : 0;
        }

        return $subtotal;
    }

    public function getTotalDiscountAttribute()
    {
        if ($this->relationLoaded('salesOrderLines') && $this->discount) {
            return $this->salesOrderLines->sum('total') * $this->discount / 100;
        }
    }

    public function getShippingCostAttribute()
    {
        return $this->salesOrderFulfillments()->sum('cost');
    }

    public function getTrackingNumbersAttribute()
    {
        return $this->salesOrderFulfillments()->pluck('tracking_number')->implode(', ');
    }

    public function getShippingAddressStringAttribute()
    {
        return $this->shippingAddress->formatForEmail();
    }

    public function getBillingAddressStringAttribute()
    {
        return $this->billingAddress->formatForEmail();
    }

    public function getTotalRevenueFormattedAttribute()
    {
        return $this->currency->code.' '.number_format($this->getTotalRevenue(), 2);
    }

    public function getAdditionalCostFormattedAttribute()
    {
        return $this->currency->code.' '.number_format($this->additional_cost, 2);
    }

    public function getProductSubtotalFormattedAttribute()
    {
        return $this->currency->code.' '.number_format($this->product_subtotal, 2);
    }

    public function getShippingCostFormattedAttribute()
    {
        return $this->currency->code.' '.number_format($this->shipping_cost, 2);
    }

    public function getStoreEmailAttribute()
    {
        if ($this->store) {
            return "<a href=\"{$this->store->email}\">{$this->store->email}</a>";
        }
    }

    public function getReceiveByDateFormattedAttribute()
    {
        return $this->deliver_by_date ?? 'accessed via tracking';
    }

    public function getFulfillmentRevenueAttribute()
    {
        return $this->salesOrderLines->sum('fulfillment_revenue');
    }

    public function getCalculatedTotalAttribute()
    {
        if ($this->relationLoaded('salesOrderLines')) {
            $total = $this->salesOrderLines->sum('subtotal') - $this->salesOrderLines->sum('discount_allocation');
        } else {
            $total = $this->total;
        }

        return round($total, 2);
    }

    public function getCalculatedDiscountTotalAttribute()
    {
        $rateDiscount = ($this->subtotal * (collect($this->discount_lines)->where('rate', '!=', null)->sum('rate') / 100));
        $totalDiscount = collect($this->discount_lines)->where('amount', '!=', null)->sum('amount') + $rateDiscount;

        return round($totalDiscount, 2);
    }

    public function getExactTaxTotalAttribute()
    {
        $totalTax = 0;
        $amount = $this->subtotal;
        if ($this->exact_discount_total && $this->exact_discount_total > 0) {
            $amount -= $this->exact_discount_total;
        }

        if ($this->tax_rate_id) {
            $totalTax = $this->getTaxAmount($this->is_tax_included, $amount, $this->taxRate->rate);
        } else {
            foreach ($this->salesOrderLines as $salesOrderLine) {
                if ($salesOrderLine->tax_rate_id) {
                    $totalTax += $this->getTaxAmount($this->is_tax_included, $salesOrderLine->subtotal, $salesOrderLine->taxRate->rate);
                }
            }
        }

        return $totalTax;
    }

    public function getExactDiscountTotalAttribute()
    {
        $rateDiscount = ($this->total * (collect($this->discount_lines)->where('rate', '!=', null)->sum('rate') / 100));
        $totalDiscount = collect($this->discount_lines)->where('amount', '!=', null)->sum('amount') + $rateDiscount;

        return $totalDiscount;
    }

    public function getProductUnitsAttribute()
    {
        return $this->salesOrderLines->sum('quantity');
    }

    public function getActiveStoreAttribute()
    {
        return $this->store ?: $this->salesChannel->store;
    }

    public function getActiveStoreIdAttribute()
    {
        return $this->store_id ?: $this->salesChannel->store_id;
    }

    public function getCustomerReferenceAttribute()
    {
        return $this->sales_order_number;
    }

    public function setCurrencyCodeAttribute($value)
    {
        $currency = cache()->remember('currency_for_code_' . $value, 60, function () use ($value) {
            return Currency::with([])->where('code', $value)->first();
        });
        $this->currency_id = $currency->id;
        $this->currency_rate = $currency->conversion;
    }

    public function setShippingMethodNameAttribute($value)
    {
        $this->shipping_method_id = empty($value) ? null : ShippingMethod::with([])->where('name', $value)->value('id');
    }

    public function getIsFullyFulfilledAttribute()
    {
        $this->loadMissing(['salesOrderFulfillments', 'salesOrderLines', 'salesOrderLines.salesOrderFulfillmentLines']);

        if($this->canceled_at){
            return true;
        }

        // all lines are non products
        if (! $this->salesOrderLines->firstWhere('is_product', true)) {
            return true;
        }

        // at least one fulfillment is not fulfilled, means it's not fully fulfilled
        if ($this->salesOrderFulfillments->firstWhere('status', '!=', SalesOrderFulfillment::STATUS_FULFILLED)) {
            return false;
        }
        // all lines fulfilled(have fulfillment lines)(through sku or externally)
        foreach ($this->salesOrderLines as $salesOrderLine) {
            if (! $salesOrderLine->fulfilled && $salesOrderLine->quantity > 0) {
                return false;
            }
        }

        return true;
    }

    public function getIsOverFulfilledAttribute()
    {
        $this->loadMissing(['salesOrderFulfillments', 'salesOrderLines', 'salesOrderLines.salesOrderFulfillmentLines']);

        return $this->salesOrderLines->sum('fulfilled_quantity') > $this->salesOrderLines->sum('quantity');
    }

    public function getIsPartiallyFulfilledAttribute()
    {
        $this->loadMissing(['salesOrderLines', 'salesOrderLines.salesOrderFulfillmentLines']);

        // not fulfilled and not awaiting tracking
        if ($this->is_fully_fulfilled || $this->is_awaiting_tracking) {
            return false;
        }

        foreach ($this->salesOrderLines as $salesOrderLine) {
            // at least one line has a fulfilled quantity or no_audit_trail
            if ($salesOrderLine->is_product && ($salesOrderLine->fulfilled_quantity > 0 || $salesOrderLine->no_audit_trail)) {
                return true;
            }
        }

        return false;
    }

    /**
     * @see SKU-3390
     */
    public function getIsAwaitingTrackingAttribute(): bool
    {
        if ($this->is_fully_fulfilled) {
            return false;
        }

        $this->loadMissing(['salesOrderFulfillments', 'purchaseOrders']);

        /** @see SKU-3391 */
        // for dropship, consider the sales order as awaiting tracking once dropship purchase order created(is not received)
        if ($this->purchaseOrders->firstWhere('receipt_status', '!=', PurchaseOrder::RECEIPT_STATUS_RECEIVED)) {
            return true;
        }
        // OR
        // at least has one fulfillment is not fulfilled
        if ($this->salesOrderFulfillments->whereNotIn('status', [SalesOrderFulfillment::STATUS_FULFILLED, SalesOrderFulfillment::STATUS_CANCELED])->isNotEmpty()) {
            return true;
        }

        return false;
    }

    public function getIsFullyFulfillableAttribute()
    {
        $this->loadMissing(['salesOrderFulfillments', 'salesOrderLines.salesOrderFulfillmentLines', 'salesOrderLines.backorderQueue']);

        if ($this->is_fully_fulfilled) {
            return false;
        }

        foreach ($this->salesOrderLines as $salesOrderLine) {
            if (! $salesOrderLine->fulfilled && ! $salesOrderLine->is_fully_fulfillable) {
                return false;
            }
        }

        return true;
    }

    public function getIsFulfillableAttribute()
    {
        $this->loadMissing(['salesOrderFulfillments', 'salesOrderLines.salesOrderFulfillmentLines', 'salesOrderLines.backorderQueue']);

        if ($this->is_fully_fulfilled || $this->isDraft()) {
            return false;
        }

        return $this->salesOrderLines->where('fulfillable_quantity', '>', 0)->isNotEmpty();
    }

    public function getProrationFulfilledAttribute(): float
    {
        if ($this->product_subtotal == 0) {
            return 0.00;
        }

        return $this->salesOrderLines->sum('fulfilled_quantity') / $this->product_subtotal;
    }

    /**
     * Gets the total amount paid for the sales order.
     */
    public function getTotalPaidAttribute(): float
    {
        $this->loadMissing('payments');

        return $this->payments()->sum('amount');
    }

    /**
     * Gets the amount paid for the sales order.
     */
    public function getPaymentPaidAttribute(): float
    {
        $this->loadMissing('payments');

        return $this->payments()->where('amount', '>', 0)->sum('amount');
    }

    /**
     * Gets the amount refunded for the sales order.
     */
    public function getPaymentRefundedAttribute(): float
    {
        $this->loadMissing('payments');

        return $this->payments()->where('amount', '<', 0)->sum('amount');
    }

    /**
     * Checks if the sales order has been fully paid.
     */
    public function getIsFullyPaidAttribute(): bool
    {
        $this->loadMissing('salesOrderLines', 'salesCredits');

        $total = $this->total;
        if ($this->exact_discount_total && $this->exact_discount_total > 0) {
            $total -= $this->exact_discount_total;
        }
        $totalExclludingSalesCredit = $total - $this->salesCredits->sum('total_credit');

        return round($this->total_paid, 2) >= round($totalExclludingSalesCredit, 2);
    }

    public function getAdditionalCostAttribute()
    {
        if ($this->relationLoaded('salesOrderLines')) {
            return $this->salesOrderLines->where('product_id', null)->sum('subtotal');
        }
    }

    public function getSalesResendAttribute()
    {
        return $this->resendLinks->map(function (OrderLink $item) {
            return $item->child;
        });
    }

    public function getParentSalesResendAttribute()
    {
        $link = $this->resendParentsLinks->first();

        if ($link) {
            return $link->parent;
        }
    }

    public function closed(): Attribute
    {
        return Attribute::get(fn () => $this->order_status === self::STATUS_CLOSED);
    }

    public function allProductsMapped(): bool
    {
        return $this->salesOrderLines()->where('is_product', true)->whereNull('product_id')->count() === 0;
    }

    public function allProductsWarehoused(): bool
    {
        return $this->salesOrderLines()
            ->where('is_product', true)
            ->where(function ($q) {
                return $q->whereNull('product_id')->orWhereNull('warehouse_id');
            })
            ->count() === 0;
    }

    public function nonProductLines(): Collection
    {
        return $this->salesOrderLines->where('is_product', false);
    }

    public function revertToDraft(): array|bool
    {
        if ($this->fulfillment_status === self::FULFILLMENT_STATUS_AWAITING_TRACKING) {
            return ["Sales orders awaiting tracking can't be converted to draft.  Try removing fulfillments first", 'SalesOrder'.Response::CODE_HAS_FULFILLMENTS, 'id'];
        }

        if ($this->order_status === self::STATUS_CLOSED) {
            return ['Closed orders cannot be reverted to draft.', 'SalesOrder'.Response::CODE_CLOSED, 'id'];
        }

        DB::transaction(function () {
            $this->reverseMovementsAndFifo();

            $this->clearRelatedRecords();

            $this->resetStatuses();

            $this->accountingTransaction?->delete();

            return $this->refresh();
        });

        return true;
    }

    protected function resetStatuses()
    {
        $this->payment_status = self::PAYMENT_STATUS_UNPAID;
        $this->fulfillment_status = self::FULFILLMENT_STATUS_UNFULFILLED;
        $this->fulfilled_at = null;
        $this->order_status = self::STATUS_DRAFT;
        $this->approved_at = null;
        $this->reserved_at = null;
        $this->canceled_at = null;
        $this->save();
    }

    protected function clearRelatedRecords()
    {
        $this->payments()->delete();
        $this->salesOrderFulfillments()->each(function (SalesOrderFulfillment $fulfillment) {
            $fulfillment->delete();
        });
        $this->salesCredits()->delete();
    }

    protected function reverseMovementsAndFifo()
    {
        $this->salesOrderLines
            ->where('is_product', true)
            ->each(
                /**
                 * @throws Exception
                 */
                function (SalesOrderLine $orderLine) {
                    SalesOrderLineLayer::with([])
                        ->where('sales_order_line_id', $orderLine->id)
                        ->where('layer_type', FifoLayer::class)
                        ->each(function (SalesOrderLineLayer $lineLayer) use ($orderLine) {
                            // Put back quantity on fifo layer
                            /** @var FifoLayer $layer */
                            if ($layer = $lineLayer->layer) {
                                $layer->fulfilled_quantity = max(0, $layer->fulfilled_quantity - $lineLayer->quantity);
                                $layer->save();

                                $orderLine->fifoLayers()->detach($layer->id);
                            }
                        });

                    // Remove inventory movements
                    $orderLine->inventoryMovements()->delete();

                    // Remove backorder queues
                    if ($orderLine->backorderQueue) {
                        $orderLine->backorderQueue->delete();
                    }
                }
            );
    }

    /**
     * Generate order barcode (base64 from PNG).
     */
    public function getBarcode($postfix = null): string
    {
        $generator = new BarcodeGeneratorPNG();

        return base64_encode($generator->getBarcode($this->customer_reference.$postfix, $generator::TYPE_CODE_128));
    }

    public function toShipStationOrder(bool $withItems = true)
    {
        $ssOrderPrefix = config('shipstation.order_prefix');

        $order = new \App\SDKs\ShipStation\Model\Order();
        $order->orderNumber = $this->fulfillment_number;
        $order->orderKey = $ssOrderPrefix.$this->id;
        $order->orderDate = $this->order_date->format('Y-m-d H:i:s');
        $order->paymentDate = $this->fully_paid_at ? $this->fully_paid_at->format('Y-m-d H:i:s') : null;
        $order->shipByDate = $this->ship_by_date ? $this->ship_by_date->format('Y-m-d H:i:s') : null;
        $order->orderStatus = $this->mapShiStationOrderStatus();
        $order->customerUsername = $this->customer->name;
        $order->customerEmail = $this->customer->email;
        $order->billTo = $this->billingAddress?->toShipStationAddress() ?? $this->shippingAddress->toShipStationAddress();
        $order->shipTo = $this->shippingAddress->toShipStationAddress();
        if ($withItems) {
            $order->items = $this->salesOrderLines->map->toShipStationOrderItem()->toArray();
        }
        $order->orderTotal = $this->getTotalRevenue(); // TODO: This should be the regular order total, not revenue

        return $order;
    }

    private function mapShiStationOrderStatus()
    {
        if ($this->order_status == self::STATUS_DRAFT) {
            return \App\SDKs\ShipStation\Model\Order::STATUS_AWAITING_PAYMENT;
        } elseif ($this->order_status == self::STATUS_OPEN) {
            return \App\SDKs\ShipStation\Model\Order::STATUS_AWAITING_SHIPMENT;
        } elseif ($this->order_status == self::STATUS_CLOSED) {
            return \App\SDKs\ShipStation\Model\Order::STATUS_SHIPPED;
        }
    }

    /**
     * Closes the sales order if it's without
     * an active backorder queue.
     */
    public function close(): bool
    {
        $hasActiveBackorders = $this->salesOrderLines()
            ->whereNotNull('product_id')
            ->whereHas('backorderQueue', function (Builder $builder) {
                return $builder->active();
            })->count() > 0;

        // if you want to cancel the order, call the cancel function first
        if ($hasActiveBackorders) {
            // We cannot close sales orders with active backorder queues.
            return false;
        }

        $productsMapped = $this->salesOrderLines()
            ->where('is_product', 1)
            ->whereNull('product_id')
            ->count() == 0;

        if (! $productsMapped) {
            // We cannot close a sales order that has unmapped lines.
            return false;
        }

        $this->order_status = self::STATUS_CLOSED;

        return $this->save();
    }

    public function cancel($cancelDate = null): self
    {
        // delete unfulfilled fulfillments
        $this->salesOrderFulfillments->where('status', '!=', SalesOrderFulfillment::STATUS_FULFILLED)->each->delete(true);

        // Changing fulfillment status to unfilfilled

        $this->fulfillment_status = self::FULFILLMENT_STATUS_UNFULFILLED;

        // Deleting accounting transaction
        $this->accountingTransaction?->delete();

        // We also need to clear inventory movements and backorders
        $this->warehousedProductLines()
            ->with('fulfilledLines')
            ->where(function (Builder $query) {
                return $query->whereNull('warehouse_routing_method')
                    ->orWhere('warehouse_routing_method', '!=', WarehouseRoutingMethod::DROPSHIP);
            })
            ->each(function (SalesOrderLine $salesOrderLine) {
                $usedQuantity = $salesOrderLine->active_backordered_quantity;

                if ($salesOrderLine->fifoLayers()->count()) {
                    /**
                     * The sales order line has used fifo layers,
                     * we put back the used quantities. This allows
                     * us to apply those fifo to other sales order lines.
                     */
                    $usedQuantity += SalesOrderLineLayer::with([])
                        ->where('sales_order_line_id', $salesOrderLine->id)
                        ->where('layer_type', FifoLayer::class)
                        ->sum('quantity');
                }

                $fulfilledQuantity = $salesOrderLine->fulfilledLines->sum('quantity');
                $usedQuantity = $usedQuantity - $fulfilledQuantity;

                if ($usedQuantity <= 0) {
                    return;
                }

                InventoryManager::with(
                    $salesOrderLine->warehouse_id,
                    $salesOrderLine->product
                )->decreaseNegativeEventQty($usedQuantity, $salesOrderLine);

                /**
                 * We clear any backorder queue and movements left behind.
                 */
                $salesOrderLine->backorderQueue?->delete();

                if ($fulfilledQuantity == 0) {
                    $salesOrderLine->inventoryMovements()->delete();

                    SalesOrderLineLayer::with([])
                        ->where('sales_order_line_id', $salesOrderLine->id)
                        ->delete();
                }

                // We re-calculate backorder queue coverages.
                dispatch(new SyncBackorderQueueCoveragesJob(null, null, [$salesOrderLine->product_id], $salesOrderLine->warehouse_id));
            });

        // mark as canceled
        $this->canceled_at = $cancelDate ?: now();
        if ($this->save()) {
            // Delete any attached accounting transaction
            $this->accountingTransaction?->delete();
            // Close the order.
            $this->close();
        }

        return $this;
    }

    public function toStarshipitOrder($withItems = true)
    {
        $ssOrderPrefix = config('shipstation.order_prefix');

        $this->load(['shippingAddress']);

        $order = new \App\SDKs\Starshipit\Model\Order();
        $order->order_date = $this->order_date->toISOString();
        $order->order_number = $ssOrderPrefix.$this->sales_order_number;
        $order->reference = $ssOrderPrefix.$this->id;
        $order->currency = $this->currency->code;
        //        $order->sender_details = $this->store->toStarshipitSenderDetails();
        $order->destination = $this->shippingAddress?->toStarshipitAddress();
        if ($withItems) {
            $order->items = $this->salesOrderLines->map->toStarshipitOrderItem()->toArray();
        }

        return $order;
    }

    public function isDraft(): bool
    {
        return $this->order_status == self::STATUS_DRAFT;
    }

    public function isReserved(): bool
    {
        return $this->order_status == self::STATUS_RESERVED;
    }

    public function isOpen(): bool
    {
        return $this->order_status == self::STATUS_OPEN;
    }

    public function open()
    {
        if ($this->isOpen()) {
            return $this;
        }
        $this->order_status = self::STATUS_OPEN;
        $this->save();

        return $this;
    }

    /**
     * To ShipStation Order.
     */
    public function toShipStationXmlModel(): Order
    {
        $order = new Order();
        $order->OrderID = $this->sales_order_number;
        $order->OrderNumber = $this->sales_order_number;
        $order->OrderDate = $this->created_at->format('m/d/Y h:m');
        $order->OrderStatus = 'paid';
        $order->LastModified = $this->created_at->format('m/d/Y h:m');
        $order->ShippingMethod = 'USPSPriorityMail';
        $order->PaymentMethod = 'PayPal';
        $order->OrderTotal = 125.1;
        $order->TaxAmount = 0.0;
        $order->ShippingAmount = 1.2;

        //    $items = new Items( [ 'Items' => [] ] );
        $order->Items = [];
        foreach ($this->salesOrderLines as $salesOrderLine) {
            $item = new Item();
            $item->SKU = 'APC-785';
            $item->Name = 'New PC';
            $item->Quantity = $salesOrderLine->quantity;
            $item->UnitPrice = 2.4;

            $order->Items[] = $item;
        }
        $customer = new \App\SDKs\ShipStation\Model\XML\Customer();
        $customer->CustomerCode = 'ahmed@m.com';
        $billTo = new BillTo();
        $billTo->Name = 'Ahmed';

        $customer->BillTo = $billTo;

        $shipTo = new ShipTo();
        $shipTo->Name = 'Ahmed';
        $shipTo->Address1 = 'Gaza';
        $shipTo->State = 'Gaza';
        $shipTo->City = 'Gaza';
        $shipTo->PostalCode = 'Gaza-845';
        $shipTo->Country = 'PS';

        $customer->ShipTo = $shipTo;

        $order->Customer = $customer;
        //    $order->Items = $items->Items;

        // TODO: here will convert this order to shipStation xml order.

        return $order;
    }

    public function getMatchingUnfulfilledSalesOrderLineFromSku($sku)
    {
        $product_id = Product::with([])->where('sku', $sku)->pluck('id')[0];

        return $this->salesOrderLines()->where('product_id', $product_id)
            ->where('no_audit_trail', 0)
            ->whereColumn('canceled_quantity', '<', 'quantity')
            ->whereDoesntHave('salesOrderFulfillmentLines')->first();
    }

    public function getMatchingUnfulfilledSalesOrderLineFromId($sales_channel_line_id)
    {
        return $this->salesOrderLines()->where('sales_channel_line_id', $sales_channel_line_id)->whereDoesntHave('salesOrderFulfillmentLines')->first();
    }

    /**
     * Get available columns to show in datatable.
     */
    public function availableColumns(): array
    {
        return config('data_table.sales_order.columns');
    }

    /**
     * Get filterable columns, must be inside available columns.
     */
    public function filterableColumns(): array
    {
        if (Str::startsWith(func_get_arg(0), [
            'customer_name.',
            'customer_id.',
            'tags.',
            'fulfillment_warehouse.name',
            'store.',
        ])) {
            return func_get_args();
        }

        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')
            ->add('customer_id')
            ->add('fulfillable')
            ->add('sourcing')
            ->add('backordered')
            ->add('covered_by_po')
            ->add('product_sku')
            ->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['item_sku', 'sales_order_number', 'order_status'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function save(array $options = [])
    {
        $this->load('salesOrderLines');

        // set default customer reference for new sales orders
        if (! $this->exists && empty($this->sales_order_number)) {
            $this->sales_order_number = self::getNextLocalNumber();
        }

        if (empty($this->order_date)) {
            $this->order_date = Carbon::now();
        }

        if (empty($this->currency_id)) {
            $this->currency_id = Currency::default()->id;
        }

        // set default shipping address from default customer shipping address
        if (empty($this->shipping_address_id)) {
            $this->shipping_address_id = $this->customer ? $this->customer->default_shipping_address_id : null;
        }

        // set default billing address from default customer billing address
        if (empty($this->billing_address_id)) {
            $this->billing_address_id = $this->customer ? $this->customer->default_billing_address_id : null;
        }

        /* @see SKU-3466 As a failsafe make sure sales orders don't get created if there is no store id */
        if (empty($this->store_id)) {
            if ($this->salesChannel?->store_id) {
                $this->store_id = $this->salesChannel->store_id;
            } elseif ($defaultSOStore = Helpers::setting(Setting::KEY_SO_DEFAULT_STORE)) {
                $this->store_id = $defaultSOStore;
            } else {
                throw new InvalidArgumentException('The sales order does not belong to a store');
            }
        }

        if (isset($options['upsert']) && $options['upsert']) {
            return $this->upsertModel(['sales_channel_id', 'sales_order_number'], []);
        }
        $this->cacheCurrencyRate($this->isDirty(['currency_id']));

        // update tax allocation for lines  when update order
        if ($this->isDirty(['is_tax_included'])) {
            $this->salesOrderLines()->each(function (SalesOrderLine $line) {
                $line->updateTaxAllocation();
            });
        }

        return parent::save($options);
    }

    public function delete()
    {
        /*
         * Currently deletion take a very long time, potentially because of the InvalidateFinancialReportingCacheObserver
         *
         * We are forcing bulk method here due to lockout issues.
         */
        set_time_limit(0);
        /*
         * Commenting this out due to test failing (Kalvin)
         */
        //        app(BulkSalesOrderRepository::class)->bulkDelete(Arr::wrap($this->id));
        //        return true;
        $this->load([
            'salesOrderLines',
            'purchaseOrders',
            'purchaseOrders.purchaseOrderLines',
            'purchaseOrdersLinks',
            'salesOrderFulfillments',
            'salesOrderFulfillments.salesOrderFulfillmentLines',
            'salesCredits',
            'salesCredits.salesCreditLines',
            'salesCredits.salesCreditReturns',
            'salesCredits.salesCreditReturns.salesCreditReturnLines',
            'salesCredits.payments',
            'shippingAddress',
            'billingAddress',
            'accountingTransaction',
        ]);

        $delete = false;

        DB::transaction(function () use (&$delete) {
            $this->salesOrderFulfillments()->delete();
            $this->childLinks()->delete();

            $this->parentLinks()->delete();
            $this->payments()->delete();
            $this->salesCredits()->each(function (SalesCredit $saleCredit) {
                $saleCredit->delete();
            });

            $this->salesOrderLines->each(function (SalesOrderLine $orderLine) {
                $orderLine->setRelation('salesOrder', $this);
                // This approach ensures that fifo and movements are reversed
                // before the line is deleted.
                $orderLine->delete();
            });
            $this->financialLines->each(function (FinancialLine $financialLine) {
                $financialLine->delete();
            });

            //$this->accountingTransaction()->delete();
            $this->syncTags([]); // detaches all tags

            if ($this->order_document) {
                // TODO: after replacing mngodb, I think we should add this function (skuOrderDeleted) to the models
                if (method_exists($this->order_document, 'skuOrderDeleted')) {
                    $this->order_document->skuOrderDeleted();
                } else {
                    if (isset($this->order_document['sku_sales_order_id']))
                    {
                        $this->order_document['sku_sales_order_id'] = null;
                    }
                    if ($this->shopifyOrder) {
                        app(ShopifyOrderMappingRepository::class)->deleteMappingsForShopifyOrder($this->order_document);
                    } elseif ($this->magentoOrder) {
                        $this->order_document['fulfillmentsMap'] = null;
                        $this->order_document['refundsMap'] = null;
                    }
                    $this->order_document->save();

                    $this->order_document->archive(); /** @see SKU-4130 */
                }
            }

            $delete = parent::delete();

            if ($delete) {
                if ($this->shippingAddress && ! $this->shippingAddress->usedByCustomers()) {
                    $this->shippingAddress->delete();
                }
                // Refresh the billing address to ensure that it's not
                // already deleted from shipping address (Cases where they have the same id)
                $this->load('billingAddress');
                if ($this->billingAddress && ! $this->billingAddress->usedByCustomers()) {
                    $this->billingAddress->delete();
                }
            }
        });

        return $delete;
    }

    /**
     * Determine if the purchase order is used.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        $relations = ['purchaseOrders', 'salesCredits'];

        $this->loadCount($relations);

        $usage = [];

        foreach ($relations as $relatedRelation) {
            $countKeyName = Str::snake($relatedRelation).'_count';
            if ($this->{$countKeyName}) {
                $relatedName = Str::singular(str_replace('_', ' ', Str::snake($relatedRelation)));

                if ($relatedName == 'sales credits link') {
                    $relatedName = $this->{$countKeyName} === 1 ? 'Sales Credit' : 'Sales Credits';
                }

                $usage[$relatedRelation] = trans_choice('messages.currently_used', $this->{$countKeyName}, [
                    'resource' => $relatedName,
                    'model' => 'sales order('.$this->sales_order_number.')',
                ]);
            }
        }

        return count($usage) ? $usage : false;
    }

    /**
     * Approve sales order (from Draft to Open status).
     */
    public function approve(int $approveDropship = 0, bool $onlyForSufficientStock = false): bool
    {
        $response = ApproveSalesOrderService::make($this)->approve($approveDropship, $onlyForSufficientStock);

        return $response;
    }

    public function reserve(int $approveDropship = 0, bool $onlyForSufficientStock = false)
    {
        return ApproveSalesOrderService::make($this)->reserve($approveDropship, $onlyForSufficientStock);
    }

    /**
     * Fully fulfill Closed Sales Order.
     *
     *
     * @throws Throwable
     */
    public function fullyFulfill(?string $trackingNumber = null, $submitTrackingToSalesChannel = false): void
    {
        FulfillSalesOrderService::make($this)->fullyFulfill($trackingNumber, $submitTrackingToSalesChannel);
    }

    public function updateFulfillmentStatus(?Carbon $fulfilledDate = null, bool $saveNow = true)
    {
        $this->load([
            'salesOrderFulfillments',
            'salesOrderLines',
            'salesOrderLines.salesOrderFulfillmentLines',
        ]);

        if ($this->is_fully_fulfilled) {
            $this->close();
            $this->fulfillment_status = self::FULFILLMENT_STATUS_FULFILLED;
            $this->fulfilled_at = $fulfilledDate ?: now();
        } elseif ($this->is_over_fulfilled) {
            $this->fulfillment_status = self::FULFILLMENT_STATUS_OVER_FULFILLED;
            $this->close();
        } elseif ($this->is_awaiting_tracking) {
            $this->fulfillment_status = self::FULFILLMENT_STATUS_AWAITING_TRACKING;
            $this->order_status = self::STATUS_OPEN;
        } elseif ($this->is_partially_fulfilled) {
            $this->fulfillment_status = self::FULFILLMENT_STATUS_PARTIALLY_FULFILLED;
            $this->order_status = $this->canceled_at ? self::STATUS_CLOSED : self::STATUS_OPEN;
        } else {
            $this->order_status = $this->canceled_at ? self::STATUS_CLOSED : self::STATUS_OPEN;
            $this->fulfillment_status = self::FULFILLMENT_STATUS_UNFULFILLED;
            $this->fulfilled_at = null;
        }

        if ($saveNow) {
            $this->save();
        }
    }

    /**
     * Determined if sales order have completed data for approved it.
     *
     * @return array|bool
     */
    public function isComplete()
    {
        // This property for validation only
        $this->load('salesOrderLines');
        $this->sales_order_lines = $this->salesOrderLines->toArray();

        $completeValidator = Validator::make($this->attributesToArray(), [
            'sales_channel_id' => 'required',
            'sales_order_number' => 'required',
            'order_status' => 'required',
            'customer_id' => 'required',
            'shipping_address_id' => 'required',
            'billing_address_id' => 'required',
            'sales_order_lines' => 'required|array|min:1',
        ]);

        // remove from its properties(attributes)
        unset($this->sales_order_lines);

        if ($completeValidator->fails()) {
            return $completeValidator->errors()->toArray();
        }

        return true;
    }

    public function getShippingMethodNameAttribute()
    {
        if ($this->relationLoaded('shippingMethod') && $this->shippingMethod) {
            return $this->shippingMethod->full_name;
        }

        return $this->requested_shipping_method;

        /** @deprecated now store in requested_shipping_method column */

        // For AFN orders, we revert to the original shipping method on
        // the amazon order.
        if ($this->fulfillment_channel === Amazon::FULFILLMENT_CHANNEL_AFN && $this->salesChannel->integrationInstance->isAmazonInstance()) {
            $order = AmazonOrder::with([])->where('AmazonOrderId', $this->sales_order_number)
                ->first();
            if ($order) {
                return $order['ShipServiceLevel'];
            }
        }

        return null;
    }

    public function setCustomer(?array $customer, bool $save = false, ?int $salesChannelId = SalesChannel::LOCAL_CHANNEL_ID)
    {
        if (empty($customer) || empty(array_filter($customer)) || is_null(@$customer['name'])) {
            return null;
        }

        // Can pass explicit customer_id, in which case all additional customer details passed are ignored.
        // Should show warning if additional customer details passed informing client that they’ve been ignored
        if ($this->isDirty('customer_id') && $this->customer_id) {
            Response::instance()->addWarning(__('messages.sales_order.ignore_address_details', ['address' => 'customer']), Response::CODE_IGNORE_DETAILS, 'customer', $customer);

            return null;
        }

        // Customers are required.  If an exact customer id is not provided, it tries to find the id using the rules outlined
        $existingCustomer = Customer::exists($customer);

        if ($existingCustomer) {
            $this->customer_id = $existingCustomer->id;
            $existingCustomer->sales_channel_origin_id = $salesChannelId;
            $existingCustomer->save();
        } else {
            $address = Address::where(collect($customer)->only([
                'name',
                'email',
                'phone',
                'address1',
                'address2',
                'address3',
                'city',
                'province',
                'province_code',
                'zip',
                'country_code',
                'country',
            ])->toArray())->first();

            if (is_null($address)) {
                $address = new Address($customer);
                $address->label = 'Default Address';
                $address->save();
            }
            $customerData = $customer;
            $customer = ($this->customer) ? $this->customer : new Customer($customerData);
            $customer->fill($customerData);
            $customer->shippingAddress()->associate($address);
            $customer->billingAddress()->associate($address);
            $customer->sales_channel_origin_id = $salesChannelId;
            $customer->save();

            $this->customer_id = $customer->id;
        }

        $this->load('customer');

        if ($save) {
            return $this->save();
        }

        return true;
    }

    public function deleteSubmittedFulfillmentsWithoutPackingSlipPrinted()
    {
        $salesOrderFulfillments = $this->salesOrderFulfillments()
            ->where('status', SalesOrderFulfillment::STATUS_SUBMITTED)
            ->whereNull('packing_slip_printed_at')
            ->get();
        // Need to delete any submitted fulfillments and tag the sales order.
        if ($salesOrderFulfillments->count() > 0) {
            $fulfillmentDeletionError = null;
            //Log::debug($salesOrderFulfillments->count() . ' fulfillments to delete');
            $salesOrderFulfillments->each(function (SalesOrderFulfillment $fulfillment) use (&$fulfillmentDeletionError) {
                //Log::debug('Deleting fulfillment ' . $fulfillment->id);
                try {
                    $fulfillment->delete(false);
                } catch (Exception $e) {
                    //Log::error('Error deleting fulfillment ' . $fulfillment->id . ': ' . $e->getMessage());
                    $fulfillmentDeletionError = $e->getMessage();
                }
            });

            $note = $fulfillmentDeletionError ?
                'We tried to delete the fulfillment from the shipping provider due to a change in the sales order lines, which failed with the following error message from the shipping provider: '.$fulfillmentDeletionError
                :
                'There were submitted fulfillments that were removed from shipping provider due to a change in the sales order lines.  So we marked the fulfillment status as Out of Sync.';

            $this->notes()->create([
                'note' => $note,
            ]);
            customlog('out-of-sync', $this->sales_order_number.' had fulfillments that were submitted (with no packing slip printed) deleted, so marking out of sync');
            $this->fulfillment_status = self::FULFILLMENT_STATUS_OUT_OF_SYNC;
            $this->save();
        }
    }

    public function setShippingAddress(?array $address, bool $save = false)
    {
        if (empty($address)) {
            return null;
        }
        if ($this->isDirty('shipping_address_id') && $this->shipping_address_id) {
            Response::instance()->addWarning(__('messages.sales_order.ignore_address_details', ['address' => 'shipping address']), Response::CODE_IGNORE_DETAILS, 'shipping_address', $address);

            return null;
        }
        $orginalShippingAddressId = $this->shipping_address_id;
        $this->shipping_address_id = $this->customer ? $this->customer->appendAddress($address)->id : null;
        $ShippingAddressChanged = $orginalShippingAddressId !== $this->shipping_address_id;

        // Need to delete any submitted fulfillments and tag the sales order.
        if ($ShippingAddressChanged) {
            $this->deleteSubmittedFulfillmentsWithoutPackingSlipPrinted();
        }

        if ($save) {
            return $this->save();
        }

        return true;
    }

    public function setBillingAddress(?array $address, bool $save = false)
    {
        if (empty($address)) {
            return null;
        }

        if ($this->isDirty('billing_address_id') && $this->billing_address_id) {
            Response::instance()->addWarning(__('messages.sales_order.ignore_address_details', ['address' => 'shipping address']), Response::CODE_IGNORE_DETAILS, 'shipping_address', $address);

            return null;
        }

        $this->billing_address_id = $this->customer->appendAddress($address)->id;

        if ($save) {
            return $this->save();
        }

        return true;
    }

    /**
     * @throws BindingResolutionException
     * @throws Exception
     */
    public static function splitBundleComponents($salesOrderLines, self $salesOrder)
    {
        $warehouses = app()->make(WarehouseRepository::class);

        $newSalesOrderLines = [];
        foreach ($salesOrderLines as $index => $salesOrderLine) {
            if (! isset($salesOrderLine['product_id'])) {
                continue;
            }
            $product = Product::find($salesOrderLine['product_id']);
            if ($product->type === Product::TYPE_BUNDLE) {
                // Delete the bundle product from the sales order lines
                unset($salesOrderLines[$index]);

                $existingComponents = $salesOrder->salesOrderLines()
                    ->where('bundle_id', $product->id)
                    ->get();

                // Disabling for now since cancel quantity is not being accounted for properly
                if ($existingComponents->count() > 0 && false) {
                    $salesOrderLinesForBundle = $existingComponents->map(function (SalesOrderLine $line) use ($salesOrderLine, $product) {
                        $quantityMultiplier = $line->bundle_quantity_cache > 0
                            ? $salesOrderLine['quantity'] / $line->bundle_quantity_cache
                            : 1;
                        $line->quantity = $line->quantity * $quantityMultiplier;
//                        $line->quantity_to_cancel = (@$salesOrderLine['quantity_to_cancel'] ?? 0) * $quantityMultiplier;
                        return $line;
                    })->toArray();


                    $newSalesOrderLines = array_merge($newSalesOrderLines, $salesOrderLinesForBundle);
                } else {
                    // Replace bundle with kit if kit is in stock and bundle is not.
                    if ($product->kit instanceof Product) {
                        $isHaveStock = $warehouses->getPriorityWarehouseIdForProduct($product->kit,
                            $salesOrderLine['quantity'], false);
                        if (! empty($isHaveStock)) {
                            $newSalesOrderLines[] = ['product_id' => $product->kit->id] + $salesOrderLine;

                            continue;
                        }
                    }

                    $bundleComponents = $product->components;
                    foreach ($bundleComponents as $bundleComponent) {
                        $component = $bundleComponent;
                        $quantity = $bundleComponent->pivot->quantity * $salesOrderLine['quantity'];
                        $externallyFulfilledQuantity = $bundleComponent->pivot->quantity * ($salesOrderLine['externally_fulfilled_quantity'] ?? 0);
                        $line = [
                            'product_id' => $component->id,
                            'sales_channel_line_id' => @$salesOrderLine['sales_channel_line_id'],
                            'product_listing_id' => @$salesOrderLine['product_listing_id'],
                            'bundle_id' => $product->id,
                            'bundle_quantity_cache' => @$salesOrderLine['quantity'],
                            'description' => $component->name ?? $component->sku,
                            'quantity' => $quantity,
                            'externally_fulfilled_quantity' => $externallyFulfilledQuantity,
                            'warehouse_routing_method' => @$salesOrderLine['warehouse_routing_method'],
                            'warehouse_id' => @$salesOrderLine['warehouse_id'],
                            'amount' => $salesOrderLine['amount'] * $component->getBundlePriceProration($product),
                            'quantity_to_cancel' => (@$salesOrderLine['quantity_to_cancel'] ?? 0) * $bundleComponent->pivot->quantity,
                        ];

                        // Add in sales order line id if the component already exists on the sales order.
                        /** @var SalesOrderLine|null $existingLine */
                        $existingLineQuery = $salesOrder->salesOrderLines()
                            ->where('product_id', $component->id);

                        if (! empty($line['sales_channel_line_id'])) {
                            $existingLineQuery->where('sales_channel_line_id', $line['sales_channel_line_id']);
                        }

                        $existingLine = $existingLineQuery->first();

                        if ($existingLine) {
                            $line['id'] = $existingLine->id;
                        } else {
                            // Cache the component quantity
                            $line['bundle_component_quantity_cache'] = $bundleComponent->pivot->quantity;
                        }

                        $newSalesOrderLines[] = $line;
                    }
                }
            }
        }

        return array_merge($salesOrderLines, $newSalesOrderLines);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function setSalesOrderLines($salesOrderLines, $checkExistByDetails = false, bool|array $sync = true): ?bool
    {
        $salesOrderLinesChanged = false;

        if ($salesOrderLines === false) {
            return null;
        }
        $existingSalesOrderLines = [];
        $salesOrderLines = $this->splitBundleComponents($salesOrderLines, $this);

        foreach ($salesOrderLines ?: [] as $orderLine) {
            $existingCanceledQuantity = 0;
            if (! empty($orderLine['id'])) {
                $salesOrderLine = SalesOrderLine::with([])->findOrFail($orderLine['id']);
                if (isset($orderLine['product_id']) && ! $orderLine['product_id']) {
                    $orderLine['product_id'] = $salesOrderLine->product_id;
                }

                if(!empty($orderLine['bundle_id']) && !empty($orderLine['bundle_quantity_cache'])){
                    // This is a bundle line.
                    // We check if the bundle quantity has changed on the order,
                    // then we mark the order as lines changed.
                    /** @var SalesOrderLine|null $matchingLine */
                    $matchingLine = $this->salesOrderLines->where('id', $orderLine['id'])->first();
                    if(
                        $matchingLine && !empty($matchingLine->bundle_quantity_cache) &&
                        $matchingLine->bundle_quantity_cache != $orderLine['bundle_quantity_cache']){
                        $salesOrderLinesChanged = true;
                    }
                }

            } elseif ($checkExistByDetails) {
                $salesOrderLine = $this->salesOrderLines()->whereDetails($orderLine);
                if ($salesOrderLine->count() > 1 || $salesOrderLine->first()?->bundle) {
                    // Likely a bundle, so can't find by details
                    $salesOrderLine = null;
                } else {
                    $salesOrderLine = $salesOrderLine->first();
                }
                $salesOrderLinesChanged = true;
            } else {
                if (isset($orderLine['is_product']) && $orderLine['is_product']) {
                    $salesOrderLinesChanged = true;
                }
                $salesOrderLine = null;
            }

            if (! isset($salesOrderLine) || (isset($orderLine['product_id']) && $salesOrderLine->product_id != $orderLine['product_id'])) {
                // Create new line if products are different. This covers bundle components as well.
                $salesOrderLine = new SalesOrderLine();
            } else {
                $existingCanceledQuantity = $salesOrderLine->canceled_quantity;
                if (($salesOrderLine->quantity + $salesOrderLine->canceled_quantity) != @$orderLine['quantity'] && (isset($orderLine['is_product']) && $orderLine['is_product'])) {
                    $salesOrderLinesChanged = true;
                }
            }

            // Cannot change line warehouse if it has fulfillments
            $hasFulfilledLines = $salesOrderLine->salesOrderFulfillmentLines->count() > 0;
            if (! empty($orderLine['warehouse_id']) && $salesOrderLine->warehouse_id && $salesOrderLine->warehouse_id != $orderLine['warehouse_id'] && $hasFulfilledLines) {
                throw new LineFulfilledAtWarehouseException($salesOrderLine->product, "Cannot change warehouse for sku: {$salesOrderLine->product->sku}, line has fulfillments.");
            }

            // with this line, quantity gets reset to gross, not including cancellations, so we have to net it out again.
            $salesOrderLine->fill($orderLine);

            if (isset($orderLine['quantity_to_cancel']) && $orderLine['quantity_to_cancel'] > 0) {
                if ($existingCanceledQuantity != $orderLine['quantity_to_cancel']) {
                    $salesOrderLinesChanged = true;
                }
                if (! $salesOrderLine->sales_order_id) {
                    $salesOrderLine->sales_order_id = $this->id;
                }
                $salesOrderLine->updateCanceledQuantity($orderLine['quantity_to_cancel'], $orderLine['quantity'], $existingCanceledQuantity);
            } elseif (isset($orderLine['quantity_to_cancel']) && $orderLine['quantity_to_cancel'] == 0 && $existingCanceledQuantity > 0) {
                $salesOrderLine->canceled_quantity = 0;
            }

            if (! empty($orderLine['tax_rate_id'])) {
                $taxRate = TaxRate::with([])->findOrFail($orderLine['tax_rate_id']);
                $salesOrderLine->tax_rate_id = $taxRate->id;
                // If tax allocation is specified, we want to use that explicit amount, otherwise calculate it
                $salesOrderLine->tax_allocation = $orderLine['tax_allocation'] ?? $this->getTaxAmount($this->is_tax_included, $salesOrderLine->subtotal, $taxRate->rate);
            }

            // This is delegated to SetLineWarehouse in SalesOrderManager::prepareOrder()
            //            // set warehouse_id
            //            if (empty($salesOrderLine->warehouse_id) && $salesOrderLine->product && $salesOrderLine->warehouse_routing_method !== WarehouseRoutingMethod::DROPSHIP) {
            //                $warehouses = app()->make(WarehouseRepository::class);
            //                $salesOrderLine->warehouse_id = $warehouses->getPriorityWarehouseIdForProduct($salesOrderLine->product,
            //                    $salesOrderLine->quantity, true, $this->shippingAddress);
            //            }

            $salesOrderLine->sales_order_id = $this->id;
            $salesOrderLine->save();

            $existingSalesOrderLines[] = $salesOrderLine->id;
            unset($salesOrderLine);
        }

        // sync sales order lines
        if ($sync) {
            $salesOrderLinesToDelete = $this->salesOrderLines()->when(is_array($sync), fn ($q) => $q->where($sync))->whereNotIn('id', $existingSalesOrderLines);
            if ($salesOrderLinesToDelete->count() > 0) {
                $salesOrderLinesChanged = true;
                if ($salesOrderLinesToDelete->clone()->whereHas('salesOrderFulfillmentLines')->count() > 0) {
                    throw new Exception('Cannot delete sales order lines with fulfillments for ' . $this->sales_order_number);
                }
                $salesOrderLinesToDelete->delete();
            }
        }

        // Need to delete any submitted fulfillments and tag the sales order.
        if ($salesOrderLinesChanged && $this->salesChannel->integrationInstance->integration->name != Integration::NAME_SKU_IO) {
            $this->deleteSubmittedFulfillmentsWithoutPackingSlipPrinted();
        }

        return true;
    }

    /**
     * @throws Exception
     */
    private function setLineTaxAllocation(SalesOrderLine $salesOrderLine, int $taxRateId): SalesOrderLine
    {
        /** @var TaxRate $taxRate */
        $taxRate = TaxRate::with([])->findOrFail($taxRateId);
        $salesOrderLine->tax_rate_id = $taxRate->id;
        $salesOrderLine->tax_allocation = $this->getTaxAmount($this->is_tax_included, $salesOrderLine->subtotal, $salesOrderLine->taxRate->rate);
        $salesOrderLine->save();

        return $salesOrderLine;
    }

    public function addSalesCredit(SalesCredit $salesCredit)
    {
        // set sales credit values from sales order values
        $salesCredit->customer_id = $this->customer_id;
        $salesCredit->from_address_id = $this->shipping_address_id;
        $salesCredit->store_id = $this->active_store_id;
        $salesCredit->currency_id = $this->currency_id;
        $salesCredit->sales_order_id = $this->id;
        $salesCredit->is_tax_included = $this->is_tax_included;

        if ($this->tax_rate_id) {
            $salesCredit->tax_rate_id = $this->tax_rate_id;
        }

        $salesCredit->save();
    }

    /**
     * Get next customer reference for local channel (sku.io).
     */
    public static function getNextLocalNumber(): string
    {
        $prefix = Helpers::setting(Setting::KEY_SO_PREFIX, 'SO-');
        $numOfDigits = Helpers::setting(Setting::KEY_SO_NUM_OF_DIGITS, 5);
        $startFrom = Helpers::setting(Setting::KEY_SO_START_NUMBER, 1);

        $lastLocalSalesOrder = self::with([])->where('sales_channel_id', SalesChannel::LOCAL_CHANNEL_ID)
            ->where('sales_order_number', 'like', "{$prefix}%")
            ->latest('id')
            ->first();
        if ($lastLocalSalesOrder) {
            $lastNumber = (int) explode($prefix, $lastLocalSalesOrder->sales_order_number)[1] ?? null;
            $nextNumber = $lastNumber ? ($lastNumber + 1) : $startFrom;
        } else {
            $nextNumber = $startFrom;
        }

        return sprintf("{$prefix}%0{$numOfDigits}d", ($nextNumber < $startFrom ? $startFrom : $nextNumber));
    }

    public function scopeFilterTaxRevenue(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        return $builder->{$function.'Raw'}("(SELECT SUM(tax_allocation) FROM sales_order_lines WHERE sales_order_id = sales_orders.id) $operator $value");
    }

    public function scopeFilterSalesChannel(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $keyExploded = explode('.', $relation['combined_key']);

        $function = $conjunction == 'and' ? 'where' : 'orWhere';
        $relation = implode('.', array_slice($keyExploded, 0, count($keyExploded) - 1));
        $lastKey = array_slice($keyExploded, -1)[0];

        return $builder->{$function.'Has'}($relation, function (Builder $builder) use ($lastKey, $value, $operator) {
            $builder->filterKey([
                'key' => $builder->qualifyColumn($lastKey),
                'is_relation' => false,
            ], $operator, $value);
        });
    }

    public function scopeAccountingReady(Builder $builder): Builder
    {
        return $builder->mappingNeeded(false)
            ->whereNull('canceled_at')
            ->where('order_status', '!=', SalesOrder::STATUS_DRAFT)
            ->whereDoesntHave('salesOrderLines', function (Builder $query) {
                $query->whereDoesntHave('salesOrderLineFinancial');
            })
            ->whereDoesntHave('financialLines', function (Builder $query) {
                $query->whereHas('financialLineType', function (Builder $query) {
                    $query->where('classification', FinancialLineClassificationEnum::REVENUE);
                });
                $query->whereDoesntHave('nominalCode');
            });
    }

    public function scopeMappingNeeded(Builder $builder, bool $value = true)
    {
        if ($value) {
            return $builder->whereHas('salesOrderLines', function (Builder $query) {
                $query->whereNull('product_id')->where('is_product', 1);
            });
        } else {
            return $builder->whereHas('salesOrderLines', function (Builder $query) {
                $query->whereNull('product_id')->where('is_product', 1);
            }, '=', 0);
        }
    }

    public function scopeFilterMappingNeeded(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        if ($value) {
            return $builder->whereHas('salesOrderLines', function (Builder $query) use ($conjunction) {
                $query->whereNull('product_id', $conjunction)->where('is_product', '=', 1, $conjunction);
            });
        } else {
            return $builder->whereHas('salesOrderLines', function (Builder $query) use ($conjunction) {
                $query->whereNull('product_id', $conjunction)->where('is_product', '=', 1, $conjunction);
            }, '=', 0);
        }
    }

    public function scopeFilterSalesOrderLines(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        return $this->scopeFilterSalesChannel($builder, $relation, $operator, $value, $conjunction);
    }

    public function scopeFilterStore(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $key = $value ? $relation['key'] : 'id';

        if ($key == 'id') {
            $builder->where(function (Builder $builder) use ($operator, $value) {
                $builder->filterKey('store_id', $operator, $value)
                    ->orWhereHas('salesChannel', function (Builder $builder) use ($operator, $value) {
                        $builder->filterKey('store_id', $operator, $value);
                    });
            }, null, null, $conjunction);
        } else {
            $builder->where(function (Builder $builder) use ($operator, $value) {
                $builder->whereHas('store', function (Builder $builder) use ($operator, $value) {
                    $builder->filterKey('name', $operator, $value);
                })/*->orWhereHas('salesChannel.store', function (Builder $builder) use ($operator, $value) {
          $builder->filterKey('name', $operator, $value);
        })*/;
            }, null, null, $conjunction);
        }
    }

    public function scopeFilterFulfillable(Builder $builder, array $relation = [], string $operator = self::FULFILLABLE_STATUS_ALL, $value = ['mode' => 'all'], $conjunction = 'and'): Builder
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        $operator = strtolower($operator);
        $value = is_string($value) ? json_decode($value, true) : $value;

        if (! in_array($operator, self::FULFILLABLE_STATUSES)) {
            return $builder;
        }

        // The orders should be open and not fulfilled.
        $builder
            ->where('order_status', self::STATUS_OPEN)
            ->where(function ($query) {
                $query->whereIn('fulfillment_status', [self::FULFILLMENT_STATUS_UNFULFILLED, self::FULFILLMENT_STATUS_PARTIALLY_FULFILLED, self::FULFILLMENT_STATUS_AWAITING_TRACKING])
                    ->orWhereNull('fulfillment_status');
            });

        return $builder->{$function}(function (Builder $builder) use ($operator, $value) {
            return match ($operator) {
                self::FULFILLABLE_STATUS_ALL => $this->bindAllFulfillableFilter($builder, $value),
                self::FULFILLABLE_STATUS_SOME => $this->bindSomeFulfillableFilter($builder, $value),
                self::FULFILLABLE_STATUS_NONE => $this->bindNoneFulfillableFilter($builder, $value),
            };
        });
    }

    private function bindAllFulfillableFilter(Builder $builder, $value): Builder
    {
        // 1. Must be open and fulfillable (This is already handled in builder)
        // 2. Must have at least 1 fully fulfillable line
        // 3. Must have no active backordered line
        $builder
            ->whereIn('id', $this->atLeastOneFulfillableLineQuery())
            ->whereNotIn('id', $this->atLeastOneActiveBackorderQueueQuery());

        return $this->bindBackorderedFilter($builder, $value['mode'], $value['po_number'] ?? null);
    }

    private function activeBackorderQueueQuery(): Closure
    {
        return function (\Illuminate\Database\Query\Builder $q) {
            $q->select('id')->from('backorder_queues', 'bq')
                ->whereColumn('bq.sales_order_line_id', 'sol.id')
                ->where(function (\Illuminate\Database\Query\Builder $q) {
                    return $q->whereNotNull('bq.priority')
                        ->whereColumn('bq.backordered_quantity', '!=', 'bq.released_quantity');
                });
        };
    }

    private function coveredActiveBackorderQueueQuery(): Closure
    {
        return function (\Illuminate\Database\Query\Builder $q) {
            $q->select('id')->from('backorder_queues', 'bq')
                ->whereColumn('bq.sales_order_line_id', 'sol.id')
                ->where(function (\Illuminate\Database\Query\Builder $q) {
                    return $q->whereNotNull('bq.priority')
                        ->whereColumn('bq.backordered_quantity', '!=', 'bq.released_quantity');
                })
                ->whereExists(function (\Illuminate\Database\Query\Builder $q) {
                    $q->select('id')->from('backorder_queue_coverages', 'bqc')
                        ->whereColumn('bqc.backorder_queue_id', 'bq.id')
                        ->whereColumn('bqc.covered_quantity', '>', 'bqc.released_quantity');
                });
        };
    }

    private function inactiveBackorderQueueQuery(): Closure
    {
        return function (\Illuminate\Database\Query\Builder $q) {
            return $q->select('id')->from('backorder_queues', 'bq')
                ->whereColumn('bq.sales_order_line_id', 'sol.id')
                ->where(function (\Illuminate\Database\Query\Builder $q) {
                    return $q->whereNull('bq.priority')
                        ->orWhereColumn('bq.backordered_quantity', '=', 'bq.released_quantity');
                });
        };
    }

    private function fulfillmentLinesFilterQuery(): \Illuminate\Database\Query\Builder
    {
        return DB::table('sales_order_fulfillment_lines', 'sofl')
            ->selectRaw('sofl.sales_order_line_id, sum(sofl.quantity) quantity')
            ->groupBy('sofl.sales_order_line_id');
    }

    public function hasProcessableQuantity(): AttributeAlias
    {
        $processableQuantity = 0;
        $this->productLines->each(function (SalesOrderLine $salesOrderLine) use (&$processableQuantity) {
            $processableQuantity += $salesOrderLine->processable_quantity;
        });

        return AttributeAlias::get(
            fn () => (bool) $processableQuantity
        );
    }

    private function atLeastOneFulfillableLineQuery(): Builder
    {
        return self::query()->select(['id'])
            ->from('sales_orders', 'so')
            ->whereExists(function ($query) {
                // Fulfillable, has unfulfilled units without active backorder queue.
                return $query->selectRaw('sol.id')
                    ->from('sales_order_lines', 'sol')
                    ->withExpression('sofl', $this->fulfillmentLinesFilterQuery())
                    ->leftJoin('sofl', 'sofl.sales_order_line_id', 'sol.id')
                    ->whereColumn('sol.sales_order_id', 'so.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->whereNotExists($this->activeBackorderQueueQuery())
                    ->whereRaw('sol.quantity  - COALESCE(sofl.quantity, 0) > 0');
            });
    }

    public function atLeastOneActiveBackorderQueueQuery(): Builder
    {
        return self::query()->select(['id'])
            ->from('sales_orders', 'so')
            ->whereExists(function ($query) {
                return $query->selectRaw('sol.id')
                    ->from('sales_order_lines', 'sol')
                    ->whereColumn('sol.sales_order_id', 'so.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->whereExists($this->activeBackorderQueueQuery());
            });
    }

    private function bindSomeFulfillableFilter(Builder $builder, array $value): Builder
    {
        // This filter finds sales orders that have some (not all) of their
        // lines fulfillable. We therefore need to have three cases that
        // must pass for each order to be in the result set.
        //
        // 1. The order must have at least 2 unfulfilled lines.This is because,
        // if the order less than 2 unfulfilled lines, then either none or all
        // of the lines are fulfillable and that will fail the "some" requirement.
        //
        // 2. The order must have at least 1 line that can be fulfilled.
        // Such lines have no active backorder queues.
        //
        // 3. The order must have at least 1 line that cannot be fulfilled.
        // Lines with active backorder queues cannot be fulfilled.
        //
        // In addition to the above 3 conditions, we need to factor in the
        // backorder model as in $value['mode'].

        $atLeastTwoUnfulfilledLinesQuery = self::query()
            ->selectRaw('so.id')
            ->from('sales_orders', 'so')
            ->joinSub(function ($query) {
                $query->selectRaw('sol.sales_order_id')
                    ->from('sales_order_lines', 'sol')
                    ->withExpression('sofl', $this->fulfillmentLinesFilterQuery())
                    ->leftJoin('sofl', 'sofl.sales_order_line_id', 'sol.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->groupBy('sol.id', 'sol.sales_order_id')
                    ->havingRaw('sum(sol.quantity) - COALESCE(sum(sofl.quantity), 0) > 0');
            }, 'sol', 'sol.sales_order_id', 'so.id')
            ->where('so.order_status', self::STATUS_OPEN)
            ->groupBy('so.id')
            ->havingRaw('count(so.id) > 1');

        $atLeastOneUnfulfillableLineQuery = self::query()->select(['id'])
            ->from('sales_orders', 'so')
            ->whereExists(function ($query) { // Unfulfillable, active backorder queue.
                return $query->selectRaw('sol.id')
                    ->from('sales_order_lines', 'sol')
                    ->whereColumn('sol.sales_order_id', 'so.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->whereExists($this->activeBackorderQueueQuery());
            });

        $atLeastOneFulfillableLineWithBackorderHistoryQuery = self::query()->select(['id'])
            ->from('sales_orders', 'so')
            ->whereExists(function ($query) {
                // Fulfillable, has unfulfilled units without active backorder queue.
                return $query->selectRaw('sol.id')
                    ->from('sales_order_lines', 'sol')
                    ->withExpression('sofl', $this->fulfillmentLinesFilterQuery())
                    ->leftJoin('sofl', 'sofl.sales_order_line_id', 'sol.id')
                    ->whereColumn('sol.sales_order_id', 'so.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->whereExists($this->inactiveBackorderQueueQuery())
                    ->whereRaw('sol.quantity  - COALESCE(sofl.quantity, 0) > 0');
            });

        $atLeastOneFulfillableLineWithoutBackorderHistoryQuery = self::query()->select(['id'])
            ->from('sales_orders', 'so')
            ->whereExists(function ($query) {
                // Fulfillable, has unfulfilled units without active backorder queue.
                return $query->selectRaw('sol.id')
                    ->from('sales_order_lines', 'sol')
                    ->withExpression('sofl', $this->fulfillmentLinesFilterQuery())
                    ->leftJoin('sofl', 'sofl.sales_order_line_id', 'sol.id')
                    ->whereColumn('sol.sales_order_id', 'so.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->whereNotExists(function ($q) {
                        return $q->select('id')->from('backorder_queues', 'bq')
                            ->whereColumn('bq.sales_order_line_id', 'sol.id');
                    })
                    ->whereRaw('sol.quantity  - COALESCE(sofl.quantity, 0) > 0');
            });

        $atLeastOneFulfillableLineReleasedByPOQuery = self::query()->select(['id'])
            ->from('sales_orders', 'so')
            ->whereExists(function ($query) use ($value) {
                // Fulfillable, has unfulfilled units without active backorder queue.
                return $query->selectRaw('sol.id')
                    ->from('sales_order_lines', 'sol')
                    ->withExpression('sofl', $this->fulfillmentLinesFilterQuery())
                    ->leftJoin('sofl', 'sofl.sales_order_line_id', 'sol.id')
                    ->whereColumn('sol.sales_order_id', 'so.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->join('backorder_queues as bq', 'bq.sales_order_line_id', 'sol.id')
                    ->join('backorder_queue_releases as bqr', 'bqr.backorder_queue_id', 'bq.id')
                    ->where('bqr.reference', @$value['po_number'])
                    ->whereRaw('sol.quantity  - COALESCE(sofl.quantity, 0) > 0');
            });

        // Bind some items fulfillable filters.
        $builder->whereIn('id', $atLeastTwoUnfulfilledLinesQuery);
        $builder->whereIn('id', $atLeastOneUnfulfillableLineQuery);

        // Bind the mode filters.
        match ($value['mode']) {
            'all' => $builder->whereIn('id', $this->atLeastOneFulfillableLineQuery()),
            'yes' => $builder->whereIn('id', $atLeastOneFulfillableLineWithBackorderHistoryQuery),
            'no' => $builder->whereIn('id', $atLeastOneFulfillableLineWithoutBackorderHistoryQuery),
            'released_by_po' => $builder->whereIn('id', $atLeastOneFulfillableLineReleasedByPOQuery),
        };

        return $builder->where('order_status', self::STATUS_OPEN);
    }

    private function bindNoneFulfillableFilter(Builder $builder, $value): Builder
    {
        // TODO: Review this query... doens't seem to work for can fulfill none
        $atLeastOneUnfulfilledLinesQuery = self::query()
            ->selectRaw('so.id')
            ->from('sales_orders', 'so')
            ->joinSub(function ($query) {
                $query->selectRaw('sol.sales_order_id')
                    ->from('sales_order_lines', 'sol')
                    ->withExpression('sofl', $this->fulfillmentLinesFilterQuery())
                    ->leftJoin('sofl', 'sofl.sales_order_line_id', 'sol.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->groupBy('sol.id', 'sol.sales_order_id')
                    ->havingRaw('sum(sol.quantity) - COALESCE(sum(sofl.quantity), 0) > 0');
            }, 'sol', 'sol.sales_order_id', 'so.id')
            ->where('so.order_status', self::STATUS_OPEN)
            ->groupBy('so.id')
            ->havingRaw('count(so.id) >= 1');

        $atLeastOneFulfillableLineQuery = self::query()->select(['id'])
            ->from('sales_orders', 'so')
            ->whereExists(function ($query) {
                // Fulfillable, has unfulfilled units without active backorder queue.
                return $query->selectRaw('sol.id')
                    ->from('sales_order_lines', 'sol')
                    ->withExpression('sofl', $this->fulfillmentLinesFilterQuery())
                    ->leftJoin('sofl', 'sofl.sales_order_line_id', 'sol.id')
                    ->whereColumn('sol.sales_order_id', 'so.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->whereNotExists($this->activeBackorderQueueQuery())
                    ->whereRaw('sol.quantity  - COALESCE(sofl.quantity, 0) > 0');
            });

        $atLeastOneUnfulfillableLineAwaitingReceiptsQuery = self::query()->select(['id'])
            ->from('sales_orders', 'so')
            ->whereExists(function ($query) { // Unfulfillable, active backorder queue.
                return $query->selectRaw('sol.id')
                    ->from('sales_order_lines', 'sol')
                    ->whereColumn('sol.sales_order_id', 'so.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->whereExists($this->coveredActiveBackorderQueueQuery());
            });

        $atLeastOneUnfulfillableLineNeedsSourcingQuery = self::query()->select(['id'])
            ->from('sales_orders', 'so')
            ->whereExists(function ($query) { // Unfulfillable, active backorder queue.
                return $query->selectRaw('sol.id')
                    ->from('sales_order_lines', 'sol')
                    ->whereColumn('sol.sales_order_id', 'so.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->whereExists(function (\Illuminate\Database\Query\Builder $q) {
                        $q->select('id')->from('backorder_queues', 'bq')
                            ->whereColumn('bq.sales_order_line_id', 'sol.id')
                            ->where(function (\Illuminate\Database\Query\Builder $q) {
                                return $q->whereNotNull('bq.priority')
                                    ->whereColumn('bq.backordered_quantity', '!=', 'bq.released_quantity');
                            })
                            ->whereNotExists(function (\Illuminate\Database\Query\Builder $q) {
                                $q->select('id')->from('backorder_queue_coverages', 'bqc')
                                    ->whereColumn('bqc.backorder_queue_id', 'bq.id')
                                    ->whereColumn('bqc.covered_quantity', '>', 'bqc.released_quantity');
                            })
                            ->where(function (\Illuminate\Database\Query\Builder $q) {
                                $q->whereNull('bq.supplier_id')
                                    ->orWhereExists(function (\Illuminate\Database\Query\Builder $q) {
                                        $q->select('id')->from('suppliers', 's')
                                            ->whereColumn('s.id', 'bq.supplier_id')
                                            ->where('auto_generate_backorder_po', 0);
                                    });
                            });
                    });
            });

        $atLeastOneUnfulfillableLineHasSourcingScheduledQuery = self::query()->select(['id'])
            ->from('sales_orders', 'so')
            ->whereExists(function ($query) { // Unfulfillable, active backorder queue.
                return $query->selectRaw('sol.id')
                    ->from('sales_order_lines', 'sol')
                    ->whereColumn('sol.sales_order_id', 'so.id')
                    ->where('sol.is_product', true)
                    ->whereNotNull('sol.product_id')
                    ->whereExists(function (\Illuminate\Database\Query\Builder $q) {
                        $q->select('id')->from('backorder_queues', 'bq')
                            ->whereColumn('bq.sales_order_line_id', 'sol.id')
                            ->where(function (\Illuminate\Database\Query\Builder $q) {
                                return $q->whereNotNull('bq.priority')
                                    ->whereColumn('bq.backordered_quantity', '!=', 'bq.released_quantity');
                            })
                            ->whereNotExists(function (\Illuminate\Database\Query\Builder $q) {
                                $q->select('id')->from('backorder_queue_coverages', 'bqc')
                                    ->whereColumn('bqc.backorder_queue_id', 'bq.id')
                                    ->whereColumn('bqc.covered_quantity', '>', 'bqc.released_quantity');
                            })
                            ->where(function (\Illuminate\Database\Query\Builder $q) {
                                $q->whereExists(function (\Illuminate\Database\Query\Builder $q) {
                                    $q->select('id')->from('suppliers', 's')
                                        ->whereColumn('s.id', 'bq.supplier_id')
                                        ->where('auto_generate_backorder_po', 1);
                                });
                            });
                    });
            });

        $builder->whereNotIn('id', $atLeastOneFulfillableLineQuery);

        match ($value['mode']) {
            'all' => $builder->whereIn('id', $atLeastOneUnfulfilledLinesQuery),
            'awaiting_receipts' => $builder->whereIn('id', $atLeastOneUnfulfillableLineAwaitingReceiptsQuery),
            'needs_sourcing' => $builder->whereIn('id', $atLeastOneUnfulfillableLineNeedsSourcingQuery),
            'sourcing_scheduled' => $builder->whereIn('id', $atLeastOneUnfulfillableLineHasSourcingScheduledQuery),
        };

        return $builder;
    }

    private function bindBackorderedFilter(Builder $builder, string $mode, ?string $value = null): Builder
    {
        if (in_array($mode, ['yes', 'no', 'released_by_po'])) {
            return $builder->filterBackordered([], '=', $value);
        }

        return $builder;
    }

    // TODO: Need updated scope to ActiveBackorder to apply to Backorder filter in UI
    public function scopeFilterBackordered(Builder $builder, array $relation, string $operator, $value = null, string $conjunction = 'and'): Builder
    {
        // if sent from a checkbox
        if (is_bool($value) && $operator == '=') {
            $operator = $value ? 'yes' : 'no';
        }

        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        $operator = strtolower($operator);

        return $builder->{$function}(function (Builder $builder) use ($operator, $value) {
            if ($operator === 'yes') {
                // There should be queues
                return $builder->whereHas('salesOrderLines.backorderQueue');
            } elseif ($operator === 'no') {
                // Doesn't have backorder queues
                return $builder->whereDoesntHave('salesOrderLines.backorderQueue');
            } elseif ($operator === 'released_by_po') {
                return $builder->whereHas('salesOrderLines.backorderQueue', function (Builder $builder) use ($value) {
                    return $builder->releasedByPurchaseOrder($value ?: null);
                });
            }
        });
    }

    /**
     * @param  array  $relation just because we used it by HasFilters trait
     */
    public function scopeFilterCoveredByPO(Builder $builder, array $relation, string $operator, $value = null, string $conjunction = 'and'): Builder
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        // $operator = strtolower($operator);

        return $builder->{$function}(function (Builder $builder) use ($operator, $value) {
            $builder->whereHas('salesOrderLines.backorderQueue', function (Builder $builder) use ($value, $operator) {
                return $builder->coveredByPurchaseOrder($value ?: null, $operator);
            });
        });
    }

    public static function getExportableFields(): array
    {
        return SalesOrderDataImporter::getExportableFields();
    }

    public static function transformExportData(array $data): array
    {
        return BaseExporter::groupByLines($data, 'sales_order_number');
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new SalesOrderDataImporter(null, $filePath);
    }

    public function updateDiscountAndTaxAllocation(array $payload = []): self
    {
        if (isset($payload['tax_total']))
        {
            $this->tax_total = $payload['tax_total'];
            $this->update();
            return $this;
        }

        // We update the discount allocation on the lines if
        // there is tax rate id in the payload but the lines haven't been updated
        if (! isset($payload['sales_order_lines']) && ! empty($payload['tax_rate_id'])) {
            $this->salesOrderLines()
                ->each(
                    fn (SalesOrderLine $orderLine) => $this->setLineTaxAllocation($orderLine, $payload['tax_rate_id'])
                );
        }

        $this->tax_total = $this->calculated_tax_total;
        $this->update();

        $totalAmount = $this->subtotal;
        $discount_total = $this->calculated_discount_total;

        /*
         * TODO: Proration is cached we don't really need it for critical things.  Does that mean discount_allocation
         *  should be cache managed as well?  No, it can be an accessor... unless i need to filter by it... but probably
         *  safe since i can still filter by the discount on the order level... but will lose all ability to set discount
         *  on a line item basis
         */
        /*$this->salesOrderLines->each(function ($lineItem) use ($totalAmount, $discount_total) {
            $shareInPercentage = 0;
            if ($totalAmount > 0) {
                $shareInPercentage = (($lineItem->amount * $lineItem->quantity) / $totalAmount);
            }
            $lineItem->proration = $shareInPercentage;
            $lineItem->discount_allocation = $shareInPercentage * $discount_total;

            $lineItem->update();*/
        return $this;
    }

    public function exportToPDF(): string
    {
        $this->loadMissing([
            'customer',
            'salesOrderLines',
            'salesOrderLines.product',
            'salesOrderLines.product.supplierProducts',
        ]);
        $path = app(SalesOrderManager::class)->generateSalesOrderInvoice($this);

        return Storage::disk('reports')->path($path);
    }

    public function extractToPdf()
    {
        $this->loadMissing('store', 'store.address');

        if ($this->customer && $this->customer->address) {
            $customerAddress = $this->customer->address->formatForInvoice();
        } else {
            $customerAddress = '';
        }
        if ($this->store && $this->store->address) {
            $storeAddress = $this->store->address->formatForInvoice();
        } else {
            $storeAddress = '';
        }

        $lines = $this->salesOrderLines->where('quantity', '>', 0)->whereNotNull('product')->groupBy('product.id')->map(function (Collection $productGroup) {
            $firstLine = $productGroup->first();
            $quantity = $productGroup->sum('quantity');

            return [
                'sku' => $firstLine->product ? ($firstLine->supplier_product->supplier_sku ?? $firstLine->product->sku) : '',
                'description' => $firstLine->description,
                'qty' => $quantity,
                'amount' => number_format($firstLine->amount, 2),
                'subtotal' => number_format($firstLine->amount * $quantity, 2),
                'tax' => 0,
                'discount' => 0,
            ];
        })->toBase()->merge($this->salesOrderLines->whereNull('product_id')->map(function (SalesOrderLine $salesOrderLine) {
            return [
                'sku' => '',
                'description' => $salesOrderLine->description,
                'qty' => $salesOrderLine->quantity,
                'amount' => number_format($salesOrderLine->amount, 2),
                'subtotal' => number_format($salesOrderLine->subtotal, 2),
                'tax' => 0,
                'discount' => 0,
            ];
        }))->values();

        return [
            'order_number' => $this->sales_order_number,
            'order_date' => $this->order_date->toFormattedDateString(),
            'store_address' => $storeAddress,
            'store_contact' => $this->store->address ? $this->store->address->email : null,
            'store_name' => $this->store->name,
            'vendor_address' => $customerAddress,
            'vendor_name' => $this->customer ? $this->customer->name : '',
            'shipping_address' => $this->shippingAddress ? $this->shippingAddress->formatForInvoice() : '',
            'shipping_method' => $this->shippingMethod ? $this->shippingMethod->name : ($this->requested_shipping_method ?: ''),
            'ship_by' => $this->shippingMethod && $this->shippingMethod->shippingCarrier ? $this->shippingMethod->shippingCarrier->name : '',
            'order_updated_at' => $this->updated_at->toFormattedDateString(),
            'total_units' => $this->salesOrderLines->where('is_product', true)->sum('quantity'),
            'items_subtotal' => number_format($this->product_subtotal, 2),
            'total' => number_format($this->salesOrderLines->sum(function ($line) {
                return $line->quantity * $line->amount;
            }), 2),
            'total_discount' => number_format($this->discount_total, 2),
            'logo' => $this->store->logo_url ? url($this->store->logo_url) : null,
            'total_tax' => number_format($this->calculated_tax_total, 2),
            'tax_included' => $this->is_tax_included,
            'items' => $lines->toArray(),
        ];
    }

    public function getSalesOrderLineTotals(): SalesOrderTotalsDto
    {
        $salesOrderTotalsDto = SalesOrderTotalsDto::from([
            'revenue' => 0,
            'cost' => 0,
            'weight' => 0,
            'volume' => 0,
            'quantity' => 0,
        ]);

        $this->salesOrderLines->each(function (SalesOrderLine $salesOrderLine) use (&$salesOrderTotalsDto) {
            $manager = new SalesOrderLineFinancialManager();
            $salesOrderTotalsDto->revenue += $manager->getRevenue($salesOrderLine);
            $salesOrderTotalsDto->cost += $manager->getCogs($salesOrderLine) * $salesOrderLine->quantity;
            $salesOrderTotalsDto->weight += $salesOrderLine->weight_extended;
            $salesOrderTotalsDto->volume += $salesOrderLine->volume_extended;
            $salesOrderTotalsDto->quantity += $salesOrderLine->quantity;
        });

        return $salesOrderTotalsDto;
    }

    public function getTaxRevenue(): float
    {
        return $this->salesOrderLines->sum('tax_allocation_in_tenant_currency');
    }

    public function getTotalRevenue(): float
    {
        return $this->salesOrderLines->sum('salesOrderLineFinancial.total_revenue');
    }

    public function getFinancialTotal(string $attribute): float
    {
        return $this->salesOrderLines->sum('salesOrderLineFinancial.'.$attribute);
    }

    public function getFinancialLinesRevenueTotal(?bool $allocatable = null): float
    {
        $query = $this->financialLines()->revenue();

        if (! is_null($allocatable)) {
            $query->allocatable($allocatable);
        }

        return $query->sum('extended_amount');
    }

    public function getTotalCost(): float
    {
        return $this->salesOrderLines->sum('salesOrderLineFinancial.total_cost');
    }

    public function getTotalProfit(): float
    {
        return $this->salesOrderLines->sum('salesOrderLineFinancial.profit');
    }

    public function scopeAmazon(Builder $query): Builder
    {
        return $query->whereHas('salesChannel', function (Builder $query) {
            $query->whereHas('integrationInstance', function (Builder $query) {
                $query->whereHas('integration', function (Builder $query) {
                    $query->where('name', Integration::NAME_AMAZON_US);
                });
            });
        });
    }

    public function getUrl(): string{
        $params = [
            'layers' => 'OrderDrawer',
            'SKUTableOrders' => 'a:0;d:expanded',
            'OrderDrawer' => 'name:OrderDrawer;id:'.$this->id.';m:view;i:'.$this->id,
            'drawers' => 'OrderDrawer',
        ];
        return url('/orders') . '?' . url('/orders') . '?' . urldecode(http_build_query($params));
    }

    public function getLatestCoveredShipByDateAttribute(): ?Carbon
    {
        return $this->backorderedLines->max(function($orderLine) {
            return $orderLine->backorderQueue->backorderQueueCoverages->max(function($backorder) {
                return $backorder->purchaseOrderLine->estimated_delivery_date ?? $backorder->purchaseOrderLine->purchaseOrder->estimated_delivery_date;
            });
        });
    }

    public function getLatestCoveredShipByDateFormattedAttribute(): ?string
    {
        return $this->latest_covered_ship_by_date?->format('M d, Y');
    }
}
