<?php

namespace App\Models;

use App\Abstractions\FinancialDocumentInterface;
use App\Abstractions\HasNotesInterface;
use App\Contracts\HasReference;
use App\Data\AccountingTransactionData;
use App\Exporters\BaseExporter;
use App\Exporters\MapsExportableFields;
use App\Exporters\TransformsExportData;
use App\Helpers;
use App\Importers\DataImporters\SalesCreditDataImporter;
use App\Managers\SalesCreditManager;
use App\Models\Concerns\Archive;
use App\Models\Concerns\CachesOrderCurrency;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasAccountingTransaction;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasNotesTrait;
use App\Models\Concerns\HasSort;
use App\Models\Concerns\LogsActivity;
use App\Models\Concerns\TaxRateTrait;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\Shopify\ShopifyOrderMapping;
use App\Repositories\Shopify\ShopifyOrderMappingRepository;
use App\SalesCreditAllocation;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionDataFromSalesCredit;
use App\Services\Payments\Payable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Spatie\Activitylog\LogOptions;
use Throwable;

/**
 * Class SalesCredit.
 *
 *
 * @property int $id
 * @property string $sales_credit_number
 * @property string $credit_status
 * @property string $return_status
 * @property int $customer_id
 * @property ?int $sales_order_id
 * @property int $from_address_id
 * @property int $to_warehouse_id
 * @property float $total_credit
 * @property int $currency_id
 * @property float $currency_rate
 * @property int $currency_id_tenant_snapshot
 * @property int $is_for_overpayment
 * @property int $tax_rate_id
 * @property float $tax_total
 * @property string|null $payment_status
 * @property Carbon $credit_date
 * @property string|null $sales_credit_note
 * @property int|null $store_id;
 * @property bool|null $is_tax_included;
 * @property Carbon|null $fully_returned_at
 * @property Carbon|null $fully_paid_at
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read Collection|SalesCreditLine[] $salesCreditLines
 * @property-read AccountingTransaction $accountingTransaction
 * @property-read SalesOrderLine[] $sales_order_lines
 * @property-read bool $fully_received
 * @property-read bool $partially_received
 * @property-read float $paid_amount
 * @property-read Customer $customer
 * @property-read TaxRate $taxRate
 * @property-read Currency $currency
 * @property-read Currency $currencyTenantSnapshot
 * @property-read float $additional_credit
 * @property-read float $tax_credit
 * @property-read float $discount
 * @property-read float $product_total
 * @property-read SalesOrder|null $salesOrder
 * @property-read SalesOrder|null $exchange_orders
 * @property-read bool $has_products
 * @property-read Collection|ShopifyOrderMapping[] $shopifyOrderMappings
 * @property-read Note[]|EloquentCollection $notes
 */
class SalesCredit extends Model
implements
    Filterable,
    FinancialDocumentInterface,
    HasNotesInterface,
    HasReference,
    MapsExportableFields,
    Payable,
    Sortable,
    TransformsExportData
{
    use Archive;
    use CachesOrderCurrency;
    use HandleDateTimeAttributes;
    use HasAccountingTransaction;
    use HasFactory;
    use HasFilters;
    use HasNotesTrait;
    use HasSort;
    use LogsActivity;
    use TaxRateTrait;

    const CREDIT_STATUS_DRAFT = 'draft';

    const CREDIT_STATUS_OPEN = 'open';

    const CREDIT_STATUS_CLOSED = 'closed';

    const CREDIT_STATUSES = [
        self::CREDIT_STATUS_DRAFT,
        self::CREDIT_STATUS_OPEN,
        self::CREDIT_STATUS_CLOSED,
    ];

    const RETURN_STATUS_NOT_RETURNED = 'not_returned';

    const RETURN_STATUS_PARTIALLY_RETURNED = 'partially_returned';

    const RETURN_STATUS_RETURNED = 'returned';

    const RETURN_STATUSES = [
        self::RETURN_STATUS_NOT_RETURNED,
        self::RETURN_STATUS_PARTIALLY_RETURNED,
        self::RETURN_STATUS_RETURNED,
    ];

    const PAYMENT_STATUS_UNPAID = 'unpaid';

    const PAYMENT_STATUS_PAID = 'paid';

    const PAYMENT_STATUS_PARTIALLY_PAID = 'partially_paid';

    const PAYMENT_STATUSES = [
        self::PAYMENT_STATUS_UNPAID,
        self::PAYMENT_STATUS_PARTIALLY_PAID,
        self::PAYMENT_STATUS_PAID,
    ];

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    protected $fillable = [
        'id',
        'sales_order_id',
        'sales_credit_number',
        'credit_status',
        'return_status',
        'customer_id',
        'from_address_id',
        'to_warehouse_id',
        'total_credit',
        'currency_code',
        'currency_id',
        'payment_status',
        'credit_date',
        'sales_credit_note',
        'is_for_overpayment',
        'store_id;',
        'is_tax_included;',
        'fully_returned_at',
        'fully_paid_at',
        'archived_at',
        'updated_at',
    ];

    protected $casts = [
        'total_credit' => 'float',
        'credit_date' => 'datetime',
        'fully_returned_at' => 'datetime',
        'fully_paid_at' => 'datetime',
        'archived_at' => 'datetime',
        'tax_total' => 'float',
        'tax_lines' => 'array',
        'tax_rate_id' => 'integer',
        'is_tax_included' => 'boolean',
    ];

    protected $attributes = [
        'credit_status' => self::CREDIT_STATUS_OPEN,
        'return_status' => self::RETURN_STATUS_NOT_RETURNED,
        'payment_status' => self::PAYMENT_STATUS_UNPAID,
    ];

    /**
     * @throws Throwable
     */
    public function getAccountingTransactionData(): AccountingTransactionData
    {
        return (new BuildAccountingTransactionDataFromSalesCredit($this))->handle();
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return $this->salesOrder->accountingTransaction;
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [];
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->id;
    }

    public function getAccountingDateFieldName(): string
    {
        return 'credit_date';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesOrder()
    {
        return $this->belongsTo(SalesOrder::class);
    }

    public function childLinks()
    {
        $relation = $this->morphMany(OrderLink::class, 'parent');

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (OrderLink $orderLink) {
                $orderLink->delete();
                $orderLink->child->delete();
            });
        });

        return $relation;
    }

    public function salesCreditLines()
    {
        $relation = $this->hasMany(SalesCreditLine::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (SalesCreditLine $salesCreditLine) {
                $salesCreditLine->delete();
            });
        });

        return $relation;
    }

    public function orderLines()
    {
        return $this->salesCreditLines();
    }

    public function salesOrderLines()
    {
        return $this->hasManyThrough(SalesOrderLine::class, SalesCreditLine::class);
    }

    public function fromAddress()
    {
        return $this->belongsTo(Address::class, 'from_address_id');
    }

    public function toWarehouse()
    {
        return $this->belongsTo(Warehouse::class, 'to_warehouse_id');
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    public function currencyTenantSnapshot(): BelongsTo
    {
        return $this->belongsTo(Currency::class, 'currency_id_tenant_snapshot');
    }

    public function store()
    {
        return $this->belongsTo(Store::class);
    }

    public function salesCreditReturns()
    {
        $relation = $this->hasMany(SalesCreditReturn::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (SalesCreditReturn $creditReturn) {
                $creditReturn->delete();
            });
        });

        return $relation;
    }

    public function salesCreditReturnsLines()
    {
        return $this->hasManyThrough(SalesCreditReturnLine::class, SalesCreditReturn::class);
    }

    public function payments()
    {
        return $this->morphMany(Payment::class, 'link');
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    /**
     * Sets the payment status of the Sales Credit.
     */
    public function setPaymentStatus($date): void
    {
        $this->paid($date);
    }

    public function tags()
    {
        return $this->morphToMany(Tag::class, 'taggable')->withTimestamps();
    }

    public function allocations()
    {
        return $this->belongsToMany(SalesOrder::class, 'sales_credit_allocations');
    }

    public function taxRate()
    {
        return $this->belongsTo(TaxRate::class);
    }

    public function shopifyOrderMappings(): MorphMany
    {
        return $this->morphMany(ShopifyOrderMapping::class, 'sku_link');
    }

    public function accountingTransaction()
    {
        return $this->morphOne(AccountingTransaction::class, 'link');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function setCurrencyCodeAttribute($value)
    {
        $this->currency_id = empty($value) ? null : Currency::with([])->where('code', $value)->value('id');
    }

    public function getSalesOrderAttribute()
    {
        return $this->salesOrder()->first();
    }

    public function getExchangeOrdersAttribute()
    {
        return $this->childLinks->where('child_type', SalesOrder::class)->pluck('child') ?? null;
    }

    public function getFullyReceivedAttribute()
    {
        foreach ($this->salesCreditLines as $salesCreditLine) {
            if (! $salesCreditLine->fully_received) {
                return false;
            }
        }

        return true;
    }

    public function getPartiallyReceivedAttribute()
    {
        foreach ($this->salesCreditLines as $salesCreditLine) {
            if ($salesCreditLine->received_quantity > 0 && $salesCreditLine->received_quantity < $salesCreditLine->quantity) {
                return true;
            }
        }

        return false;
    }

    public function getPaidAmountAttribute()
    {
        // Payments are stored as negative amounts (refunds)
        return -$this->payments()->sum('amount');
    }

    public function getAdditionalCreditAttribute()
    {
        if ($this->relationLoaded('salesCreditLines')) {
            return $this->salesCreditLines->where('is_product', false)->sum('subtotal');
        }
    }

    public function getTaxCreditAttribute()
    {
        if ($this->relationLoaded('salesCreditLines')) {
            return $this->salesCreditLines->sum('tax_allocation');
        }
    }

    public function getProductTotalAttribute()
    {
        if ($this->relationLoaded('salesCreditLines')) {
            return $this->salesCreditLines->where('is_product', true)->sum('subtotal');
        }
    }

    public function getDiscountAttribute()
    {
        if ($this->relationLoaded('salesCreditLines')) {
            return $this->salesCreditLines->sum('discount_value');
        }
    }

    public function getHasProductsAttribute()
    {
        return $this->salesCreditLines->where('product_id', '!=', null)->isNotEmpty();
    }

    public function getSalesOrderNumberAttribute()
    {
        return $this->salesOrder ? $this->salesOrder->sales_order_number : '';
    }

    public function getShippingAddressStringAttribute()
    {
        return $this->salesOrder ? $this->salesOrder->shipping_address_string : '';
    }

    public function shippingAddress()
    {
        return $this->salesOrder ? $this->salesOrder->shippingAddress : new Address;
    }

    public function getShippingMethodNameAttribute()
    {
        if ($this->salesOrder && $this->salesOrder->shippingMethod) {
            return $this->salesOrder->shippingMethod->name;
        }
    }

    public function getOrderDateAttribute()
    {
        return $this->salesOrder ? $this->salesOrder->order_date : null;
    }

    public function getShippingCostAttribute()
    {
        return $this->salesOrder ? $this->salesOrder->shipping_cost : 0;
    }

    public function getTotalCreditFormattedAttribute()
    {
        return $this->currency ? $this->currency->code.' '.number_format($this->total_credit, 2) : '';
    }

    public function getShippingCostFormattedAttribute()
    {
        return $this->currency ? $this->currency->code.' '.number_format($this->shipping_cost, 2) : '';
    }

    public function productLines(): HasMany
    {
        return $this->salesCreditLines()->where('is_product', true);
    }

    public function getReasonsAttribute()
    {
        $reasons = '';
        $lines = $this->salesCreditReturnsLines;

        /** @var SalesCreditReturnLine $line */
        foreach ($lines as $line) {
            if ($line->returnReason) {
                $reasons .= $line->returnReason->name.'<br>';
            }
        }

        return $reasons;
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function save(array $options = [])
    {
        // set default customer reference for new sales orders
        if (! $this->exists && empty($this->sales_credit_number)) {
            $this->sales_credit_number = self::getNextLocalNumber();
        }

        // set default credit date
        if (empty($this->credit_date)) {
            $this->credit_date = now();
        }

        // Set to warehouse if not set
        if (empty($this->to_warehouse_id)) {
            $this->to_warehouse_id = Helpers::setting(Setting::KEY_SC_DEFAULT_WAREHOUSE, null, true);
        }

        // set currency from the default currency of the system
        if (empty($this->currency_id)) {
            $this->currency_id = Currency::default()->id;
        }

        $this->cacheCurrencyRate($this->isDirty(['currency_id']));

        return parent::save($options);
    }

    public function returned(?Carbon $returnDate = null)
    {
        $this->load('salesCreditLines', 'salesCreditLines.salesCreditReturnLines');

        if ($this->fully_received) {
            $this->credit_status = self::CREDIT_STATUS_CLOSED;
            $this->return_status = self::RETURN_STATUS_RETURNED;
            $this->fully_returned_at = $this->fully_returned_at ?: ($returnDate ?: now());
        } elseif ($this->partially_received) {
            $this->return_status = self::RETURN_STATUS_PARTIALLY_RETURNED;
            $this->fully_returned_at = null;
        }

        $this->save();
    }

    /**
     * Delete the sales credit.
     */
    public function delete(): bool
    {
        return DB::transaction(function () {
            $this->salesCreditLines()->each(function (SalesCreditLine $creditLine) {
                $creditLine->delete();
            });
            $this->salesCreditReturns()->delete();
            $this->payments()->delete();
            // delete allocations, this exception thrown when trying to delete them by the relation!
            // SQLSTATE[HY000]: General error: 1205 Lock wait timeout exceeded; try restarting transaction (SQL: delete `payments` from `payments` inner join `sales_credit_allocations` on `payments`.`id` = `sales_credit_allocations`.`payment_id` where `sales_credit_allocations`.`sales_credit_id` = 120)
            $allocations = SalesCreditAllocation::query()->where('sales_credit_id', $this->id)->get();
            if ($allocations->isNotEmpty()) {
                SalesCreditAllocation::query()->whereIn('id', $allocations->pluck('id'))->delete();
            }

            app(ShopifyOrderMappingRepository::class)->deleteMappingsForSalesCredit($this);

            return parent::delete();
        });
    }

    public function isUsed()
    {
        return false;
    }

    public function paid(?Carbon $paymentDate = null)
    {
        $this->load('payments');

        if ($this->total_credit <= $this->paid_amount) {
            $this->credit_status = self::CREDIT_STATUS_CLOSED;
            $this->payment_status = self::PAYMENT_STATUS_PAID;
            $this->fully_paid_at = $this->fully_paid_at ?: ($paymentDate ?: now());
        } elseif ($this->paid_amount > 0) {
            $this->payment_status = self::PAYMENT_STATUS_PARTIALLY_PAID;
            $this->credit_status = self::CREDIT_STATUS_OPEN;
            $this->fully_paid_at = null;
        } else {
            $this->payment_status = self::PAYMENT_STATUS_UNPAID;
            $this->credit_status = self::CREDIT_STATUS_OPEN;
            $this->fully_paid_at = null;
        }

        $this->save();
    }

    public function getReference(): ?string
    {
        return $this->sales_credit_number;
    }

    /**
     * @return array|null array of the credit lines created/updated.
     *
     * @throws Throwable
     */
    public function setSalesCreditLines($salesCreditLines, bool $sync = true): ?array
    {
        if ($salesCreditLines === false) {
            return null;
        }

        $createdLines = [];

        foreach ($salesCreditLines ?: [] as $creditLine) {
            if (! empty($creditLine['id'])) {
                /** @var SalesCreditLine $salesCreditLine */
                $salesCreditLine = SalesCreditLine::with([])->findOrFail($creditLine['id']);
            }

            if (! isset($salesCreditLine)) {
                $salesCreditLine = new SalesCreditLine();
            }

            if (isset($creditLine['sales_order_line_id'])) {
                $salesOrderLine = SalesOrderLine::where('id', $creditLine['sales_order_line_id'])->first();
                $creditLine['unit_cost'] = (new SalesCreditManager())->getUnitCost($salesOrderLine, $creditLine['quantity']);
            }

            $salesCreditLine->fill($creditLine);

            if (isset($creditLine['sales_order_line_id']) || $salesCreditLine->sales_order_line_id) {
                $salesOrderLineId = $salesCreditLine->sales_order_line_id ?? $creditLine['sales_order_line_id'];

                /** @var SalesOrderLine $salesOrderLine */
                $salesOrderLine = SalesOrderLine::query()->find($salesOrderLineId);

                if ($salesOrderLine) {
                    // Set product id for the sales credit ine if the sales order line is a product.
                    $salesCreditLine->product_id = $salesOrderLine->product_id;
                    if ($salesOrderLine->tax_rate_id) {
                        $taxRate = TaxRate::with([])->findOrFail($salesOrderLine->tax_rate_id);
                        $salesCreditLine->tax_rate_id = $taxRate->id;
                    }
                }
            }

            //Used in update SKU-4951
            if (isset($creditLine['tax_rate_id'])) {
                $salesCreditLine->tax_rate_id = @$creditLine['tax_rate_id'];
            }
            $this->salesCreditLines()->save($salesCreditLine);

            $createdLines[] = $salesCreditLine->id;
            unset($salesCreditLine);
        }

        // sync sales order lines
        if ($sync) {
            $this->salesCreditLines()->whereNotIn('id', $createdLines)->delete();
        }

        $this->setTotal();
        $this->returned();
        $this->paid();

        return $createdLines;
    }

    /**
     * Set credit total from credit lines.
     */
    public function setTotal()
    {
        $total = 0;
        foreach ($this->salesCreditLines as $creditLine) {
            $total += $creditLine->amount * $creditLine->quantity;
        }

        $this->total_credit = $total;
        $this->save();
    }

    /**
     * Get next customer reference.
     */
    public static function getNextLocalNumber(): string
    {
        $prefix = Helpers::setting(Setting::KEY_SC_PREFIX, 'SC-');
        $numOfDigits = Helpers::setting(Setting::KEY_SC_NUM_DIGITS, 5);
        $startFrom = Helpers::setting(Setting::KEY_SC_START_NUMBER, 1);

        $lastLocalSalesCredit = self::with([])->where('sales_credit_number', 'like', "{$prefix}%")
            ->latest('id')
            ->first();

        if ($lastLocalSalesCredit) {
            $lastNumber = (int) explode($prefix, $lastLocalSalesCredit->sales_credit_number)[1] ?? null;
            $nextNumber = $lastNumber ? ($lastNumber + 1) : $startFrom;
        } else {
            $nextNumber = $startFrom;
        }

        return sprintf("{$prefix}%0{$numOfDigits}d", ($nextNumber < $startFrom ? $startFrom : $nextNumber));
    }

    public function parentLinks()
    {
        return $this->morphMany(OrderLink::class, 'child');
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.sales_credit.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        if (func_num_args() && Str::startsWith(func_get_arg(0), ['customer_name.', 'store.', 'item_sku.'])) {
            return func_get_args();
        }

        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['sales_credit_number', 'credit_status'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeAccountingReady(Builder $builder): Builder
    {
        return $builder
            ->where('credit_status', '!=', SalesCredit::CREDIT_STATUS_DRAFT)
            ->whereHas('salesCreditLines')
            ->whereHas('salesOrder', function (Builder $builder) {
                $builder->whereNull('canceled_at');
            });
    }

    public function scopeFilterSalesCreditLines(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        if ($relation['key'] == 'has_products') {
            return $builder->whereHas('salesCreditLines', function (Builder $builder) use ($value) {
                if ($value) { // at least one product in the lines
                    return $builder->whereNotNull('product_id');
                } else { // all lines are a non product
                    return $builder->groupBy(['sales_credit_id'])->havingRaw('SUM(`product_id`) is NULL');
                }
            });
        }

        // default behavior
        return false;
    }

    public function extractToPdf()
    {
        if ($this->customer && $this->customer->address) {
            $customerAddress = collect([
                $this->customer->address->address1,
                $this->customer->address->city,
                $this->customer->address->province_code,
                $this->customer->address->zip,
                $this->customer->address->country,
            ]);
            $customerAddress = $customerAddress->filter()->implode(', ');
        } else {
            $customerAddress = '';
        }

        if ($this->store && $this->store->address) {
            $storeAddress = collect([
                $this->store->address->address1,
                $this->store->address->city,
                $this->store->address->province_code,
                $this->store->address->zip,
                $this->store->address->country,
            ]);
            $storeAddress = $storeAddress->filter()->implode(', ');
        } else {
            $storeAddress = '';
        }

        return [
            'credit_number' => $this->sales_credit_number,
            'credit_date' => $this->credit_date->toFormattedDateString(),
            'customer_address' => $customerAddress,
            'store_address' => $storeAddress,
            'store_contact' => $this->store->address->email,
            'originating_order' => $this->salesOrder ? $this->salesOrder->sales_order_number : '',
            'logo' => $this->store->logo_url ? url($this->store->logo_url) : null,
            'items' => $this->salesCreditLines->map(function ($line) {
                return [
                    'sku' => $line->product ? $line->product->sku : '',
                    'description' => $line->description,
                    'qty' => $line->quantity,
                    'amount' => $line->amount,
                    'subtotal' => $line->subtotal,
                    'tax' => $line->tax ?? 0,
                    'discount' => $line->discount ?? 0,
                ];
            })->toArray(),
            'total_units' => $this->salesCreditLines->count(),
            'items_subtotal' => $this->salesCreditLines->sum('subtotal'),
            'total_discound' => $this->discount,
            'credit_total' => $this->total_credit,
        ];
    }

    public static function getExportableFields(): array
    {
        return SalesCreditDataImporter::getExportableFields();
    }

    public static function transformExportData(array $data): array
    {
        return BaseExporter::groupByLines($data, 'customer_reference');
    }

    public function setTaxAllocation()
    {
        $linesWithTaxRates = $this->salesCreditLines->whereNotNull('tax_rate_id')->count();

        if ($linesWithTaxRates == 0 && ($calculatedTaxTotal = $this->calculated_tax_total) > 0) {
            // Lines have no tax rate id but the SC has a tax total so we prorate
            $totalAmount = round($this->salesCreditLines->sum('amount'), 4);

            $this->salesCreditLines->whereNotNull('tax_rate_id')->each(function ($lineItem) use ($totalAmount, $calculatedTaxTotal) {
                $shareInPercentage = 0;
                if ($totalAmount > 0) {
                    $shareInPercentage = ($lineItem->amount / $totalAmount);
                }
                $lineItem->proration = $shareInPercentage;
                $lineItem->tax_allocation = $shareInPercentage * $calculatedTaxTotal;
                $lineItem->update();
            });
        } else {
            // Calculate tax allocations the obvious way (SKU-5998)
            $this->salesCreditLines()->whereNotNull('tax_rate_id')->each(function (SalesCreditLine $lineItem) {
                $allocation = $lineItem->taxRate ? $lineItem->taxRate->rate * $lineItem->amount * $lineItem->quantity / 100 : 0;
                $lineItem->update(['tax_allocation' => $allocation]);
            });
        }

        // Set allocation to 0 for lines without tax rate id.
        $this->salesCreditLines()->whereNull('tax_rate_id')->update(['tax_allocation' => 0]);

        $this->tax_total = $this->salesCreditLines()->sum('tax_allocation');
        $this->save();
    }
}
