<?php

namespace App\Models;

use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Repositories\SalesOrder\SalesOrderRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * Class SalesChannel.
 *
 *
 * @property int $id
 * @property int $integration_instance_id
 * @property int|null $store_id
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read array|null $credentials
 * @property-read string $sales_channel_type
 * @property-read SalesChannelType $type
 * @property-read IntegrationInstance|null $integrationInstance
 * @property-read Integration $integration
 * @property-read Store $store
 * @property string $name
 */
class SalesChannel extends Model implements Filterable, Sortable
{
    use HasFactory;
    use HasFilters, HasSort;

    const LOCAL_CHANNEL_ID = 1;

    const UNSPECIFIED_SHIPPING_METHOD = 'Unspecified';

    protected $casts = [
        'connection_settings' => 'array',
        'integration_settings' => 'array',
    ];

    protected $fillable = [
        'store_id',
        'integration_instance_id',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function type()
    {
        return $this->belongsTo(SalesChannelType::class, 'sales_channel_type_id');
    }

    public function store(): BelongsTo
    {
        return $this->belongsTo(Store::class);
    }

    public function integrationInstance()
    {
        return $this->belongsTo(IntegrationInstance::class);
    }

    public function productListings()
    {
        $relation = $this->hasMany(ProductListing::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (ProductListing $productListing) {
                $productListing->setRelation('salesChannel', $this);
                $productListing->delete();
            });
        });

        return $relation;
    }

    public function salesOrders()
    {
        $relation = $this->hasMany(SalesOrder::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (SalesOrder $salesOrder) {
                $salesOrder->delete();
            });
        });

        return $relation;
    }

    public function shippingMethodMappings()
    {
        return $this->hasMany(ShippingMethodMappingsSalesChannelToSku::class);
    }

    public function paymentMethodMappings()
    {
        return $this->hasMany(PaymentMethodMappingSalesChannelToSku::class);
    }

    public function emailsCustomers(): bool
    {
        $instance = $this->integrationInstance;
        if ($instance->isAmazonInstance()) {
            return false;
        }

        $settings = $instance->integration_settings;
        if (is_string($settings)) {
            $settings = json_decode($settings, true);
        }

        if (! empty(@$settings['emailCustomers'])) {
            return $settings['emailCustomers'];
        }

        return false;
    }

    public function getSalesChannelTypeAttribute()
    {
        return SalesChannelType::getTypeByID($this->sales_channel_type_id);
    }

    public function getIntegrationAttribute()
    {
        if (empty($this->integration_instance_id)) {
            return Integration::getSkuIntegration();
        }

        return $this->integrationInstance->integration;
    }

    public function getNameAttribute()
    {
        if (! isset($this->integration_instance_id)) {
            return Integration::NAME_SKU_IO;
        }

        return $this->integrationInstance->name ?? null;
    }

    public function delete()
    {
        // Delete productListings
        ProductListing::query()->where('sales_channel_id', $this->id)->where('is_fba', false)->delete();
        // Delete sales orders, don't delete it directly by  the relation (not all lines are deleted!)
        app(SalesOrderRepository::class)->bulkDelete($this->salesOrders()->select(['id'])->pluck('id')->toArray());
        // delete fba listings, don't delete it directly by  the relation (not all lines are deleted!)
        $this->productListings()->eachById(fn (ProductListing $pl) => $pl->delete());
        // delete shipping method mappings
        $this->shippingMethodMappings()->delete();
        // delete payment method mappings
        $this->paymentMethodMappings()->delete();

        return parent::delete();
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.sales_channel.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name', 'country'];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }
}
