<?php

namespace App\Models;

use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * Class ReturnReasons.
 *
 *
 * @property int $id
 * @property string $name
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 */
class ReturnReason extends Model implements Filterable, Sortable
{
    use Archive, HasFactory, HasFilters, HasSort;

    protected $fillable = ['name'];

    protected $casts = ['archived_at' => 'datetime'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesCreditReturnLines()
    {
        return $this->hasMany(SalesCreditReturnLine::class, 'reason_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * delete return reason
     *
     * @return array|bool|null
     *
     * @throws Exception
     */
    public function delete()
    {
        // If the model was used from any related relation
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return parent::delete();
    }

    /**
     * check if there is related purchase orders.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        // load related relations
        if (! isset($this->sales_credit_return_lines_count)) {
            $this->loadCount('salesCreditReturnLines');
        }

        if ($this->sales_credit_return_lines_count) {
            return [
                'salesCreditReturnLines' => trans_choice('messages.currently_used', $this->sales_credit_return_lines_count, [
                    'resource' => 'sales credit return line',
                    'model' => 'return reason('.$this->name.')',
                ]),
            ];
        }

        return false;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.return_reason.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }
}
