<?php

namespace App\Models;

use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;

/**
 * Class ReportingDailyProductAccountingFinancials.
 *
 *
 * @property int $id
 * @property Carbon $reporting_date
 * @property int $product_id
 * @property int $supplier_id
 * @property int $brand_id
 * @property int $quantity_sold
 * @property int $num_orders
 * @property float $revenue
 * @property float $cost_order
 * @property float $cost_credits
 * @property float $cost_landed
 * @property float $cost_overhead
 * @property array $cost_breakdown
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read Product $product
 * @property-read SalesOrderLine[]|Collection $salesOrderLines
 */
class ReportingDailyProductAccountingFinancials extends Model implements Filterable, Sortable
{
    use HasFilters, HasSort, Upsert;

    protected $casts = [
        'quantity_sold' => 'integer',
        'num_orders' => 'integer',
        'revenue' => 'float',
        'cost_order' => 'float',
        'cost_credits' => 'float',
        'cost_landed' => 'float',
        'cost_overhead' => 'float',
        'cost_breakdown' => 'array',
    ];

    protected $fillable = ['reporting_date', 'product_id'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }

    public function brand()
    {
        return $this->belongsTo(ProductBrand::class);
    }

    public function salesOrderLines()
    {
        return $this->hasMany(SalesOrderLine::class, 'product_id', 'product_id');
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return [
            'reporting_date',
            'product_id',
            'supplier_id',
            'brand_id',
            'quantity_sold',
            'num_orders',
            'revenue',
            'cost_order',
            'cost_overhead',
            'cost_breakdown',
        ];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    public function sortableColumns()
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['reporting_date'];
    }
}
