<?php

namespace App\Models;

use App\Abstractions\FinancialDocumentInterface;
use App\Contracts\HasReference;
use App\Data\AccountingTransactionData;
use App\Exceptions\NegativeInventoryFulfilledSalesOrderLinesException;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasAccountingTransaction;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionDataFromPurchaseOrderReceipt;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\InventoryManagement\InventoryManager;
use Carbon\Carbon;
use Countable;
use Exception;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Throwable;

/**
 * Class PurchaseOrderShipmentReceipt.
 *
 *
 * @property int $id
 * @property int $purchase_order_shipment_id
 * @property int $user_id
 * @property Carbon $received_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated
 * @property-read User $user
 * @property-read PurchaseOrderShipment $purchaseOrderShipment
 * @property-read Collection|PurchaseOrderShipmentReceiptLine[] $purchaseOrderShipmentReceiptLines
 *
 * @method static withCostValues(?Builder $builder = null)
 */
class PurchaseOrderShipmentReceipt extends Model implements HasReference, FinancialDocumentInterface
{
    use HandleDateTimeAttributes, HasAccountingTransaction, HasFactory;

    protected $fillable = ['purchase_order_shipment_id', 'received_at', 'user_id'];

    protected $casts = [
        'received_at' => 'datetime',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function purchaseOrderShipment()
    {
        return $this->belongsTo(PurchaseOrderShipment::class);
    }

    public function purchaseOrderShipmentReceiptLines()
    {
        $relation = $this->hasMany(PurchaseOrderShipmentReceiptLine::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (PurchaseOrderShipmentReceiptLine $receiptLine) {
                $receiptLine->delete();
            });
        });

        return $relation;
    }

    public function getReference(): ?string
    {
        return $this->purchaseOrderShipment->getReference();
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function accountingTransaction()
    {
        return $this->morphOne(AccountingTransaction::class, 'link');
    }

    /**
     * @throws Throwable
     */
    public function getAccountingTransactionData(): AccountingTransactionData
    {
        return (new BuildAccountingTransactionDataFromPurchaseOrderReceipt($this))->handle();
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return $this->purchaseOrderShipment->purchaseOrder->accountingTransaction;
    }

    public function getAccountingDateFieldName(): string
    {
        return 'received_at';
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     * @throws Throwable
     */
    public function delete(): ?bool
    {
        $purchaseOrder = $this->purchaseOrderShipment->purchaseOrder;
        if ($purchaseOrder->dropshipping) {
            $this->purchaseOrderShipmentReceiptLines->each(function (PurchaseOrderShipmentReceiptLine $receiptLine) {
                $receiptLine->delete();
            });
        } else {
            (new BulkInventoryManager())->bulkDeletePositiveInventoryEvents($this->purchaseOrderShipmentReceiptLines);
        }

        $delete = parent::delete();

        // For now, there is only one receipt per shipment, so we can just delete the shipment when deleting the receipt
        $this->purchaseOrderShipment?->delete(false);

        return $delete;
    }

    public function save(array $options = [])
    {
        if (empty($this->user_id)) {
            $this->user_id = auth()->id();
        }

        return parent::save($options);
    }

    /**
     * @throws Exception
     */
    public function addLineToInventory(PurchaseOrderShipmentReceiptLine $receiptLine, ?int $warehouseId = null)
    {
        $warehouseId = $warehouseId ?: $this->purchaseOrderShipment->purchaseOrder->destination_warehouse_id;
        if (! $warehouseId) {
            return;
        } // We only add inventory if linked to warehouse

        $product = $receiptLine->purchaseOrderShipmentLine->purchaseOrderLine->product;

        // Receipt line must be linked to product
        if (! $product) {
            return;
        }

        /**
         * Add the quantity on the receipt line
         * to stock.
         */
        InventoryManager::with(
            $warehouseId,
            $product
        )->addToStock($receiptLine->quantity, $receiptLine, false);

        /**
         * We sync up the backorder queue coverages to reflect
         * the current inbound quantity of the PO line. This ensures
         * that there are no ghost coverages and affected backorder queues
         * can be covered by a future PO.
         */
        dispatch(new SyncBackorderQueueCoveragesJob(Arr::wrap($receiptLine->purchaseOrderShipmentLine->purchaseOrderLine->id)));
    }

    /**
     * Add FifoLayer and InventoryMovements for Receipt Lines.
     *
     *
     * @throws Exception
     */
    public function addLinesToInventory(Arrayable|Countable $receiptLines, ?int $warehouseId = null)
    {
        $warehouseId = $warehouseId ?: $this->purchaseOrderShipment->purchaseOrder->destination_warehouse_id;
        if (! $warehouseId) {
            return;
        } // We only add inventory if linked to warehouse

        // calculate receipt lines cost
        $this->calculateReceiptLinesCost($receiptLines);

        // update purchase order destination warehouse if not fill
        $this->updatePurchaseOrderDestinationWarehouse($warehouseId);

        //        $progress = (new SKUProgressEvent(
        //          $this->purchaseOrderShipment->purchaseOrder->getReceiptBroadcastChannel(),
        //          $receiptLines->count()
        //        ))->send();

        /** @var PurchaseOrderShipmentReceiptLine $receiptLine */
        foreach ($receiptLines as $receiptLine) {
            $product = $receiptLine->purchaseOrderShipmentLine->purchaseOrderLine->product;

            // Receipt line must be linked to product
            if (! $product) {
                continue;
            }

            /**
             * Add the quantity on the receipt line
             * to stock.
             */
            InventoryManager::with(
                $warehouseId,
                $product
            )->addToStock($receiptLine->quantity, $receiptLine);
            //            $progress->increment()->send();
        }
    }

    /**
     * Calculate total cost for every PO Shipment Receipt Line.
     *
     * @param  Arrayable|mixed  $receiptLines
     */
    public function calculateReceiptLinesCost($receiptLines): void
    {
        // TODO: The way purchase order line is loaded here seems unnecessary... review logic

        // TODO: Not sure why we need to calculate this here if we calculate it in calculateLineUnitCost
        // $productTotal = $this->purchaseOrderShipment->purchaseOrder->load('purchaseOrderLines')->product_total;
        foreach ($receiptLines as $receiptLine) {
            $purchaseOrderLineId = $receiptLine->purchaseOrderShipmentLine->purchase_order_line_id;
            /** @var PurchaseOrderLine $purchaseOrderLine */
            $purchaseOrderLine = $this->purchaseOrderShipment->purchaseOrder->purchaseOrderLines->firstWhere('id', $purchaseOrderLineId);

            // total cost of the receipt line
            // TODO: Not sure why we need to pass $productTotal here if we calculate it in calculateLineUnitCost
            //$receiptLine->total_cost = $receiptLine->quantity * $this->calculateLineUnitCost($purchaseOrderLine, true, $productTotal);
            $receiptLine->total_cost = $receiptLine->quantity * $this->calculateLineUnitCost($purchaseOrderLine, true);
            $receiptLine->product_id = $purchaseOrderLine->product_id;
            $receiptLine->purchaseOrderLine = $purchaseOrderLine;
            $receiptLine->purchaseOrderShipmentReceipt = $this;
        }
    }

    /**
     * Calculates the unit cost for the purchase order line.
     */
    //public function calculateLineUnitCost(PurchaseOrderLine $orderLine, bool $byDefaultCurrency = true, $productTotal = null): float
    public function calculateLineUnitCost(PurchaseOrderLine $orderLine, bool $byDefaultCurrency = true): float
    {
        if ($orderLine->quantity == 0) {
            return $orderLine->amount;
        }
        // use the default currency for purchase order lines values
        // TODO: The existing values passed (if we don't use the postfix) would be in the transaction currency.  So enabling this would cause a double conversion.
        // PurchaseOrderLine::useDefaultCurrency(true);

        //$productsSubtotal = ! $productTotal ? $this->purchaseOrderShipment->purchaseOrder->load('purchaseOrderLines')->product_total : $productTotal;
        //$additionalCost = $this->purchaseOrderShipment->purchaseOrder->additional_cost;

        $purchaseOrder = $orderLine->purchaseOrder->load('purchaseOrderLines');

        $postfix = $byDefaultCurrency ? '_in_tenant_currency' : '';
        $productsSubtotal = $purchaseOrder->{'product_total'.$postfix};
        $additionalCost = $purchaseOrder->{'additional_cost'.$postfix};

        // the ratio of purchase order line from products total
        $lineRatio = $productsSubtotal == 0 ? 0 : $orderLine->{'subtotal'.$postfix} / $productsSubtotal;
        // total cost of the purchase order line(subtotal + the additional cost value on the line )
        $lineTotalCost = $orderLine->{'subtotal'.$postfix} + ($additionalCost * $lineRatio);

        // use original purchase order lines value
        // TODO: See TODO Above
        // PurchaseOrderLine::useDefaultCurrency(false);

        // unit cost of the purchase order line after adding the additional cost
        return $orderLine->quantity == 0 ? 0 : $lineTotalCost / $orderLine->quantity;
    }

    /**
     * Set destination warehouse id for Purchase Order if empty.
     */
    private function updatePurchaseOrderDestinationWarehouse(int $warehouseId)
    {
        if (empty($this->purchaseOrderShipment->purchaseOrder->destination_warehouse_id)) {
            $this->purchaseOrderShipment->purchaseOrder->destination_warehouse_id = $warehouseId;
            $this->purchaseOrderShipment->purchaseOrder->save();
        }
    }

    public static function aggregateReceiptLines(array $receiptLines): array
    {
        // Fill in purchase order line id and aggregate quantities.
        $receiptLines = array_map(function ($receiptLine) {
            if (! (isset($receiptLine['purchase_order_line_id'])) && isset($receiptLine['purchase_order_line_reference'])) {
                $receiptLine['purchase_order_line_id'] = PurchaseOrderLine::with([])->where('line_reference', $receiptLine['purchase_order_line_reference'])
                    ->firstOrFail()->id;
            } elseif (! (isset($receiptLine['purchase_order_line_id'])) && isset($receiptLine['purchase_order_shipment_line_id'])) {
                $receiptLine['purchase_order_line_id'] = PurchaseOrderShipmentLine::with([])->findOrFail($receiptLine['purchase_order_shipment_line_id'])->purchaseOrderLine->id;
            }

            return $receiptLine;
        }, $receiptLines);

        $uniqueLineIds = array_unique(array_map(function ($line) {
            return $line['purchase_order_line_id'];
        }, $receiptLines));
        $uniqueLines = [];

        foreach ($uniqueLineIds as $lineId) {
            $matched = array_values(array_filter($receiptLines, function ($line) use ($lineId) {
                return $line['purchase_order_line_id'] === $lineId;
            }));

            $uniqueLines[] = array_merge($matched[0], [
                'purchase_order_line_id' => $lineId,
                'quantity' => array_sum(array_values(array_map(function ($line) {
                    return $line['quantity'];
                }, $matched))),
            ]);
        }

        return $uniqueLines;
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeWithCostValues(Builder $builder): Builder
    {
        return $builder->with([
            'purchaseOrderShipmentReceiptLines.purchaseOrderShipmentLine.purchaseOrderLine' => fn ($q) => $q->withCostValues(),
        ]);
    }

    public function scopeAccountingReady(Builder $builder): Builder
    {
        $builder->whereDoesntHave('purchaseOrderShipment.purchaseOrder', function (Builder $builder) {
            $builder->whereNotNull('sales_order_id');
            $builder->orWhereNull('destination_warehouse_id');
        });

        return $builder;
    }
}
