<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Contracts\HasReference;
use App\Exceptions\CannotDeleteReceivedPurchaseOrderLineException;
use App\Helpers;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasCurrencyAttributes;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute as Attribute;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Arr;
use Kirschbaum\PowerJoins\PowerJoins;
use Spatie\Activitylog\LogOptions;
use Throwable;

/**
 * Class PurchaseOrderLine.
 *
 *
 * @property int $id
 * @property int $purchase_order_id
 * @property string $description
 * @property int $product_id
 * @property float $quantity
 * @property float $received_quantity
 * @property float $unreceived_quantity
 * @property float $amount
 * @property float $tax_allocation
 * @property int $tax_rate_id
 * @property float $discount_rate
 * @property float $discount_amount
 * @property float $discount_amount_extended
 * @property float $total_amount
 * @property Carbon|null $estimated_delivery_date
 * @property int $nominal_code_id
 * @property string|mixed $line_reference
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read float $amount_in_tenant_currency
 * @property-read float $discount_in_tenant_currency
 * @property-read float $tax_allocation_in_tenant_currency
 * @property-read SupplierProduct|null $supplierProduct
 * @property-read float $subtotal
 * @property-read float $subtotal_in_tenant_currency
 * @property-read float $prorated_cost
 * @property-read float $total_cost
 * @property-read float $total_cost_in_tenant_currency
 * @property-read float $proration_of_product_costs
 * @property-read float $discount_value
 * @property-read float $discount_value_in_tenant_currency
 * @property-read float $tax_value
 * @property-read float $tax_value_in_tenant_currency
 * @property-read bool $fully_shipped
 * @property-read bool $fully_invoiced
 * @property-read bool $fully_received
 * @property-read float $unfulfilled_quantity
 * @property-read float $uninvoiced_quantity
 * @property-read float $invoice_total
 * @property-read Carbon|null $last_receipt_date
 * @property-read int $backorder_queue_coverage_quantity
 * @property-read int $available_quantity_for_backorders
 * @property-read PurchaseOrder $purchaseOrder
 * @property-read Product $product
 * @property-read PurchaseOrderShipmentReceiptLine[]|Collection $purchaseOrderShipmentReceiptLines
 * @property-read PurchaseOrderShipmentReceipt[]|Collection $purchaseOrderShipmentReceipts
 * @property-read PurchaseInvoiceLine[]|Collection $purchaseInvoiceLines
 * @property-read PurchaseInvoice[]|Collection $purchaseInvoices
 * @property-read BackorderQueueCoverage[]|Collection $coveredBackorderQueues
 * @property-read BackorderQueueRelease[]|Collection $backorderQueueReleases
 * @property-read InventoryAdjustment[]|Collection $adjustments
 * @property-write string $nominal_code
 * @property-write string $sku
 *
 * @method static withCostValues(?Builder $builder = null, ?int $purchaseOrderId = null)
 */
class PurchaseOrderLine extends Model implements Filterable, HasReference, UniqueFieldsInterface
{
    use BulkImport, Concerns\LogsActivity, HandleDateTimeAttributes, HasCurrencyAttributes, HasFactory, HasFilters, PowerJoins;

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    protected $casts = [
        'quantity' => 'float',
        'estimated_delivery_date' => 'datetime',
        'amount' => 'float',
        'discount' => 'float',
        'linked_backorders' => 'array',
    ];

    protected $fillable = [
        'id',
        'purchase_order_id',
        'description',
        'product_id',
        'sku',
        'nominal_code',
        'quantity_invoiced',
        'quantity',
        'amount',
        'tax_allocation',
        'received_quantity',
        'tax_rate_id',
        'tax_rate',
        'discount_rate',
        'estimated_delivery_date',
        'nominal_code_id',
        'line_reference',
        'linked_backorders',
        'updated_at',
    ];

    protected $attributes = ['quantity' => 1];

    protected $currencyAttributes = ['amount'];

    protected $touches = [
        'purchaseOrder',
        'purchaseOrderShipmentReceipts',
        'purchaseInvoices',
    ];

    public static function getUniqueFields(): array
    {
        return [
            'purchase_order_id',
            'product_id',
            'description',
            'quantity',
            'amount',
        ];
    }


    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function purchaseOrder()
    {
        return $this->belongsTo(PurchaseOrder::class);
    }

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function inventoryMovement()
    {
        return $this->morphOne(InventoryMovement::class, 'link');
    }

    public function purchaseInvoiceLines()
    {
        return $this->hasMany(PurchaseInvoiceLine::class);
    }

    public function purchaseInvoices(): HasManyThrough
    {
        return $this->hasManyThrough(
            PurchaseInvoice::class,
            PurchaseInvoiceLine::class,
            'purchase_order_line_id',
            'id',
            'id',
            'purchase_invoice_id',
        );
    }

    public function supplierProducts()
    {
        return $this->hasMany(SupplierProduct::class, 'product_id', 'product_id');
    }

    public function nominalCode()
    {
        return $this->belongsTo(NominalCode::class);
    }

    public function purchaseOrderShipmentLines()
    {
        return $this->hasMany(PurchaseOrderShipmentLine::class);
    }

    public function taxRate()
    {
        return $this->belongsTo(TaxRate::class);
    }

    public function purchaseOrderShipmentReceiptLines(): HasManyThrough
    {
        return $this->hasManyThrough(
            PurchaseOrderShipmentReceiptLine::class,
            PurchaseOrderShipmentLine::class,
        );
    }

    public function purchaseOrderShipmentReceipts(): HasManyThrough
    {
        return $this->hasManyThrough(
            PurchaseOrderShipmentReceipt::class,
            PurchaseOrderShipmentReceiptLine::class,
            'purchase_order_line_id',
            'id',
            'id',
            'purchase_order_shipment_receipt_id',
        );
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    /**
     * @throws CannotDeleteReceivedPurchaseOrderLineException
     */
    public function delete(): bool
    {
        if($this->purchaseOrderShipmentReceiptLines->count() > 0)
        {
            throw new CannotDeleteReceivedPurchaseOrderLineException();
        }
        $this->purchaseInvoiceLines()->delete();

        $this->forecastItemLink()->delete();

        return parent::delete();
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function forecastItemLink(): HasOne{
        return $this->hasOne(ForecastItemPOLineLink::class, 'purchase_order_line_id');
    }

    public function amountInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->amount * $this->purchaseOrder->currency_rate);
    }

    public function discountInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->discount_rate * $this->purchaseOrder->currency_rate);
    }

    public function taxAllocationInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->tax_allocation * $this->purchaseOrder->currency_rate);
    }

    public function getSupplierProductAttribute()
    {
        return $this->supplierProducts()->firstWhere('supplier_id', $this->purchaseOrder?->supplier_id);
    }

    public function getSubtotalAttribute()
    {
        return ($this->quantity * $this->amount) - $this->discount_value;
    }

    public function getSubtotalInTenantCurrencyAttribute()
    {
        return ($this->quantity * $this->amount_in_tenant_currency) - $this->discount_value_in_tenant_currency;
    }

    public function getProratedCostAttribute(): float
    {
        return $this->proration_of_product_costs * $this->purchaseOrder->additional_cost;
    }

    public function getProrationOfProductCostsAttribute(): float
    {
        if ($this->purchaseOrder->product_total == 0) {
            return 0.00;
        }

        return $this->subtotal / $this->purchaseOrder->product_total;
    }

    public function getTotalCostAttribute(): float
    {
        if (isset($this->attributes['total_cost'])) {
            return $this->attributes['total_cost'];
        }

        return $this->subtotal + $this->prorated_cost;
    }

    public function getTotalCostInTenantCurrencyAttribute(): float
    {
        return $this->total_cost * $this->purchaseOrder->currency_rate;
    }

    public function getDiscountValueAttribute()
    {
        return $this->quantity * $this->amount * $this->discount_rate;
    }

    public function getDiscountValueInTenantCurrencyAttribute()
    {
        return $this->quantity * $this->amount_in_tenant_currency * $this->discount_in_tenant_currency;
    }

    public function getTaxValueAttribute()
    {
        return $this->quantity * $this->amount / 100;
    }

    public function getTaxValueInTenantCurrencyAttribute()
    {
        return $this->quantity * $this->amount_in_tenant_currency / 100;
    }

    public function getFullyShippedAttribute()
    {
        $this->loadMissing('purchaseOrderShipmentLines');

        if (! $this->product_id) {
            return true;
        }

        return $this->quantity <= $this->purchaseOrderShipmentLines->sum('quantity');
    }

    public function getFullyInvoicedAttribute()
    {
        $this->loadMissing('purchaseInvoiceLines');

        return $this->quantity <= $this->invoice_total;
    }

    public function getInvoiceTotalAttribute()
    {
        return $this->purchaseInvoiceLines->sum('quantity_invoiced');
    }

    public function getFullyReceivedAttribute(): bool
    {
        return (int)$this->quantity - (int)$this->received_quantity <= 0;
    }

    public function adjustments(): MorphMany
    {
        return $this->morphMany(InventoryAdjustment::class, 'link');
    }

    /**
     * Set/Update quantity when creating purchase invoice.
     */
    public function setQuantityInvoicedAttribute($value)
    {
        if (! $this->exists) { // for a new lines
            $this->quantity = $value;
        }
    }

    public function getUnfulfilledQuantityAttribute()
    {
        if (! $this->product_id) {
            return 0;
        }

        return $this->quantity - $this->purchaseOrderShipmentLines->sum('quantity');
    }

    public function getUninvoicedQuantityAttribute()
    {
        return $this->quantity - $this->purchaseInvoiceLines->sum('quantity_invoiced');
    }

    public function getLastReceiptDateAttribute()
    {
        /** @var PurchaseOrderShipmentReceiptLine $latestReceiptLine */
        $latestReceiptLine = $this->purchaseOrderShipmentReceiptLines()->latest()->first();
        if ($latestReceiptLine) {
            return $latestReceiptLine->purchaseOrderShipmentReceipt->received_at;
        }

        return null;
    }

    public function setNominalCodeAttribute($value)
    {
        $this->nominal_code_id = empty($value) ? null : NominalCode::with([])->where('code', $value)->value('id');
    }

    public function setSkuAttribute($value)
    {
        $this->product_id = empty($value) ? null : Product::with([])->where('sku', $value)->value('id');
    }

    public function getReference(): ?string
    {
        return $this->purchaseOrder->getReference();
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * @throws Throwable
     */
    public function save(array $options = [])
    {
        $product = Product::find($this->product_id);
        if ($product && $product->type !== Product::TYPE_STANDARD) {
            return false;
        }

        if (empty($this->nominal_code_id)) {
            if ($this->product_id) {
                $this->nominal_code_id = $this->product->cogs_nominal_code_id ?: Helpers::setting(Setting::KEY_NC_MAPPING_COGS);
            } else {
                $this->nominal_code_id = Helpers::setting(Setting::KEY_NC_MAPPING_COGS);
            }
        }

        $quantityChanged = $this->isDirty('quantity');

        $saved = parent::save($options);

        if($saved){

            if (($quantityChanged || $this->wasRecentlyCreated) && $this->purchaseOrder->order_status === PurchaseOrder::STATUS_OPEN && ! is_null($this->product_id) && !empty($this->id)) {
                dispatch(new SyncBackorderQueueCoveragesJob(purchaseOrderLineIds: Arr::wrap($this->id)));
            }
        }

        return $saved;
    }

    public function coveredBackorderQueues(): HasMany
    {
        return $this->hasMany(BackorderQueueCoverage::class);
    }

    public function getBackorderQueueCoverageQuantityAttribute()
    {
        return $this->coveredBackorderQueues->sum('unreleased_quantity');
    }

    public function getAvailableQuantityForBackordersAttribute()
    {
        return $this->unreceived_quantity - $this->backorder_queue_coverage_quantity;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return ['receipt_status', 'description', 'is_product', 'quantity', 'discount', 'amount', 'estimated_delivery_date', 'nominal_code_id'];
    }

    public function scopeFilterIsProduct(Builder $builder, array $relation, string $operator, $value, $conjunction): Builder
    {
        $function = ($operator == '=' && $value == 'true') ? 'whereHas' : 'whereDoesntHave';

        return $builder->{$function}('product');
    }

    public function scopeFilterReceiptStatus(Builder $builder, array $relation, string $operator, $value, $conjunction): Builder
    {
        if ($operator == '=' && $value == 'fully_received') {
            return $builder->whereHas('purchaseOrderShipmentReceiptLines', function ($query) {
                $query->whereColumn('purchase_order_shipment_receipt_lines.quantity', 'purchase_order_lines.quantity');
            });
        } elseif ($operator == '=' && $value == 'partially_received') {
            return $builder->whereHas('purchaseOrderShipmentReceiptLines', function ($query) {
                $query->whereColumn('purchase_order_shipment_receipt_lines.quantity', '!=', 'purchase_order_lines.quantity');
            });
        } elseif ($operator == '=' && $value == 'not_received') {
            return $builder->whereDoesntHave('purchaseOrderShipmentReceiptLines');
        } else {
            return $builder;
        }
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['sku', 'description', 'barcode', 'receipt_status'];
    }

    /**
     * @deprecated use BackorderManager::coverBackorderQueues() instead
     */
    public function reduceBackorderCoveragesToMatchQuantity()
    {
        if ($this->coveredBackorderQueues()->count() === 0) {
            return;
        }

        $coverages = $this->coveredBackorderQueues()
            ->with(['backorderQueue'])
            ->get()
            ->sortBy('backorderQueue.priority');
        $quantity = $this->unreceived_quantity;
        $updated = [];
        foreach ($coverages as $coverage) {
            // dd($this->unreceived_quantity, $coverage->toArray());
            /** @var BackorderQueueCoverage $coverage */
            if ($coverage->unreleased_quantity >= $quantity) {
                /**
                 * The active covered quantity on the coverage
                 * already exhausts the unreceived quantity on
                 * the purchase order line, we make sure the coverage
                 * quantity matches and abort (essentially getting rid
                 * of the other coverages).
                 */
                $coverage->covered_quantity = min($quantity + $coverage->released_quantity, $this->quantity);
                $coverage->save();
                if ($coverage->covered_quantity == 0) {
                    $coverage->delete();
                } else {
                    $updated[] = $coverage->id;
                }
                break;
            } else {
                $quantity -= $coverage->unreleased_quantity;
                $updated[] = $coverage->id;
            }
        }

        // Synchronise
        $this->coveredBackorderQueues()
            ->whereNotIn('id', $updated)
            ->delete();
    }

    /**
     * Get purchase order line by storing request data.
     */
    public static function findByStoreRequest(?int $purchaseOrderId, array $orderLine): ?self
    {
        $purchaseOrderLine = null;

        if (! empty($orderLine['id']) || ! empty($orderLine['purchase_order_line_id'])) {
            $purchaseOrderLine = self::with([])->find($orderLine['id'] ?? $orderLine['purchase_order_line_id']);
        }

        // Check if sku is provided rather than product_id then fetch the product.
        if (! $purchaseOrderLine && isset($orderLine['sku']) && ! isset($orderLine['product_id'])) {
            $orderLine['product_id'] = Product::with([])->where('sku', $orderLine['sku'])->firstOrFail()->id;
        }

        // Try update with the description
        if (! $purchaseOrderLine && $purchaseOrderId && ! empty($orderLine['product_id'])) {
            $purchaseOrderLine = self::with([])->where('purchase_order_id', $purchaseOrderId)
                ->where('product_id', $orderLine['product_id'])
                ->first();
        }

        // custom line by description
        if (! $purchaseOrderLine && empty($orderLine['sku']) && empty($orderLine['product_id']) && ! empty($orderLine['description'])) {
            $purchaseOrderLine = self::with([])->where('purchase_order_id', $purchaseOrderId)
                ->where('description', $orderLine['description'])
                ->first();
        }

        return $purchaseOrderLine;
    }

    /**
     * {@inheritDoc}
     */
    public function getTransactionCurrency(): Currency
    {
        return $this->purchaseOrder->currency;
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeStartDate(Builder $query, $date): Builder
    {
        return $query->where('purchase_orders.purchase_order_date', '>=', Carbon::parse($date));
    }

    public function scopeEndDate(Builder $query, $date): Builder
    {
        return $query->where('purchase_orders.purchase_order_date', '<', Carbon::parse($date)->addDay());
    }

    public function scopeTrailingDays(Builder $query, int $trailing_days): Builder
    {
        return $query->where('purchase_orders.purchase_order_date', '>=', Carbon::now()->subDays($trailing_days));
    }

    public function scopeWithCostValues($builder, $purchaseOrderId = null, $useDefaultCurrency = true)
    {
        $purchaseOrderLineAmount = 'purchase_order_lines.amount';
        if ($useDefaultCurrency) {
            $purchaseOrderLineAmount = '(purchase_order_lines.amount * currencies.conversion)';
        }

        $alias = 'purchase_orders_computed';
        $totalProductCost = $alias.'.total_product_cost';
        $proration = "($purchaseOrderLineAmount * purchase_order_lines.quantity) / $totalProductCost";
        $totalOtherCost = $alias.'.total_other_cost';
        $productCost = "(($proration) * $totalProductCost)";
        $otherCost = "(($proration) * ($totalOtherCost))";
        $totalCost = "((($proration) * ($totalProductCost)) + (($proration) * ($totalOtherCost)))-purchase_order_lines.discount_amount_extended";

        $query = self::selectRaw('purchase_order_cost_query.id as purchase_order_id')
            ->selectRaw(($useDefaultCurrency ? '(purchase_order_cost_query.total_cost * currencies.conversion)' : 'purchase_order_cost_query.total_cost').' as total_cost')
            ->selectRaw('SUM(
			            CASE 
                            WHEN purchase_order_lines.product_id IS NULL 
                            THEN (purchase_order_lines.quantity * '.$purchaseOrderLineAmount.')
		                    ELSE
			                0
			            END
                    ) AS total_other_cost')
            ->selectRaw('SUM(
			            CASE 
                            WHEN purchase_order_lines.product_id IS NOT NULL 
                            THEN (purchase_order_lines.quantity * '.$purchaseOrderLineAmount.')
                            ELSE 0
                            END
                        ) AS total_product_cost')
            ->leftJoinRelationship('purchaseOrder', function ($join) {
                $join->as('purchase_order_cost_query');
            })
            ->leftJoin('currencies', function ($join) {
                $join->on('purchase_order_cost_query.currency_id', 'currencies.id');
            });

        if (! is_null($purchaseOrderId)) {
            $query->where('purchase_order_id', $purchaseOrderId);
        }

        $query->groupBy('purchase_order_cost_query.id', 'total_cost');

        $builder->withExpression('purchase_orders_computed', $query)
            ->selectRaw('purchase_order_lines.*')
            ->selectRaw("$productCost as product_cost")
            ->selectRaw("$otherCost as other_cost")
            ->selectRaw("$totalCost as total_cost")
            ->leftJoin('purchase_orders', function ($join) {
                $join->on('purchase_orders.id', 'purchase_order_lines.purchase_order_id');
            })
            ->leftJoin('currencies', function ($join) {
                $join->on('purchase_orders.currency_id', 'currencies.id');
            })
            ->join('purchase_orders_computed', function ($join) {
                $join->on((new self())->getTable().'.purchase_order_id', '=', 'purchase_orders_computed.purchase_order_id');
            });

        return $builder;
    }

    public function scopeWithReceivedQuantity($builder, $productIdsTable)
    {
        $receivedLinesQuantityQuery = self::selectRaw('SUM(purchase_order_shipment_receipt_lines.quantity) as received_quantity')
            ->selectRaw('purchase_order_lines.id as purchase_order_line_id')
            ->leftJoinRelationship('purchaseOrderShipmentReceiptLines')
            ->leftJoinRelationship('purchaseOrderShipmentLine.purchaseOrderLine.product')
            ->whereIn('products.id', function ($query) use ($productIdsTable) {
                $query->select('product_id')->from($productIdsTable);
            })
            ->groupBy('purchase_order_lines.id');

        $builder->leftJoinSub($receivedLinesQuantityQuery, 'received_quantity_subquery', function ($join) {
            $join->on('purchase_order_lines.id', '=', 'received_quantity_subquery.purchase_order_line_id');
        })
            ->whereIn('product_id', function ($query) use ($productIdsTable) {
                $query->select('product_id')->from($productIdsTable);
            });

        return $builder;
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->purchase_order_id;
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [
            'id' => $this->id,
            'sku' => $this->product?->sku,
            'description' => $this->description,
        ];
    }

    // Return nominal code id based on specific to general logic for settings
    public function getNominalCodeId(?Product $product)
    {
        $product = $product ?? $this->product;
        if ($product && $product->cogs_nominal_code_id) {
            return $product->cogs_nominal_code_id;
        }

        return Helpers::setting(Setting::KEY_NC_MAPPING_COGS);
    }
}
