<?php

namespace App\Models;

use App\Exporters\MapsExportableFields;
use App\Importers\DataImporter;
use App\Importers\DataImporters\ProductPricingTierDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\Magento\CustomerGroup;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * Class ProductPricingTier.
 *
 *
 * @property int $id
 * @property string $name
 * @property string $currency_code
 * @property bool $is_default
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $archived_at
 */
class ProductPricingTier extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable
{
    use Archive, HasFactory, HasFilters, HasSort;

    protected $casts = ['is_default' => 'boolean', 'archived_at' => 'datetime'];

    private static $default = null;

    protected $fillable = ['name', 'is_default'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function products()
    {
        return $this->belongsToMany(Product::class, 'product_pricing')
            ->using(ProductPricing::class)
            ->withPivot('price');
    }

    public function productPrices()
    {
        return $this->hasMany(ProductPricing::class);
    }

    public function magentoCustomerGroups()
    {
        return $this->hasMany(CustomerGroup::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function save(array $options = [])
    {
        if (empty($this->currency_code)) {
            $this->currency_code = Currency::default()->code;
        }

        if ($this->is_default) {
            // reset default product pricing tier
            if (($this->exists && ! $this->getOriginal('is_default')) || ! $this->exists) {
                self::with([])->update(['is_default' => false]);
            }
        }

        return parent::save($options);
    }

    /**
     * {@inheritDoc}
     */
    public function delete()
    {
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        $this->products()->detach();

        return parent::delete();
    }

    public function isUsed()
    {
        $usage = [];
        if ($this->is_default) {
            $usage['is_default'] = __('messages.product.delete_pricing_failed', ['id' => $this->name]);
        }

        $this->loadCount('productPrices');
        if ($this->product_prices_count) {
            $usage['products'] = trans_choice('messages.currently_used', $this->product_prices_count, [
                'resource' => 'product',
                'model' => 'product pricing tier('.$this->name.')',
            ]);
        }

        // is it used in integration instance import mappings
        $mappingCount = DataImportMapping::query()->where('mapping.sku_field', "price.{$this->name}.value")->count();
        if ($mappingCount) {
            $usage['import_mappings'] = trans_choice('messages.currently_used', $mappingCount, [
                'resource' => 'integration instance listing mapping',
                'model' => 'product pricing tier('.$this->name.')',
            ]);
        }

        // is it used by magento customer groups
        $this->loadCount('magentoCustomerGroups');
        if ($this->magento_customer_groups_count) {
            $usage['magento_customer_groups_count'] = trans_choice('messages.currently_used', $this->magento_customer_groups_count, [
                'resource' => 'magento customer group',
                'model' => 'product pricing tier('.$this->name.')',
            ]);
        }

        return count($usage) ? $usage : false;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.product_pricing_tier.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name', 'currency_code'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new ProductPricingTierDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return ProductPricingTierDataImporter::getExportableFields();
    }

    public static function default()
    {
        if (! static::$default) {
            static::$default = static::with([])->where('is_default', true)->first() ?: self::with([])->first();
        }

        return static::$default;
    }
}
