<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Data\IntegrationInstanceInventoryData;
use App\Data\InventoryLocationData;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Models\Concerns\Archive;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Kirschbaum\PowerJoins\PowerJoins;
use Modules\Amazon\Entities\AmazonFbaReportInventory;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\WooCommerce\Entities\WooCommerceProduct;

/**
 * Class ProductListing.
 *
 *
 * @property int $id
 * @property int $product_id
 * @property string $title
 * @property int $sales_channel_id
 * @property string $sales_channel_listing_id
 * @property string|null $listing_sku
 * @property float $price
 * @property float $product_price;
 * @property int|null $product_pricing_tier_id
 * @property int|null $fulfillment_latency
 * @property float|null $proforma_marketplace_cost_percentage
 * @property string|null $master_of_price
 * @property float|null $quantity
 * @property string|null $master_of_stock
 * @property float|null $sales_channel_qty
 * @property Carbon|null $sales_channel_qty_last_updated
 * @property bool $is_fba
 * @property array|null $inventory_rules
 * @property string $document_id
 * @property string $document_type
 * @property array $metadata,
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read array $sales_channel_settings
 * @property-read SalesChannel $salesChannel
 * @property-read SalesOrderLine[]|Collection $salesOrderLines
 * @property-read ProductPricingTier $productPricingTier
 * @property-read Product $product
 * @property-read string $general_master_price
 * @property-read string $general_master_qty
 * @property-read array $general_pricing_tier
 * @property-read string $active_master_of_price
 * @property-read number $active_price
 * @property-read string $active_master_of_stock
 * @property-read int $active_pricing_tier_id
 * @property-read int[] $active_warehouses_ids
 * @property-read array $active_inventory_rules
 * @property-read int $active_fulfillment_latency
 * @property-read \App\Models\Shopify\ShopifyProduct $document
 * @property-read \App\Models\Shopify\ShopifyProduct $shopifyListing
 * @property-read AmazonProduct $amazonListing
 */
class ProductListing extends Model implements Filterable, Sortable, UniqueFieldsInterface
{
    use Archive, BulkImport, HasFilters, HasSort, PowerJoins;
    use HasFactory;

    const BULK_THRESHOLD = 50;

    /**
     * Masters of price and stock.
     */
    const MASTER_SKU = 'sku.io';

    const MASTER_SALES_CHANNEL = 'sales_channel';

    const MASTER_NEITHER = 'neither';

    const MASTER_AMAZON = 'amazon';

    const MASTERS = [
        self::MASTER_SKU,
        self::MASTER_SALES_CHANNEL,
        self::MASTER_NEITHER,
    ];

    protected $fillable = [
        'sales_channel_id',
        'product_id',
        'master_of_price',
        'product_pricing_tier_id',
        'master_of_stock',
        'sales_channel_listing_id',
        'is_fba',
        'document_id',
        'document_type',
        'title',
        'fulfillment_latency',
        'proforma_marketplace_cost_percentage',
        'inventory_rules',
        'listing_sku',
        'sales_channel_qty',
        'sales_channel_qty_last_updated',
        'price',
        'metadata',
    ];

    protected $casts = [
        'last_updated_competitive_price' => 'datetime',
        'competitive_price' => 'array',
        'amazon_fees_estimates' => 'array',
        'is_fba' => 'boolean',
        'inventory_rules' => 'array',
        'quantity' => 'int',
        'sales_channel_qty' => 'int',
        'sales_channel_qty_last_updated' => 'datetime',
        'metadata' => 'array',
    ];

    protected $appends = [];

    // TODO: Change unique index to just sales_channel_id / sales_channel_listing_id once SKU-6290 gets pushed
    //  It is currently blocked by Amazon incorrectly using ASIN for sales_channel_listing_id which is a non unique field
    public static function getUniqueFields(): array
    {
        return ['sales_channel_id', 'sales_channel_listing_id', 'listing_sku'];
    }

    /**
     * Add document relations to product listings.
     */
    public static function loadDocuments(
        Collection $productListings,
        ?IntegrationInstance $integrationInstance = null
    ): Collection {
        if ($productListings->isEmpty()) {
            return $productListings;
        }

        if (! $integrationInstance) {
            $integrationInstance = $productListings->first()->salesChannel->integrationInstance;
        }

        $documents = $integrationInstance->integration->getProductsModelPath()::whereIn(
            '_id',
            $productListings->pluck('document_id')
        )->get();

        return $productListings->map(function (self $productListing) use ($integrationInstance, $documents) {
            $document = $documents->firstWhere('_id', $productListing->document_id);
            if ($document) {
                $document->setRelation('integrationInstance', $integrationInstance);
            }

            return $productListing->setRelation('document', $document);
        });
    }

    /**
     * Relations.
     */
    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function salesChannel()
    {
        return $this->belongsTo(SalesChannel::class);
    }

    public function productPricingTier()
    {
        return $this->belongsTo(ProductPricingTier::class);
    }

    public function productPricing()
    {
        return $this->hasMany(ProductPricing::class, 'product_id', 'product_id');
    }

    public function delete()
    {
        customlog('SKU-6200', 'Product listing ID '.$this->id.' sales channel listing id '.$this->sales_channel_listing_id.' '.$this->listing_sku.' is being deleted.', [
            'debug' => debug_pretty_string(),
        ], 7);
        // Restrict unmapping/deletion for any listing associated to a sales order line (SKU-2356)
        if ($this->is_fba) {
            if ($this->salesOrderLines()->count() > 0) {
                return false;
            }
        }

        if ($this->salesChannel->integrationInstance->integration->name === Integration::NAME_EBAY) {
            return parent::delete();
        }

        // unset the productListing reference in the listing model
        $listingPath = $this->salesChannel->integrationInstance->integration->getProductsModelPath();

        if ($listingPath !== AmazonProduct::class && $listingPath !== WooCommerceProduct::class) {
            $listingPath::where('product', $this->id)->update(['product' => null]);
        }

        return parent::delete();
    }

    public function salesOrderLines()
    {
        return $this->hasMany(SalesOrderLine::class);
    }

    public function isBundle(): bool
    {
        return $this->product->type === Product::TYPE_BUNDLE;
    }

    /**
     * Get filterable columns, must be inside available columns.
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * Get available columns to show in datatable.
     */
    public function availableColumns(): array
    {
        return config('data_table.product_listing.columns');
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['sales_channel_listing_id'];
    }

    public function document(): MorphTo
    {
        return $this->morphTo('document');
        //        $model = $this->salesChannel->integration->getProductsModelPath();
        //        if ($model) {
        //            return $this->belongsTo($model, 'document_id');
        //        }
    }

    public function ebayLegacyProduct()
    {
        return $this->belongsTo(EbayLegacyProduct::class, 'document_id');
    }

    public function amazonOrder()
    {
        return $this->belongsTo(AmazonProduct::class, 'document_id');
    }

    public function shopifyListing()
    {
        $model = (new Integration())->forceFill(['name' => Integration::NAME_SHOPIFY])->getProductsModelPath();

        return $this->belongsTo($model, 'document_id');
    }

    public function amazonListing()
    {
        $model = (new Integration())->forceFill(['name' => Integration::NAME_AMAZON_US])->getProductsModelPath();

        return $this->belongsTo($model, 'document_id');
    }

    public function getInventoryRulesData(): IntegrationInstanceInventoryData
    {
        return IntegrationInstanceInventoryData::from($this->inventory_rules ?? []);
    }

    public function getProductPriceAttribute()
    {
        if ($this->master_of_price == 'sku.io') {
            $productPricing = ProductPricing::with([])->where(
                'product_pricing_tier_id',
                $this->product_pricing_tier_id
            )->where('product_id', $this->product_id)->first();

            return $productPricing ? $productPricing->price : null;
        }

        return $this->document ? $this->document->price : null;
    }

    public function getGeneralMasterPriceAttribute()
    {
        $settings = $this->salesChannel->integrationInstance->integration_settings;

        if (isset($settings['pricing']['masterOfPrice']['name'])) {
            return $settings['pricing']['masterOfPrice']['name'];
        }
    }

    public function getGeneralMasterQtyAttribute()
    {

        $inventoryData = $this->salesChannel->integrationInstance->getInventoryData();

        if (
            $this->salesChannel->integrationInstance->isShopify() ||
            $this->salesChannel->integrationInstance->isMagento()
        ) {
            return $inventoryData->locations->count() > 0 ? $inventoryData->locations->first()->masterOfStock : null;
        }

        return $inventoryData->masterOfStock;
    }

    public function getGeneralPricingTierAttribute()
    {
        $settings = $this->salesChannel->integrationInstance->integration_settings;

        if (isset($settings['pricing']['pricingTier'])) {
            return $settings['pricing']['pricingTier'];
        }
    }

    public function getRawListingAttribute()
    {
        $type = SalesChannelType::getTypeByID($this->salesChannel->sales_channel_type_id);

        $model = 'App\\Model\\'.$type.'\\Product';

        if ($type == SalesChannelType::TYPE_AMAZON) {
            $marketPlaceId = $this->salesChannel->integration_settings['marketplaceId'] ?? Amazon::MARKETPLACE_US;

            $model .= Amazon::MARKETPLACES[$marketPlaceId];

            $model .= $this->is_fba ? 'FBA' : '';
        }

        return $this->salesChannelListing($model)->first();
    }

    public function salesChannelListing($model)
    {
        return $this->hasOne($model, 'listing_id', 'sales_channel_listing_id');
    }

    public function getSalesChannelSettingsAttribute()
    {
        return $this->salesChannel->integrationInstance->integration_settings ?? [];
    }

    public function getActiveMasterOfPriceAttribute()
    {
        return $this->master_of_price ?: $this->general_master_price;
    }

    public function getActivePriceAttribute()
    {
        if ($this->general_master_price == 'Neither' || $this->general_master_price == 'Amazon') {
            return $this->document ? $this->document['price'] : $this->price;
        }

        return $this->price;
    }

    public function getActiveMasterOfStockAttribute()
    {
        return $this->master_of_stock ?: $this->general_master_qty;
    }

    public function getActivePricingTierIdAttribute()
    {
        return $this->master_of_price ? $this->product_pricing_tier_id : ($this->general_pricing_tier['id'] ?? null);
    }

    public function getActiveInventoryRulesAttribute()
    {
        // not inherited
        if ($this->master_of_stock) {
            return $this->inventory_rules;
        }

        $inventoryData = $this->salesChannel->integrationInstance->getInventoryData();

        // inherited
        if (
            $this->salesChannel->integrationInstance->isShopify() ||
            $this->salesChannel->integrationInstance->isMagento()
        ) {
            return $inventoryData->locations->count() > 0 ? $inventoryData->locations->first() : [];
        }

        return $inventoryData->toArray();
    }

    public function getActiveWarehousesIdsAttribute()
    {
        return $this->active_inventory_rules['selectedWarehouses'] ?? [];
    }

    public function getActiveFulfillmentLatencyAttribute()
    {
        return $this->master_of_stock ? ($this->inventory_rules['fulfillmentLatency'] ?? null) : ($this->sales_channel_settings['inventory']['fulfillmentLatency'] ?? null);
    }

    public function getActiveSubstractBufferStockAttribute()
    {
        return $this->master_of_stock ? ($this->inventory_rules['substractBufferStock'] ?? 0) : ($this->sales_channel_settings['inventory']['substractBufferStock'] ?? 0);
    }

    public function save(array $options = [])
    {
        // set Price
        if ($this->wasRecentlyCreated || ($options['set_price'] ?? false)) {
            if (strtolower($this->active_master_of_price) == strtolower(self::MASTER_SKU)) {
                $productPricing = ProductPricing::with([])

                    ->where('product_pricing_tier_id', $this->active_pricing_tier_id)
                    ->where('product_id', $this->product_id)
                    ->first();
                $this->product_pricing_tier_id = $productPricing->product_pricing_tier_id;
                $this->price = $productPricing?->price;
            } elseif (strtolower($this->active_master_of_price) == strtolower(self::MASTER_AMAZON) || strtolower($this->active_master_of_price) == strtolower(self::MASTER_SALES_CHANNEL)) {
                $productModel = $this->salesChannel->integration->getProductsModelPath();
                $product = $productModel::with([])->where($productModel::SELLER_SKU_COLUMN, $this->listing_sku)
                    ->where('integration_instance_id', $this->salesChannel->integration_instance_id)
                    ->first();

                $this->price = $product?->price;
            } else {
                $this->price = null;
            }
        }
        if ($this->wasRecentlyCreated || ($options['set_quantity'] ?? false)) {
            $this->saveInventoryLocations(request()->inventoryLocations);
        }

        // set Fulfillment Latency
        $this->fulfillment_latency = $this->active_fulfillment_latency;

        // reset pricing tier if inherited
        if (! $this->master_of_price) {
            $this->product_pricing_tier_id = null;
        }

        return parent::save($options);
    }

    private function saveInventoryLocations($inventoryLocations): void
    {
        if ($this->wasRecentlyCreated && empty($inventoryLocations)) {

            $inventoryData = IntegrationInstanceInventoryData::from(
                $this->sales_channel_settings['inventory'] ?? []
            );

            $inventoryLocations = collect($inventoryData->locations)->map(function (InventoryLocationData $location) {
                return [
                    'id' => $location->id,
                    'inventory_rules' => [],
                    'master_of_stock' => null,
                ];
            });
        }
        if (is_null($inventoryLocations)) {
            return;
        }

        $inventoryLocations = InventoryLocationData::collection($inventoryLocations);

        /** @var InventoryLocationData $locationData */
        foreach ($inventoryLocations as $locationData) {

            /** @var ProductListingInventoryLocation $location */
            $location = ProductListingInventoryLocation::updateOrCreate(
                [
                    'product_listing_id' => $this->id,
                    'sales_channel_location_id' => $locationData->id,
                ],
                [
                    'inventory_rules' => $locationData,
                    'quantity' => 0,
                    'fulfillment_latency' => $locationData->fulfillmentLatency ?? null,
                    'master_of_stock' => $locationData->masterOfStock ?? null,
                ]
            );

            if (! $location->master_of_stock) {
                $location->fulfillment_latency = null;
                $location->inventory_rules = null;
            }

            $location->save();
            dispatch(
                new GenerateCacheProductListingQuantityJob(
                    integrationInstance: $this->salesChannel->integrationInstance,
                    productIds: [$this->product_id],
                )
            );

        }
    }

    public function inventoryLocations()
    {
        return $this->hasMany(ProductListingInventoryLocation::class);
    }

    public function scopeForProductIds($builder, string $tempTable)
    {
        $builder->whereIn('product_listings.product_id', function ($query) use ($tempTable) {
            $query->select('product_id')->from($tempTable);
        });

        return $builder;
    }

    public function scopeForProductInventoryRules($builder, IntegrationInstance $integrationInstance)
    {
        $instanceInventoryRules = json_encode($integrationInstance->getInventoryData());

        $queryToJoin = ProductInventory::query()->join(
            'product_listings',
            function ($join) use ($instanceInventoryRules, $integrationInstance) {
                $join->on('product_listings.product_id', '=', 'products_inventory.product_id')
                    ->where('sales_channel_id', $integrationInstance->salesChannel->id)
                    ->whereRaw("JSON_CONTAINS(JSON_EXTRACT(IFNULL(`product_listings`.`inventory_rules`, '$instanceInventoryRules'), '$.selectedWarehouses'),CAST(`products_inventory`.`warehouse_id` AS JSON))");
            }
        )->groupBy('products_inventory.product_id')
            ->select([
                'products_inventory.product_id',
                DB::raw('sum(`inventory_available`) inventory_available'),
            ]);

        $builder->leftJoinSub($queryToJoin, 'pi', 'pi.product_id', 'product_listings.product_id');

        return $builder;
    }
}
