<?php

namespace App\Models;

use App\Abstractions\HasNotesInterface;
use App\Abstractions\UniqueFieldsInterface;
use App\Actions\SetInitialInventory;
use App\Data\InitialInventoryData;
use App\DTO\ReportingDailyFinancialDto;
use App\Exceptions\KitWithMovementsCantBeChangedException;
use App\Exceptions\ProductBundleException;
use App\Exporters\TransformsExportData;
use App\Helpers;
use App\Importers\DataImporter;
use App\Importers\DataImporters\ProductDataImporter;
use App\Importers\ImportableInterface;
use App\Lib\SphinxSearch\SphinxSearch;
use App\Models\Abstractions\AbstractReportable;
use App\Models\Concerns\Archive;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasNotesTrait;
use App\Models\Concerns\HasSort;
use App\Models\Concerns\LogsActivity;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Queries\ProductIdsByAttribute;
use App\Repositories\ProductInventoryRepository;
use App\Repositories\SalesOrderLineFinancialsRepository;
use App\Repositories\SupplierProductPricingRepository;
use App\Services\Product\CreateUpdateProductService;
use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Kirschbaum\PowerJoins\PowerJoins;
use Modules\Ebay\Entities\EbayProductSetting;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInventory;
use Spatie\Activitylog\LogOptions;
use Spatie\Tags\HasTags;
use Throwable;

/**
 * Class Product.
 *
 *
 * @property int $id
 * @property int|null $parent_id
 * @property int|null $brand_id
 * @property string $type
 * @property string|null $sku
 * @property string|null $name
 * @property string|null $barcode
 * @property string|null $mpn
 * @property float $weight
 * @property string $weight_unit
 * @property float $length
 * @property float $width
 * @property float $height
 * @property float|null $proforma_shipping_cost
 * @property float|null $proforma_landed_cost_percentage
 * @property float|null $proforma_marketplace_cost_percentage
 * @property string $dimension_unit
 * @property string|null $fba_prep_instructions
 * @property float|null $case_quantity
 * @property float|null $case_length
 * @property float|null $case_width
 * @property float|null $case_height
 * @property string|null $case_dimension_unit
 * @property float|null $case_weight
 * @property string|null $case_weight_unit
 * @property float|null $average_cost
 * @property int|null $sales_nominal_code_id
 * @property int|null $cogs_nominal_code_id
 * @property float|null $daily_average_consumption
 * @property float|null $supplier_price
 * @property float|null $unit_cost
 * @property bool $is_variation
 * @property bool $is_dropshippable
 * @property array $shared_children_attributes
 * @property-read float $price
 * @property Carbon|null $archived_at
 * @property Carbon|null $deleted_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read Collection|FifoLayer[] $activeFifoLayers
 * @property-read Collection|InventoryMovement[] $inventoryMovements
 * @property-read Collection|SalesOrderLine[] $salesOrderLines
 * @property-read string $image
 * @property-read  SupplierProduct|null $defaultSupplierProduct
 * @property-read  SupplierProduct|SupplierProduct[] $supplierProducts
 * @property-read Attribute|Collection $productAttributes
 * @property-read Collection|PurchaseOrderLine[] $purchaseOrderLines
 * @property-read Collection|PurchaseOrderLine[] $purchaseOrderLinesOpened
 * @property-read Collection|BackorderQueue[] $backorderQueues
 * @property-read Collection|ProductImage[] $images
 * @property-read Collection|ProductAttribute[] $productAttributeValues
 * @property-read Collection|Product[] $variations
 * @property-read ProductBlemished|null $productBlemished
 * @property-read Product|null $parent
 * @property-read ProductImage|null $primaryImage
 * @property-read Collection|ProductInventory[] $productInventory
 * @property-read Collection|InventoryAdjustment[] $inventoryAdjustments
 * @property-read Collection|InventoryAssemblyLine[] $inventoryAssemblyLines
 * @property-read Collection|ProductListing[] $productListings
 * @property-read Collection|SalesCreditLine[] $salesCreditLines
 * @property-read Collection|StockTakeItem[] $stockTakeItems
 * @property-read Collection|WarehouseTransferLine[] $warehouseTransferLines
 * @property-read Collection|ProductBlemished[] $productBlemisheds
 * @property-read Collection|ProductComponent[] $components
 * @property-read Collection|ProductComponent[] $bundles
 * @property-read ShipMyOrdersInventory $shipMyOrdersInventory
 * @property-read Collection|EbayProductSetting[] $ebayProductSettings
 * @property-read Collection|InventoryForecastItem[] $inventoryForecastItems
 * @property-read Collection|Note[] $notes
 */
class Product extends AbstractReportable implements Filterable, HasNotesInterface, ImportableInterface, Sortable, TransformsExportData, UniqueFieldsInterface
{
    use Archive;
    use BulkImport;
    use HasFactory;
    use HasFilters;
    use HasNotesTrait;
    use HasSort;
    use HasTags;
    use LogsActivity;
    use PowerJoins;

    /**
     * Product Types.
     */
    const TYPE_STANDARD = 'standard';

    const TYPE_BUNDLE = 'bundle';

    const TYPE_MATRIX = 'matrix';

    const TYPE_KIT = 'kit';

    const TYPE_BLEMISHED = 'blemished';

    const TYPES = [self::TYPE_STANDARD, self::TYPE_BUNDLE, self::TYPE_KIT, self::TYPE_MATRIX, self::TYPE_BLEMISHED];
    const TYPES_WITH_INVENTORY = [self::TYPE_STANDARD, self::TYPE_KIT, self::TYPE_BLEMISHED];

    /**
     * Product weight units.
     */
    const WEIGHT_UNIT_G = 'g';

    const WEIGHT_UNIT_LB = 'lb';

    const WEIGHT_UNIT_KG = 'kg';

    const WEIGHT_UNIT_OZ = 'oz';

    const WEIGHT_UNITS = [self::WEIGHT_UNIT_LB, self::WEIGHT_UNIT_G, self::WEIGHT_UNIT_KG, self::WEIGHT_UNIT_OZ];

    /**
     * Product dimension units.
     */
    const DIMENSION_UNIT_INCH = 'in';

    const DIMENSION_UNIT_CM = 'cm';

    const DIMENSION_UNIT_MM = 'mm';

    const DIMENSION_UNITS = [self::DIMENSION_UNIT_INCH, self::DIMENSION_UNIT_CM, self::DIMENSION_UNIT_MM];

    const INVALID_DAILY_AVERAGE_CONSUMPTION_KEY = 'invalid_daily_average_consumption';

    /*
     * Casting
     */
    protected $casts = [
        'name' => 'string',
        'type' => 'string',
        'weight' => 'float',
        'weight_unit' => 'string',
        'length' => 'float',
        'width' => 'float',
        'height' => 'float',
        'archived_at' => 'datetime',
        'deleted_at' => 'datetime',
        'barcode' => 'string',
        'average_cost' => 'float',
        'daily_average_consumption' => 'float',
        'has_fba' => 'boolean',
        'case_weight' => 'float',
        'case_quantity' => 'float',
        'case_length' => 'float',
        'case_width' => 'float',
        'case_height' => 'float',
        'shared_children_attributes' => 'array',
        'unit_cost' => 'float',
        'is_taxable' => 'boolean',
        'is_dropshippable' => 'boolean',
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'parent_id',
        'sku',
        'barcode',
        'mpn',
        'name',
        'brand_id',
        'type',
        'weight',
        'weight_unit',
        'length',
        'width',
        'height',
        'dimension_unit',
        'proforma_shipping_cost',
        'proforma_landed_cost_percentage',
        'proforma_marketplace_cost_percentage',
        'daily_average_consumption',
        'fba_prep_instructions',
        'case_quantity',
        'case_length',
        'case_width',
        'case_height',
        'case_dimension_unit',
        'case_weight',
        'case_weight_unit',
        'sales_nominal_code_id',
        'cogs_nominal_code_id',
        'blemished_sku',
        'unit_cost',
        'is_taxable',
        'is_dropshippable',
        'default_tax_rate_id',
    ];

    protected $attributes = [
        'weight' => 0,
        'weight_unit' => self::WEIGHT_UNIT_LB,
        'length' => 0,
        'width' => 0,
        'height' => 0,
        'dimension_unit' => self::DIMENSION_UNIT_INCH,
    ];

    /**
     * Used in general filter to use fulltext search.
     *
     * @var bool
     */
    //protected $generalSearchByFulltext = false;
    protected $generalSearchByFulltext = true;

    protected $fulltextSearchParams = [
        'index' => 'products',
        'fk_field' => 'products.id',
        'field_weights' => 'sku,60,mpn,20,name,10',
        'order_by_raw' => 'if(products.sku=?,0,1)',
    ];

    public static function boot()
    {
        parent::boot();

        $sphinx_updater = function ($model) {
            $fulltext_index = Arr::get($model->fulltextSearchParams, 'index', false);
            if ($fulltext_index) {
                $data = [
                    'id' => $model->id,
                    'sku' => $model->sku,
                    'name' => $model->name,
                    'barcode' => $model->barcode,
                    'mpn' => (string) $model->mpn,
                    'parent_id' => $model->parent_id,
                    'brand_id' => $model->brand_id,
                ];
                SphinxSearch::updateRecord(env('SPHINX_INDEX_PREFIX', 'sku_').$fulltext_index, $data);
            }
        };
        self::created($sphinx_updater);
        self::updated($sphinx_updater);

        // Not needed here because we have an explicit delete method
        // self::deleted(function($model){
        //     SphinxSearch::deleteRecord($fulltext_index, $this->id);
        // });
    }

    public function getReference(): ?string
    {
        return $this->sku;
    }

    public static function getUniqueFields(): array
    {
        return ['sku'];
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->id;
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [
            'id' => $this->id,
            'sku' => $this->sku,
            'name' => $this->name,
        ];
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
     */

    public function parent()
    {
        return $this->belongsTo(self::class, 'parent_id')->whereNull('parent_id');
    }

    public function variations()
    {
        return $this->hasMany(self::class, 'parent_id');
    }

    public function inventoryAssemblyLine()
    {
        return $this->hasMany(InventoryAssemblyLine::class);
    }

    public function inventoryForecastItems(): HasMany
    {
        return $this->hasMany(InventoryForecastItem::class);
    }

    public function brand()
    {
        return $this->belongsTo(ProductBrand::class, 'brand_id', 'id');
    }

    public function productAttributes(): BelongsToMany
    {
        return $this->belongsToMany(Attribute::class, 'product_attributes')
            ->using(ProductAttribute::class)
            ->withPivot('value')
            ->withTimestamps();
    }

    public function productAttributeValues()
    {
        return $this->hasMany(ProductAttribute::class);
    }

    /**
     * Get all sales order lines of the product.
     */
    public function salesOrderLines()
    {
        return $this->hasMany(SalesOrderLine::class);
    }

    /**
     * Get all sales orders of the product.
     */
    public function salesOrders()
    {
        return $this->belongsToMany(SalesOrder::class, 'sales_order_lines');
    }

    public function inventoryAdjustments(): HasMany
    {
        return $this->hasMany(InventoryAdjustment::class);
    }

    public function purchaseOrderLines()
    {
        return $this->hasMany(PurchaseOrderLine::class);
    }

    public function fifoLayers()
    {
        return $this->hasMany(FifoLayer::class)->orderBy('fifo_layer_date');
    }

    public function backorderQueues(): HasManyThrough
    {
        return $this->hasManyThrough(
            BackorderQueue::class,
            SalesOrderLine::class,
            'product_id',          // Foreign key on sales_order_lines table
            'sales_order_line_id', // Foreign key on backorder_queues table
            'id',                  // Local key on products table
            'id'                   // Local key on sales_order_lines table
        );
    }

    public function activeFifoLayers(bool $useCache = true): Builder|HasMany
    {
        if ($useCache) {
            return $this->hasMany(FifoLayer::class)
                ->where('available_quantity', '>', 0)
                ->orderBy('fifo_layer_date');
        }

        return $this->hasMany(FifoLayer::class)
            ->whereHas('inventoryMovements', function (Builder $query) {
                $query->selectRaw('SUM(quantity) as total_quantity')
                    ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                    ->having('total_quantity', '>', 0);
            })
            ->orderBy('fifo_layer_date');
    }

    public function inventoryMovements()
    {
        return $this->hasMany(InventoryMovement::class);
    }

    public function productPricingTiers()
    {
        return $this->belongsToMany(ProductPricingTier::class, 'product_pricing')
            ->using(ProductPricing::class)
            ->withTimestamps()
            ->withPivot('price');
    }

    public function supplierProducts()
    {
        $relation = $this->hasMany(SupplierProduct::class);

        // overwrite delete function to delete its pricing
        $relation->onDelete(function (Builder $builder) {
            SupplierProductPricing::with([])->whereIn('supplier_product_id',
                $builder->pluck('id')->toArray())->delete();

            return $builder->toBase()->delete();
        });

        return $relation;
    }

    public function suppliers()
    {
        return $this->belongsToMany(Supplier::class, 'supplier_products', 'product_id', 'supplier_id')
            ->using(SupplierProduct::class)
            ->withTimestamps();
    }

    public function components()
    {
        return $this->belongsToMany(self::class, 'product_components', 'parent_product_id',
            'component_product_id')->using(ProductComponent::class)->withPivot([
                'id',
                'quantity',
            ]);
    }

    public function kit()
    {
        return $this->belongsTo(self::class, 'parent_id')->where('type', self::TYPE_KIT);
    }

    public function bundles()
    {
        return $this->belongsToMany(self::class, 'product_components', 'component_product_id',
            'parent_product_id')->using(ProductComponent::class)->withPivot([
                'id',
                'quantity',
            ]);
    }

    public function ProductComponents()
    {
        return $this->hasMany(ProductComponent::class, 'parent_product_id');
    }

    public function productComponentsFromChild()
    {
        return $this->hasMany(ProductComponent::class, 'component_product_id');
    }

    public function parentProducts()
    {
        return $this->belongsToMany(
            self::class,
            'product_components',
            'component_product_id',
            'parent_product_id'
        )->using(ProductComponent::class)->withPivot([
            'id',
            'quantity',
        ]);
    }

    public function inBundlingProducts()
    {
        return $this->hasMany(ProductComponent::class, 'component_product_id');
    }

    public function bundle()
    {
        return $this->hasOneThrough(
            self::class,
            ProductComponent::class,
            'component_product_id',
            'id',
            'id',
            'parent_product_id'
        );
    }

    public function productListings()
    {
        return $this->hasMany(ProductListing::class);
    }

    public function warehouseTransfers()
    {
        return $this->hasMany(WarehouseTransfer::class);
    }

    public function warehouseTransferLines()
    {
        return $this->hasMany(WarehouseTransferLine::class);
    }

    public function defaultSupplierProduct()
    {
        return $this->hasOne(SupplierProduct::class)->where('is_default', 1);
    }

    public function productPricing()
    {
        return $this->hasMany(ProductPricing::class);
    }

    public function categories()
    {
        return $this->belongsToMany(ProductCategory::class, 'product_to_categories', 'product_id', 'category_id')
            ->using(ProductToCategory::class)
            ->withTimestamps()
            ->withPivot('is_primary');
    }

    public function attributeGroups()
    {
        return $this->hasMany(ProductAttribute::class)->whereNotNull('attribute_group_id')
            ->select('attribute_group_id')
            ->groupBy('attribute_group_id');
        //        return $this->belongsToMany( AttributeGroup::class, 'products_to_attribute_groups' )
        //                ->using( ProductToAttributeGroup::class )
        //                ->withTimestamps();
    }

    public function primaryCategory()
    {
        return $this->belongsToMany(ProductCategory::class, 'product_to_categories', 'product_id', 'category_id')
            ->using(ProductToCategory::class)
            ->withTimestamps()
            ->wherePivot('is_primary', true)
            ->withPivot('is_primary');
    }

    public function otherCategories()
    {
        return $this->belongsToMany(ProductCategory::class, 'product_to_categories', 'product_id', 'category_id')
            ->using(ProductToCategory::class)
            ->withTimestamps()
            ->wherePivot('is_primary', false)
            ->withPivot('is_primary');
    }

    public function toCategories()
    {
        return $this->hasMany(ProductToCategory::class);
    }

    public function images()
    {
        $relation = $this->hasMany(ProductImage::class);

        // overwrite delete function to delete its pricing
        $relation->onDelete(function (Builder $builder) {
            return $this->images->each(function (ProductImage $productImage) {
                $productImage->delete();
            });
        });

        return $relation;
    }

    public function otherImages()
    {
        return $this->images()->where('is_primary', false);
    }

    public function primaryImage()
    {
        return $this->hasOne(ProductImage::class)->where('is_primary', true);
    }

    public function blemishedProducts()
    {
        return $this->hasMany(ProductBlemished::class, 'derived_from_product_id');
    }

    public function productBlemished()
    {
        return $this->hasOne(ProductBlemished::class, 'product_id');
    }

    public function suppliersInventory()
    {
        return $this->hasMany(SupplierInventory::class);
    }

    public function salesNominalCode()
    {
        return $this->belongsTo(NominalCode::class, 'sales_nominal_code_id');
    }

    public function cogsNominalCode()
    {
        return $this->belongsTo(NominalCode::class, 'cogs_nominal_code_id');
    }

    public function inventoryAssemblyLines()
    {
        return $this->hasMany(InventoryAssemblyLine::class);
    }

    public function salesCreditLines()
    {
        return $this->hasMany(SalesCreditLine::class);
    }

    public function stockTakeItems()
    {
        return $this->hasMany(StockTakeItem::class);
    }

    public function productBenchmarks()
    {
        return $this->hasMany(InventorySnapshot::class);
    }

    public function reportingProducts()
    {
        return $this->morphMany(ReportingDailyFinancial::class, 'reportable');
    }

    public function shipMyOrdersInventory(): HasOne
    {
        return $this->hasOne(ShipMyOrdersInventory::class, 'SKU', 'sku');
    }

    public function ebayProductSettings(): HasMany
    {
        return $this->hasMany(EbayProductSetting::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Relations to Calculate Quantities
    |--------------------------------------------------------------------------
     */

    public function totalQuantity()
    {
        return $this->hasOne(InventoryMovement::class)
            ->where('inventory_status', '!=', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
            ->select('product_id', DB::raw('sum(quantity) as quantity'))
            ->groupBy(['product_id']);
    }

    public function availableQuantity()
    {
        return $this->hasOne(InventoryMovement::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('type', '!=', InventoryMovement::TYPE_TRANSFER)
            ->select('product_id', DB::raw('sum(quantity) as quantity'))
            ->groupBy('product_id');
    }

    public function reservedQuantity()
    {
        return $this->hasOne(InventoryMovement::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('type', '!=', InventoryMovement::TYPE_TRANSFER)
            ->select('product_id', DB::raw('sum(quantity) as quantity'))
            ->groupBy(['product_id']);
    }

    public function purchaseOrderLinesOpened()
    {
        return $this->hasMany(PurchaseOrderLine::class)->whereHas('purchaseOrder', function (Builder $builder) {
            $builder->where('order_status', PurchaseOrder::STATUS_OPEN);
        })->with(['purchaseOrderShipmentReceiptLines.fifoLayers', 'purchaseOrder']);
    }

    public function inTransitQuantity()
    {
        return $this->productInventory()
            ->where('warehouse_id', '!=', 0)
            ->whereDoesntHave('warehouse', function (Builder $builder) {
                $builder->where('type', Warehouse::TYPE_AMAZON_FBA);
            })
            ->select('product_id', 'warehouse_id', DB::raw('sum(inventory_in_transit) as inventory_in_transit'))
            ->groupBy(['warehouse_id', 'product_id']);
    }

    public function inWarehousesQuantity()
    {
        return $this->productInventory()
            ->where('warehouse_id', '!=', 0)
            ->whereDoesntHave('warehouse', function (Builder $builder) {
                $builder->where('type', Warehouse::TYPE_AMAZON_FBA);
            })
            ->select('product_id', 'warehouse_id', DB::raw('sum(inventory_total) as inventory_total'))
            ->groupBy(['warehouse_id', 'product_id']);
    }

    public function inWarehouseAvailableQuantity()
    {
        return $this->productInventory()
            ->where('warehouse_id', '!=', 0)
            ->whereDoesntHave('warehouse', function (Builder $builder) {
                $builder->where('type', Warehouse::TYPE_AMAZON_FBA);
            })
            ->select('product_id', 'warehouse_id', DB::raw('sum(inventory_available) as inventory_available'))
            ->groupBy(['warehouse_id', 'product_id']);
    }

    public function inWarehousesReservedQuantity()
    {
        return $this->productInventory()
            ->where('warehouse_id', '!=', 0)
            ->whereDoesntHave('warehouse', function (Builder $builder) {
                $builder->where('type', Warehouse::TYPE_AMAZON_FBA);
            })
            ->select('product_id', 'warehouse_id', DB::raw('sum(inventory_reserved) as inventory_reserved'))
            ->groupBy(['warehouse_id', 'product_id']);
    }

    public function inWarehouseTransitQuantity()
    {
        return $this->productInventory()
            ->where('warehouse_id', '!=', 0)
            ->whereDoesntHave('warehouse', function (Builder $builder) {
                $builder->where('type', Warehouse::TYPE_AMAZON_FBA);
            })
            ->select('product_id', 'warehouse_id', DB::raw('sum(inventory_in_transit) as inventory_in_transit'))
            ->groupBy(['warehouse_id', 'product_id']);
    }

    /*
     * TODO: I believe this is wrong because it is adding the total cost of any unfulfilled fifo layer instead of the total unfulfilled cost
     * may need to be sum((total_cost/original_quantity) * (original_quantity - fulfilled_quantity))
     * also based on the table structure it is supposed to be per warehouse.  If per warehouse, cannot rely on the fifo layers,
     * but instead have to rebuild using inventory movements
     */
    public function inventoryStockValue()
    {
        return $this->hasOne(FifoLayer::class)
            ->where('available_quantity', '>', 0)
            ->select('product_id', DB::raw('sum(total_cost) as total_cost'))
            ->groupBy('product_id');
    }

    public function salesQuantity()
    {
        return $this->hasOne(SalesOrderLine::class)
            ->whereDate('created_at', '>=',
                Carbon::now()->subDays(Helpers::setting(Setting::KEY_DAYS_SALES_HISTORY, 90)))
            ->select('product_id', DB::raw('sum(quantity) as quantity'))
            ->groupBy('product_id');
    }

    public function dailyMovements()
    {
        return $this->hasMany(InventoryMovement::class)
            ->whereDate('inventory_movement_date', '>=',
                Carbon::now()->subDays(Helpers::setting(Setting::KEY_DAYS_SALES_HISTORY, 90)))
            ->select('product_id', DB::raw('Date(inventory_movement_date) as inventory_movement_date'),
                DB::raw('SUM(quantity) as quantity'))
            ->groupBy('product_id', DB::raw('Date(inventory_movement_date)'))
            ->orderBy('inventory_movement_date');
    }

    public function totalQuantityBeforePeriod()
    {
        return $this->totalQuantity()
            ->whereDate('inventory_movement_date', '<',
                Carbon::now()->subDays(Helpers::setting(Setting::KEY_DAYS_SALES_HISTORY, 90)));
    }

    public function purchaseOrderLinesQuantity()
    {
        return $this->hasOne(PurchaseOrderLine::class)
            ->select('product_id', DB::raw('sum(quantity) as quantity'))
            ->groupBy('product_id');
    }

    public function warehouseTransfersQuantity()
    {
        return $this->hasOne(InventoryMovement::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
            ->select('product_id', DB::raw('sum(quantity) as quantity'))
            ->groupBy('product_id');
    }

    public function dailyAverageConsumption()
    {
        $daysSalesHistory = Helpers::setting(Setting::KEY_DAYS_SALES_HISTORY, 90);

        $fromDate = now()->subDays($daysSalesHistory)->format('Y-m-d');

        $days = 'SUM(`inventory_in_stock`)';
        if (! Helpers::setting(Setting::KEY_ADJUST_FORECAST_OUT_OF_STOCK, true)) {
            $days = $daysSalesHistory;
        }

        return $this->hasOne(InventorySnapshot::class)
            ->where('benchmark_date', '>', $fromDate)
            ->select('product_id', DB::raw("(SUM(`daily_sales`) / {$days}) as value"))
            ->groupBy(['product_id']);
    }

    public function warehousesInventory()
    {
        return $this->hasMany(ProductInventory::class)->where('warehouse_id', '!=',
            ProductInventory::$idForTotalWarehouses);
    }

    public function inventory_available_warehouses()
    {
        return $this->hasMany(ProductInventory::class)->where('warehouse_id', '!=',
            ProductInventory::$idForTotalWarehouses);
    }

    public function totalInventory()
    {
        return $this->hasOne(ProductInventory::class)->where('warehouse_id', ProductInventory::$idForTotalWarehouses);
    }

    public function productInventory()
    {
        return $this->hasMany(ProductInventory::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
     */
    public function getDefaultPricingTierNameAttribute()
    {
        $tier = ProductPricingTier::default();

        return $tier ? $tier->name : 'N/A';
    }

    public function getIsVariationAttribute()
    {
        return ! is_null($this->parent_id);
    }

    public function getCurrentFifoLayerForWarehouse(Warehouse $warehouse): ?FifoLayer
    {
        return $this->activeFifoLayers->where('warehouse_id', $warehouse->id)->first();
    }

    public function setWeightUnitAttribute($value)
    {
        if ($value === 'lbs' || $value == 'pounds') {
            $this->attributes['weight_unit'] = self::WEIGHT_UNIT_LB;
        } else {
            /**
             * If empty or null set default value.
             */
            $this->attributes['weight_unit'] = $value ?: self::WEIGHT_UNIT_LB;
        }
    }

    public function setDimensionUnitAttribute($value)
    {
        if ($value === 'inches') {
            $this->attributes['dimension_unit'] = self::DIMENSION_UNIT_INCH;
        } else {
            /**
             * If empty or null set default value.
             */
            $this->attributes['dimension_unit'] = $value ?: self::DIMENSION_UNIT_INCH;
        }
    }

    public function getTotalStockAttribute()
    {
        $productInventory = new ProductInventoryRepository($this);

        return $productInventory->totalAvailable();
    }

    public function getWarehouseStock(Warehouse $warehouse)
    {
        $productInventory = new ProductInventoryRepository($this);

        return $productInventory->available($warehouse);
    }

    public function isBlemished(): bool
    {
        return $this->type === self::TYPE_BLEMISHED;
    }

    public function isStandard(): bool
    {
        return $this->type === self::TYPE_STANDARD;
    }

    public function hasInventoryMovements(): bool
    {
        return $this->inventoryMovements()->count() > 0;
    }

    public function updateBlemishedInfo(array $info)
    {
        if (empty($info)) {
            return;
        }

        $blemished = $this->productBlemished;

        if (array_key_exists('condition', $info)) {
            $blemished->condition = $info['condition'];
        }

        if (array_key_exists('reference', $info)) {
            $blemished->reference = $info['reference'];
        }

        $blemished->save();
    }

    public function getUnitCostAtWarehouse(int $warehouseId): float
    {
        /** @var FifoLayer $activeFifoLayer */
        $activeFifoLayer = $this->activeFifoLayers()
            ->where('warehouse_id', $warehouseId)
            ->where('total_cost', '>', 0)
            ->first();
        if ($activeFifoLayer) {
            return $activeFifoLayer->avg_cost ?? 0;
        }

        // No active fifo layer,
        // we use average cost if there are used fifo layers.
        if ($this->fifoLayers()->count() > 0 && $this->average_cost > 0) {
            return $this->average_cost ?? 0;
        }

        /*
         * At this point we are checking if fifo layers exist at other warehouses
         */
        $otherLayer = FifoLayer::query()
            ->where('product_id', $this->id)
            ->where('total_cost', '>', 0)
            ->orderBy('fifo_layer_date', 'desc')
            ->first();
        if ($otherLayer && $otherLayer->avg_cost > 0) {
            return $otherLayer->avg_cost ?? 0;
        }

        if ($this->unit_cost) {
            return $this->unit_cost;
        }

        $supplierPrice = app(SupplierProductPricingRepository::class)->getSensibleSupplierProductPrice($this);
        if ($supplierPrice > 0) {
            return $supplierPrice;
        }

        return 0;
    }

    /**
     * Get product image from product attributes.
     */
    public function getImageAttribute(): ?string
    {
        return $this->primaryImage->url ?? null;
    }

    public function getHasFbaAttribute()
    {
        return collect($this->productListings)->where('is_fba', 1)->isNotEmpty();
    }

    /**
     * Get Category path.
     */
    public function getCategoryPath(?ProductCategory $category = null, array $category_path = []): array
    {
        if (empty($category) && $this->primaryCategory->isEmpty()) {
            return $category_path;
        }

        // set product primary category as default
        if (empty($category)) {
            $category = $this->primaryCategory->first();
        }

        $category_path[] = $category->only('id', 'name', 'parent_id', 'attributes', 'attributeGroups');
        if (! empty($category->parent)) {
            return $this->getCategoryPath($category->parent, $category_path);
        }

        return $category_path;
    }

    public function getDefaultPricing($defaultPricingTierId)
    {
        return $this->productPricing->firstWhere('product_pricing_tier_id', $defaultPricingTierId);
    }

    /**
     * Get available columns to show in datatable.
     */
    public function availableColumns(): array
    {
        return config('data_table.product.columns');
    }

    /**
     * Get filterable columns to show in datatable.
     */
    public function filterableColumns(): array
    {
        if (Str::startsWith(func_get_arg(0), [
            'price',
            'supplier_pricing',
            'inventory',
            'attributes',
            'default_supplier.',
            'brand.',
            'tags.',
            'parent_product.',
            'category_main.',
            'category_main_path.',
            'category_others.',
            'category_others_path.',
            'suppliersInventory.',
        ])) {
            return func_get_args();
        }

        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['sku', 'name', 'barcode', 'mpn'];
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new ProductDataImporter(null, $filePath);
    }

    public function unreleasedBackorderQueuesAtWarehouse($warehouseId)
    {
        return $this->backorderLayers()
            ->whereColumn('backorder_layers.quantity_backordered', '>', 'backorder_layers.quantity_released')
            ->whereHas('salesOrderLines', function (Builder $builder) use ($warehouseId) {
                $builder->where('warehouse_id', $warehouseId);
            })
            ->orderBy('created_at')
            ->get();
    }

    public function unreleasedBackorderLayersAtWarehouse($warehouseId)
    {
        return $this->backorderLayers()
            ->whereColumn('backorder_layers.quantity_backordered', '>', 'backorder_layers.quantity_released')
            ->whereHas('salesOrderLines', function (Builder $builder) use ($warehouseId) {
                $builder->where('warehouse_id', $warehouseId);
            })
            ->orderBy('created_at')
            ->get();
    }

    /**
     * Get sortable columns.
     */
    public function sortableColumns(): array
    {
        if (Str::startsWith(func_get_arg(0), [
            'price',
            'supplier_pricing',
            'inventory',
            'attributes',
            'default_supplier.',
            'brand.',
            'tags.',
            'parent_product.',
            'category_main.',
            'category_main_path.',
            'category_others.',
            'category_others_path.',
        ])) {
            return func_get_args();
        }

        $sortable = collect($this->availableColumns())
            ->where('sortable', 1)
            ->pluck('data_name')->all();

        // We add other sortable fields based on relations
        return $this->bindRelatedSortableFields($sortable);
    }

    private function bindRelatedSortableFields(array $fields)
    {
        $fields[] = 'brand.name';
        $fields[] = 'default_supplier.name';
        $fields[] = 'category_main.name';

        // Add other fields here
        return $fields;
    }

    /**
     * {@inheritDoc}
     */
    public function save(array $options = [])
    {
        // set name from sku if it was empty when creation.
        if (! $this->exists && empty($this->name)) {
            $this->name = $this->sku;
        }

        if (! $this->exists && empty(trim($this->sku))) {
            throw new \InvalidArgumentException('The sku should not be empty');
        }

        return parent::save($options);
    }

    /**
     * Delete the model from the database.
     *
     * @return bool|null|array
     *
     * @throws Exception
     */
    public function delete()
    {
        // If the model was used from any related relation, we will archive it and its variants
        if ($related = $this->isUsed()) {
            return $related;
        } else { // If not used, we will delete it from the database.
            // delete product attributes and bundle components
            $this->productAttributeValues()->forceDelete();
            $this->ProductComponents()->forceDelete();
            $this->productListings()->delete();
            $this->productPricing()->delete();
            $this->supplierProducts()->delete();
            $this->suppliersInventory()->delete();
            $this->images()->delete();
            $this->variations()->forceDelete();
            $this->productInventory()->delete();
            $this->productBenchmarks()->delete();
            $this->reportingProducts()->delete();
            $this->toCategories()->delete();
            $this->productBlemished()->delete();
            $this->inventoryForecastItems()->delete();

            $fulltext_index = Arr::get($this->fulltextSearchParams, 'index', false);

            if ($fulltext_index) {
                SphinxSearch::deleteRecord($fulltext_index, $this->id);
            }

            return parent::delete();
        }
    }

    /**
     * Archive the model.
     */
    public function archive()
    {
        // is already archived
        if (! empty($this->archived_at)) {
            return false;
        }

        $this->archived_at = Carbon::now();

        $saved = $this->save();

        if ($saved) {
            $this->fireModelEvent('archived', false);

            // archived product's children
            $this->variations()->update(['archived_at' => $this->archived_at]);
        }

        return $saved;
    }

    /**
     * Unarchived the model.
     */
    public function unarchived()
    {
        // is already unarchived
        if (empty($this->archived_at)) {
            return false;
        }

        $this->archived_at = null;

        $saved = $this->save();

        if ($saved) {
            $this->fireModelEvent('unarchived', false);

            // unarchived product's children
            $this->variations()->update(['archived_at' => null]);
        }

        return $saved;
    }

    /**
     * Get incoming quantity to the product from opened purchase order lines.
     */
    public function getIncomingQuantity(): Collection
    {
        $incomingQuantity = collect();
        foreach ($this->purchaseOrderLinesOpened as $purchaseOrderLine) {
            $incoming = $purchaseOrderLine->quantity - $purchaseOrderLine->purchaseOrderShipmentReceiptLines->sum('quantity');

            $incomingQuantity->add([
                'warehouse_id' => $purchaseOrderLine->purchaseOrder->destination_warehouse_id,
                'quantity' => $incoming,
            ]);
        }

        return $incomingQuantity;
    }

    public function getIncomingQuantityAtWarehouse(?int $warehouseId): int|float
    {
        if (empty($warehouseId)) {
            return 0;
        }
        $lines = $this->purchaseOrderLinesOpened()
            ->whereHas('purchaseOrder', function ($q) use ($warehouseId) {
                return $q->where('destination_warehouse_id', $warehouseId);
            })
            ->get();

        $quantity = 0;
        foreach ($lines as $purchaseOrderLine) {
            $quantity += max($purchaseOrderLine->quantity - $purchaseOrderLine->purchaseOrderShipmentReceiptLines->sum('quantity'),
                0);
        }

        return $quantity;
    }

    /**
     * Fire event after restore the model.
     */
    public function restored()
    {
        $this->fireModelEvent('restored', false);
    }

    /**
     * Check the model is used in its related relations.
     */
    public function isUsed(): bool|array
    {
        $relatedRelations = [
            'salesOrderLines',
            'purchaseOrderLines',
            'inventoryMovements',
            'inBundlingProducts',
        ];

        // load related relations and its children relations
        $this->loadCount($relatedRelations);
        $this->load([
            'variations' => function (HasMany $query) use ($relatedRelations) {
                $query->withCount($relatedRelations);
            },
        ]);

        // check any children product is used
        foreach ($this->variations as $variantProduct) {
            if ($related = $this->hasRelated($variantProduct, $relatedRelations)) {
                return $related;
            }
        }

        // check the model
        return $this->hasRelated($this, $relatedRelations);
    }

    private function hasRelated($model, $relatedRelations): bool|array
    {
        $related = [];

        foreach ($relatedRelations as $relatedRelation) {
            $countKeyName = str_replace('-', '_', Str::kebab($relatedRelation)).'_count';
            if ($model->{$countKeyName}) {
                $relatedName = Str::singular(str_replace('-', ' ', Str::kebab($relatedRelation)));

                $related[$relatedRelation] = trans_choice('messages.currently_used', $model->{$countKeyName}, [
                    'resource' => $relatedName,
                    'model' => 'product('.$model->sku.')',
                ]);
            }
        }

        return count($related) ? $related : false;
    }

    /**
     * Set product brand by brand name.
     */
    public function setBrandName(?string $brandName, bool $save = false): bool
    {
        return CreateUpdateProductService::make($this)->setBrandName($brandName, $save);
    }

    /**
     * Set pricing tiers to the product.
     */
    public function setPricing(?array $prices, bool $detaching = true): ?array
    {
        return CreateUpdateProductService::make($this)->setPricing($prices, $detaching);
    }

    /**
     * Set supplier products.
     */
    public function setSupplierProducts(?array $supplierProducts, bool $detaching = true): ?array
    {
        return CreateUpdateProductService::make($this)->setSupplierProducts($supplierProducts, $detaching);
    }

    /**
     * Set bundle components.
     *
     * @throws ProductBundleException
     */
    public function setBundleComponents(?array $components): ?bool
    {
        return CreateUpdateProductService::make($this)->setBundleComponents($components);
    }

    /**
     * Set kit components.
     *
     * @throws ProductBundleException
     * @throws KitWithMovementsCantBeChangedException
     */
    public function setKitComponents(?array $components): ?bool
    {
        return CreateUpdateProductService::make($this)->setKitComponents($components);
    }

    /**
     * Assign product categories.
     */
    public function setCategories(?array $categories, bool $detaching = true): ?array
    {
        return CreateUpdateProductService::make($this)->setCategories($categories, $detaching);
    }

    /**
     * Assign attribute groups.
     */
    public function setAttributeGroups(
        ?array $attributeGroups,
        string $operation = Helpers::OPERATION_SET
    ): array|int|null {
        return CreateUpdateProductService::make($this)->setAttributeGroups($attributeGroups, $operation);
    }

    /**
     * Set attributes.
     */
    public function setProductAttributes(?array $productAttributes, bool $detaching = true): ?array
    {
        return CreateUpdateProductService::make($this)->setProductAttributes($productAttributes, $detaching);
    }

    /**
     * Get default supplier price for the product.
     *
     * @return float | null
     */
    public function getSupplierPriceAttribute(): float
    {
        $defaultSupplierProduct = $this->defaultSupplierProduct;

        if ($defaultSupplierProduct) {
            $supplierProductPricing = $defaultSupplierProduct->supplierProductPricing()->first();
            if ($supplierProductPricing) {
                return $supplierProductPricing->price;
            }
        }
    }

    /**
     * Add images.
     *
     *
     * @throws FileNotFoundException
     * @throws Exception
     */
    public function addImages(?array $images): ?bool
    {
        return CreateUpdateProductService::make($this)->addImages($images);
    }

    /**
     * Set variations.
     *
     *
     * @throws FileNotFoundException
     */
    public function setVariations(?array $variations, ?bool $sync = false): ?bool
    {
        return CreateUpdateProductService::make($this)->setVariations($variations, $sync);
    }

    public function setSharedChildrenAttributes($attributeIds)
    {
        if (is_null($attributeIds) || $this->type !== self::TYPE_MATRIX) {
            return;
        }
        $this->shared_children_attributes = $attributeIds;
        $this->save();
    }

    /**
     * Add variation.
     *
     *
     * @throws FileNotFoundException
     */
    public function addVariation(array $variation)
    {
        CreateUpdateProductService::make($this)->addVariation($variation);
    }

    /**
     * Add/Get product blemished based on the condition.
     */
    public function addBlemished(string $condition, ?string $reference = null): ProductBlemished
    {
        // check product blemished exists with same condition
        /** @var ProductBlemished $productBlemished */
        $productBlemished = $this->blemishedProducts()->where('condition', $condition)->first();

        if ($productBlemished) {
            return $productBlemished;
        }

        // blemished copy from the original
        $blemishedCopy = $this->replicate();
        $blemishedCopy->type = self::TYPE_BLEMISHED;
        $blemishedCopy->sku = $this->sku.'-blemished-'.$condition; // maybe need to settings
        $blemishedCopy->save();

        // create a new product blemished
        $productBlemished = new ProductBlemished();
        $productBlemished->product_id = $blemishedCopy->id;
        $productBlemished->condition = $condition;
        $productBlemished->reference = $reference;
        $this->blemishedProducts()->save($productBlemished);

        return $productBlemished;
    }

    /**
     * @throws Throwable
     */
    public function setInitialInventory(int $warehouse_id, int $quantity, float $unit_cost = 0): void
    {
        (new SetInitialInventory($this, InitialInventoryData::from([
            'warehouse_id' => $warehouse_id,
            'quantity' => $quantity,
            'unit_cost' => $unit_cost,
        ])))->handle();
    }

    /**
     * Gets the total cost of the quantity based on fifo layers.
     *
     *
     * @return float|int|mixed
     */
    public function getTotalCostByFifoLayers($quantity, array $excluded = [])
    {
        $activeLayers = $this->activeFifoLayers->whereNotIn('id', $excluded);
        $cost = 0;
        foreach ($activeLayers as $fifoLayer) {
            /** @var FifoLayer $fifoLayer */
            if ($fifoLayer->available_quantity >= $quantity) {
                // The fifo layer is enough to fulfill the quantity,
                // we do that and exit.
                $cost += $quantity * $fifoLayer->avg_cost;
                break;
            } else {
                // The fifo layer can't fulfill the entire quantity, we
                // fulfill what it can and continue.
                $neededQuantity = min($fifoLayer->available_quantity, $quantity);
                $cost += $neededQuantity * $fifoLayer->avg_cost;
                $quantity -= $neededQuantity;
            }

            if ($quantity <= 0) {
                break;
            }
        }

        return $cost;
    }

    public function getAvailableFifoLayerQuantity($warehouseId): int
    {
        $original = $this->activeFifoLayers()->where('warehouse_id', $warehouseId)->sum('original_quantity');
        $fulfilled = $this->activeFifoLayers()->where('warehouse_id', $warehouseId)->sum('fulfilled_quantity');

        return $original - $fulfilled;
    }

    /**
     * Retrieve Product Cumulative Available Inventory.
     */
    public function getCumulativeAvailableInventory(): float
    {
        // Summation of Unfulfilled quantities in active FIFO Layers
        return $this->activeFifoLayers->sum('available_quantity');
    }

    /**
     * Retrieve Product Cumulative Cost.
     */
    public function getCumulativeCost(): float
    {
        // Summation of actual costs in active FIFO Layers
        return $this->activeFifoLayers->sum('actual_cost');
    }

    public function getWeightedAverageCost(?int $warehouse_id = null): float|bool
    {
        $fifoLayers = $warehouse_id ? $this->fifoLayers->where('warehouse_id', $warehouse_id) : $this->fifoLayers;

        $fifoLayersTotalQuantity = $fifoLayers->sum('original_quantity');

        if ($fifoLayersTotalQuantity == 0) {
            return false;
        }

        return $fifoLayers->sum('total_cost') / $fifoLayersTotalQuantity;
    }

    /**
     * Update Product Average Cost.
     */
    public function updateAverageCost()
    {
        if ($inventoryCount = $this->getCumulativeAvailableInventory()) {
            $this->average_cost = $this->getCumulativeCost() / $inventoryCount;
        } elseif ($this->fifoLayers->isNotEmpty() && $totalQuantity = $this->fifoLayers->sum('original_quantity')) {
            $this->average_cost = $this->fifoLayers->sum('total_cost') / $totalQuantity;
        } else {
            $this->average_cost = 0;
        }

        $this->save();
    }

    public function getExternalImageUrl()
    {
        if ($this->primaryImage && $this->primaryImage->is_external) {
            return $this->primaryImage->url;
        }

        return null;
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
     */

    public function scopeSortInventory(Builder $builder, array $relation, bool $ascending = true)
    {
        return $builder->joinSub(function ($query) use ($relation) {
            $query->from('products_inventory')
                ->join('warehouses', 'warehouses.id', '=', 'products_inventory.warehouse_id')
                ->where('warehouses.name', $relation['key'])
                ->select('product_id', 'inventory_total');
        }, 'productInventory', function ($join) {
            $join->on('products.id', '=', 'productInventory.product_id');
        })
            ->orderBy('productInventory.inventory_total', $ascending ? 'ASC' : 'DESC');
    }

    public function scopeSortInventory_available_warehouses(Builder $builder, array $relation, bool $ascending = true)
    {
        return $builder->leftJoinSub(function ($query) use ($relation) {
            $query->from('products_inventory')
                ->join('warehouses', 'warehouses.id', '=', 'products_inventory.warehouse_id')
                ->where('warehouses.name', $relation['key'])
                ->select('product_id', DB::raw('COALESCE(inventory_available, 0) as inventory_available'));
        }, 'productInventory', function ($join) {
            $join->on('products.id', '=', 'productInventory.product_id');
        })
            ->orderBy('productInventory.inventory_available', $ascending ? 'ASC' : 'DESC');
    }

    public function scopeSortInventoryIncoming(Builder $builder, array $relation, bool $ascending = true)
    {
        $builder
            ->orderByRaw('(SELECT SUM(purchase_order_lines.unreceived_quantity) FROM purchase_order_lines WHERE products.id = purchase_order_lines.product_id AND EXISTS (SELECT 1 FROM purchase_orders WHERE purchase_order_lines.purchase_order_id = purchase_orders.id AND order_status = ?)) '.($ascending ? 'ASC' : 'DESC'),
                [PurchaseOrder::STATUS_OPEN]);
    }

    public function scopeSortPrice(Builder $builder, array $relation, bool $ascending = true)
    {
        // Alias for scopeSortProductPricingTiers (alias used for tests)
        return $this->scopeSortProductPricingTiers($builder, $relation, $ascending);
    }

    public function scopeSortProductPricingTiers(Builder $builder, array $relation, bool $ascending = true)
    {
        static $round = 0;
        $round++;
        // $this->header_print("scopeSortProductPricingTiers","CALLED");
        // $this->header_print("relation",json_encode($relation));

        $allowed_subkeys = ['price', 'currency_code'];

        if (! ($relation['subkey'] && in_array($relation['subkey'], $allowed_subkeys))) {
            return;
        }

        if ($relation['subkey'] == 'price') {
            $field = 'pp.price';
        }

        // /* currency_code is not sortable, actually
        // so this block is only here to illustrate how to add subkeys */
        // elseif( $relation["subkey"] == "currency_code" ) {
        //   $field = "ppt.currency_code";
        // }

        $subquery = "
        (
            select pp.product_id, $field as field from `product_pricing_tiers` ppt
                inner join `product_pricing` pp on ppt.`id` = pp.`product_pricing_tier_id`
            where
                ppt.`name` = ?
            group by pp.product_id
        ) as sppt$round";

        $builder
            ->leftJoin(DB::raw($subquery), 'products.id', '=', "sppt$round.product_id")
            ->addBinding($relation['key'])
            ->orderBy("sppt$round.field", ($ascending ? 'asc' : 'desc'));
    }

    public function scopeFilterProductAttributes(
        Builder $builder,
        array $relation,
        string $operator,
        $value,
        $conjunction
    ) {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        if ($operator == 'isNotAssigned') {
            // Attribute is not assigned to product
            $this->productDoesntHaveAttribute($builder, $function, $relation);
        } elseif ($operator == 'isAssigned') {
            // Attribute is assigned to product
            $this->productHasAttribute($builder, $function, $relation);
        } elseif ($operator == 'isEmpty') {
            /**
             * isEmpty:
             *    the attribute is not associated with product
             *  OR
             *    the attribute associated with product but its value is empty.
             */
            $builder->{$function}(function (Builder $builder) use ($relation) {
                $builder->whereHas('productAttributes', function (Builder $builder) use ($relation) {
                    $builder->where($builder->qualifyColumn('name'), $relation['key'])
                        ->whereNull('product_attributes.value');
                })->orWhereDoesntHave('productAttributes', function (Builder $builder) use ($relation) {
                    $builder->where($builder->qualifyColumn('name'), $relation['key']);
                });
            });
        } elseif ($operator == 'isNotEmpty') {
            $builder->{$function.'In'}(
                'products.id',
                function ($subquery) use ($relation) {
                    $attribute_id = Attribute::where('name', $relation['key'])->first()->id;
                    $subquery->from('product_attributes')
                        ->select('product_id')
                        ->where('attribute_id', $attribute_id)
                        ->whereNotNull('value');
                }
            );
        } else {
            /** TODO: ProductIdsByAttribute consider the operator always just is equal(=) but the filter allows many operators (>,<...)
             *      and the filter should be depend on attribute type
             */
            // array means the attribute type is Date
            if (is_array($value)) {
                $value = $this->getDateFromMode($value['mode'], $value['numberOfDays'] ?? 0)->toDateString();
            }

            $foundProductIds = collect((new ProductIdsByAttribute($relation['key'],
                $value))->get())->pluck('product_id')->toArray();

            $builder->whereIn('products.id', $foundProductIds);
        }

        return $builder;
    }

    public function scopeFilterInventoryIncoming(
        Builder $builder,
        array $relation,
        string $operator,
        $value,
        $conjunction
    ): Builder {
        if ($operator == 'isEmpty') {
            return $builder->doesntHave('purchaseOrderLines')
                ->orWhereHas('purchaseOrderLines', function (Builder $query) {
                    $query->whereHas('purchaseOrder', function ($query) {
                        $query->where('order_status', PurchaseOrder::STATUS_OPEN);
                    })->groupBy('product_id')
                        ->havingRaw('SUM(unreceived_quantity)  = 0 ');
                });
        }
        if ($operator == 'isNotEmpty') {
            return $builder->whereHas('purchaseOrderLines', function (Builder $query) {
                $query->whereHas('purchaseOrder', function ($query) {
                    $query->where('order_status', PurchaseOrder::STATUS_OPEN);
                })->groupBy('product_id')
                    ->havingRaw('SUM(unreceived_quantity)  > 0 ');
            });
        }
        if ($operator == 'isAnyOf') {
            if (empty($value)) {
                return $this->builder;
            }
            $value = (array) $value;
            $placeholders = implode(',', array_fill(0, count($value), '?'));

            return $builder->whereHas('purchaseOrderLines', function (Builder $query) use ($value, $placeholders) {
                $query->whereHas('purchaseOrder', function ($query) {
                    $query->where('order_status', PurchaseOrder::STATUS_OPEN);
                })->groupBy('product_id')
                    ->havingRaw("SUM(unreceived_quantity)  IN ($placeholders)", $value); // Bind values to placeholders
            });
        }

        return $builder->whereHas('purchaseOrderLines', function (Builder $query) use ($operator, $value) {
            $query->whereHas('purchaseOrder', function ($query) {
                $query->where('order_status', PurchaseOrder::STATUS_OPEN);
            })->groupBy('product_id')
                ->havingRaw('SUM(unreceived_quantity) '.$operator.' ?', [$value]);
        });
    }

    public function scopeFilterComponentsParents(
        Builder $builder,
        array $relation,
        string $operator,
        $value,
        $conjunction
    ) {
        return $this->scopeFilterComponents($builder, $relation, $operator, $value, $conjunction, [
            'productToComponentRelationship' => 'productComponentsFromChild',
            'componentToProductRelationship' => 'parentProduct',
        ]);
    }

    public function scopeFilterComponentsChildren(
        Builder $builder,
        array $relation,
        string $operator,
        $value,
        $conjunction
    ) {
        return $this->scopeFilterComponents($builder, $relation, $operator, $value, $conjunction, [
            'productToComponentRelationship' => 'productComponents',
            'componentToProductRelationship' => 'childProduct',
        ]);
    }

    public function scopeFilterComponents(
        Builder $builder,
        array $relation,
        string $operator,
        $value,
        $conjunction,
        array $componentRelation
    ) {
        $key = $relation['key'];
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        return $builder->{$function}(function (Builder $builder) use ($operator, $value, $key, $componentRelation) {
            $builder->whereHas($componentRelation['productToComponentRelationship'],
                function (Builder $builder) use ($operator, $value, $key, $componentRelation) {
                    if ($key == 'id') {
                        $componentField = $componentRelation['componentToProductRelationship'] == 'parentProduct' ? 'parent_product_id' : 'component_product_id';
                        switch ($operator) {
                            case '=':
                                $builder->where('product_components.'.$componentField, $value);
                                break;
                            case 'isAnyOf':
                                $builder->whereIn('product_components.'.$componentField, $value);
                                break;
                            default:
                                return $builder->whereRaw('1 = 0');
                        }
                    }
                    if ($key == 'sku') {
                        $builder->whereHas($componentRelation['componentToProductRelationship'],
                            function (Builder $builder) use ($operator, $value, $key) {
                                switch ($operator) {
                                    case '=':
                                        $builder->where($key, $value);
                                        break;
                                    case '!=':
                                        $builder->where($key, '!=', $value);
                                        break;
                                    case 'contains':
                                        $builder->where($key, 'like', '%'.$value.'%');
                                        break;
                                    case 'doesNotContain':
                                        $builder->where($key, 'not like', '%'.$value.'%');
                                        break;
                                    case 'startsWith':
                                        $builder->where($key, 'like', $value.'%');
                                        break;
                                    case 'endsWith':
                                        $builder->where($key, 'like', '%'.$value);
                                        break;
                                    case 'isEmpty':
                                        $builder->whereNull($key);
                                        break;
                                    case 'isNotEmpty':
                                        $builder->whereNotNull($key);
                                        break;
                                    case 'isAnyOf':
                                        $builder->whereIn($key, $value);
                                        break;
                                    default:
                                        return $builder->whereRaw('1 = 0');
                                }

                            });
                    }
                });
        });
    }

    private function productDoesntHaveAttribute(Builder &$builder, $function, array $relation)
    {
        $builder->{$function}(function (Builder $builder) use ($relation) {
            $builder->whereDoesntHave('productAttributes', function (Builder $builder) use ($relation) {
                $builder->where($builder->qualifyColumn('name'), $relation['key']);
            });
        });
    }

    private function productHasAttribute(Builder &$builder, $function, array $relation)
    {
        $builder->{$function}(function (Builder $builder) use ($relation) {
            $builder->whereHas('productAttributes', function (Builder $builder) use ($relation) {
                $builder->where($builder->qualifyColumn('name'), $relation['key']);
            });
        });
    }

    public function scopeFilterProductPricingTiers(
        Builder $builder,
        array $relation,
        string $operator,
        $value,
        $conjunction
    ) {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        // if key is not specified, we will consider "name" as key to prevent any exception
        if (count($keyExploded = explode('.', $relation['combined_key'])) > 2) {
            $lastKey = array_slice($keyExploded, -1)[0];
            $lastKey = $lastKey == 'value' ? 'price' : $lastKey; // price = value

            // add pivot table
            if ($lastKey == 'price') {
                $lastKey = 'product_pricing.'.$lastKey;
            }
        } else {
            $lastKey = 'name';
        }

        if ($operator == 'isEmpty') {
            /**
             * isEmpty:
             *    the product pricing tier is not associated with product
             *  OR
             *    the product pricing tier associated with product but its price is 0.
             */
            $builder->{$function}(function (Builder $builder) use ($lastKey, $relation) {
                $builder->whereHas('productPricingTiers', function (Builder $builder) use ($lastKey, $relation) {
                    $builder->where($builder->qualifyColumn('name'), $relation['key']);
                    if ($lastKey == 'name' || $lastKey == 'product_pricing.price') {
                        $builder->where('product_pricing.price', 0);
                    } else {
                        $builder->whereNull($lastKey);
                    }
                })->orWhereDoesntHave('productPricingTiers', function (Builder $builder) use ($relation) {
                    $builder->where($builder->qualifyColumn('name'), $relation['key']);
                });
            });
        } else {
            $builder->{$function.'Has'}('productPricingTiers',
                function (Builder $builder) use ($lastKey, $value, $operator, $relation) {
                    $builder->where($builder->qualifyColumn('name'), $relation['key']);
                    if ($lastKey != 'name') {
                        $builder->filterKey(['key' => $lastKey, 'is_relation' => false], $operator, $value);
                    }
                    if ($operator == 'isNotEmpty' && ($lastKey == 'name' || $lastKey == 'product_pricing.price')) {
                        $builder->where('product_pricing.price', '!=', 0);
                    }
                });
        }

        return $builder;
    }

    public function scopeFilterInventory(Builder $builder, array $relation, string $operator, ?int $value, $conjunction)
    {
        $warehouseId = Warehouse::with([])->where('name', $relation['key'])->value('id') ?: 0;

        if ($operator == 'isEmpty' || ($operator == '=' && $value == 0)) {
            return $builder->where(function (Builder $builder) use ($warehouseId) {
                $builder->whereDoesntHave('productInventory', fn ($q) => $q->where('warehouse_id', $warehouseId));
                $builder->orWhereHas('productInventory', function (Builder $builder) use ($warehouseId) {
                    $builder->where('warehouse_id', $warehouseId)
                        ->where('inventory_total', 0);
                });
            }, boolean: $conjunction);
        }

        $function = $conjunction == 'and' ? 'whereHas' : 'orWhereHas';

        if ($operator == 'isNotEmpty' || ($operator == '!=' && $value == 0)) {
            return $builder->{$function}('productInventory', function (Builder $builder) use ($warehouseId) {
                $builder->where('warehouse_id', $warehouseId)
                    ->where('inventory_total', '!=', 0);
            });
        }

        return $builder->{$function}('productInventory',
            function (Builder $builder) use ($warehouseId, $value, $operator) {
                $builder->where('warehouse_id', $warehouseId)
                    ->filterKey('inventory_total', $operator, $value);
            });
    }

    public function scopeFilterBrand(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        /* Debugging headers * /
        static $counter=0;
        $counter++;
        header('X-scopeFilterBrand-'.$counter.'-operator: '.$operator);
        header('X-scopeFilterBrand-'.$counter.'-value: '.$value);
        header('X-scopeFilterBrand-'.$counter.'-conjunction: '.$conjunction);
        /* */

        $function = ($conjunction == 'and' ? 'where' : 'orWhere');

        if ($operator == 'isEmpty') {
            $function .= 'Null';
            $builder->$function('brand_id');

            return;
        } elseif ($operator == 'isNotEmpty') {
            $function .= 'NotNull';
            $builder->$function('brand_id');

            return;
        }

        if ($operator == 'is one of') {
            // If $value is a string, separate by commas
            // for integer parts use as IDs for ProductBrands
            // for string parts use as names for ProductBrands
            $parts = explode(',', $value);
            $int_parts = array_values(array_filter(array_map('intval', $parts)));
            $string_parts = array_values(array_filter($parts, function ($el) {
                return intval($el) == 0;
            }));
            // header('X-scopeFilterBrand-'.$counter.'-int_parts: '.json_encode($int_parts));
            // header('X-scopeFilterBrand-'.$counter.'-string_parts: '.json_encode($string_parts));

            $builder->$function(
                function ($query) use ($int_parts, $string_parts) {
                    $query->whereIn('brand_id', $int_parts);
                    $query->orWhereIn('brand_id', ProductBrand::select('id')->whereIn('name', $string_parts));
                }
            );

            return;
        }

        $negative_operator_opposite = [  // If operator is a negative, turn positive because we don't want double negatives
            '!=' => '=',
            'doesNotContain' => 'contains',
        ];

        $function .= array_key_exists($operator, $negative_operator_opposite) ? 'NotIn' : 'In';

        $sub_operator = Arr::get($negative_operator_opposite, $operator, $operator);

        $builder->$function('brand_id',
            ProductBrand::select('id')->filterKey('name', $sub_operator, $value, $conjunction));
    }

    /**
     * Sorts product by brand name.
     */
    public function scopeSortBrand(Builder $builder, array $relation, bool $ascending): Builder
    {
        // If the table is already joined, we abort
        if ($this->isTableJoined($builder, 'product_brands')) {
            return $builder;
        }
        $builder
            ->leftJoin('product_brands', 'products.brand_id', '=', 'product_brands.id');

        $builder->orderBy('product_brands.name', $ascending ? 'asc' : 'desc');

        return $builder;
    }

    public function scopeSortSupplierProducts(Builder $builder, array $relation, bool $ascending): Builder
    {
        if ($relation['combined_key'] === 'supplierProducts.supplier_sku') {
            return $this->scopeSortSupplierProductsSku($builder, $ascending);
        }

        // Join supplier products if not joined.
        if (! $this->isTableJoined($builder, 'supplier_products')) {
            $builder->leftJoin('supplier_products', 'products.id', '=', 'supplier_products.product_id');
        }

        // Join suppliers if not joined.
        if (! $this->isTableJoined($builder, 'suppliers')) {
            $builder->leftJoin('suppliers', 'supplier_products.supplier_id', '=', 'suppliers.id');
        }

        $builder->orderBy('suppliers.name', $ascending ? 'asc' : 'desc');

        return $builder;
    }

    public function scopeFilterDefaultSupplierProduct(
        Builder $builder,
        array $relation,
        string $operator,
        $value,
        $conjunction
    ) {
        $keys = explode('.', $relation['combined_key']);
        $supplierPricingName = $keys[2];
        $function = $conjunction == 'and' ? 'whereHas' : 'orWhereHas';

        // filter on supplier product price
        return $builder->{$function}('supplierProducts.supplierProductPricing',
            function (Builder $builder) use ($supplierPricingName, $value, $operator) {
                $builder->whereRelation('supplierPricingTier', 'name', $supplierPricingName);
                $builder->filterKey('price', $operator, $value);
            });
    }

    public function scopeSortSupplierProductsSku(Builder $builder, bool $ascending): Builder
    {
        if (! $this->isTableJoined($builder, 'supplier_products')) {
            $builder->leftJoin('supplier_products', 'products.id', '=', 'supplier_products.product_id');
        }

        $builder->orderBy('supplier_products.supplier_sku', $ascending ? 'asc' : 'desc');

        return $builder;
    }

    public function scopeSortPrimaryCategory(Builder $builder, array $relation, bool $ascending)
    {
        // Join product to categories table
        if (! $this->isTableJoined($builder, 'product_to_categories')) {
            $builder->leftJoin('product_to_categories', 'products.id', '=', 'product_to_categories.product_id');
        }

        if (! $this->isTableJoined($builder, 'product_categories')) {
            $builder->leftJoin('product_categories', 'product_to_categories.category_id', '=', 'product_categories.id');
        }

        $builder->where('product_to_categories.is_primary', true)
            ->orderBy('product_categories.name', $ascending ? 'asc' : 'desc');

        return $builder;
    }

    public function scopeSortDefaultSupplierProduct(Builder $builder, array $relation, bool $ascending)
    {
        if ($relation['key'] === 'supplierPricingTiers') {
            $relationName = $relation['name'].'.'.$relation['key'];

            return $builder->joinRelationship($relationName)
                ->where('supplier_pricing_tiers.name', $relation['subkey'])
                ->orderBy('supplier_product_pricing.price', $ascending ? 'asc' : 'desc');
        } else {
            return $builder;
        }
    }

    public function scopeFilterSupplierProducts(
        Builder $builder,
        array $relation,
        string $operator,
        $value,
        $conjunction
    ) {
        if ($relation['combined_key'] == 'supplierProducts.supplier.name') {
            if (empty($value) && $operator == '=') {
                $operator = 'isEmpty';
            }

            if ($operator == 'isEmpty') {
                $function = $conjunction == 'and' ? 'whereDoesntHave' : 'orWhereDoesntHave';

                return $builder->{$function}($relation['name']);
            }
            if ($operator == 'isNotEmpty') {
                $function = $conjunction == 'and' ? 'whereHas' : 'orWhereHas';

                return $builder->{$function}($relation['name']);
            }
            if ($operator == '!=') {
                $function = $conjunction == 'and' ? 'whereDoesntHave' : 'orWhereDoesntHave';

                /** @see SKU-4362 just is not the default supplier */
                return $builder->{$function}('defaultSupplierProduct.supplier', fn ($q) => $q->where('name', $value));
            }
        } elseif ($relation['key'] == 'supplierPricingTiers' || $relation['key'] == 'supplier') {
            $relation = explode('.', $relation['combined_key']);

            $keys = array_slice($relation, 2);
            $relation = implode('.', array_slice($relation, 0, 2));

            return $this->builder->has($relation, '>=', 1, $conjunction,
                function (Builder $builder) use ($relation, $operator, $value, $keys) {
                    if (Str::endsWith($relation, 'supplier')) {
                        $builder->filterKey($keys[0], $operator, $value);
                    } else {
                        $lastKey = $keys[1];
                        if ($lastKey == 'price') {
                            $lastKey = 'supplier_product_pricing.price';
                        }

                        $builder->where('name', $keys[0])
                            ->filterKey(['key' => $lastKey, 'is_relation' => false], $operator, $value);
                    }
                });
        }

        return false; // continue with default behavior
    }

    public function scopeFilterInventory_available_warehouses(
        Builder $builder,
        array $relation,
        string $operator,
        $value,
        $conjunction
    ) {
        $function = $conjunction == 'and' ? 'whereHas' : 'orWhereHas';
        $warehouseId = Warehouse::where('name', $relation['key'])->value('id');

        if ($operator == 'isNotEmpty') {
            $operator = '>';
            $value = 0;
        }

        if ($operator == 'isEmpty') {
            return $builder->has('productInventory', '=', 0)
                ->orWhereHas('productInventory', function (Builder $builder) use ($warehouseId) {
                    $builder->where('warehouse_id', $warehouseId)
                        ->where(function ($builder) {
                            $builder->where('inventory_available', '<=', 0)
                                ->orWhereNull('inventory_available');
                        });
                });
        }

        return $builder->{$function}('productInventory',
            function (Builder $builder) use ($warehouseId, $operator, $value) {
                $builder->where('warehouse_id', $warehouseId);
                if ($operator == 'isAnyOf') {
                    $builder->whereIn('inventory_available', $value);
                } else {
                    $builder->where('inventory_available', $operator, $value);
                }
            });
    }

    /**
     * Transforms product data export.
     */
    public static function transformExportData(array $data): array
    {
        if (empty($data)) {
            return $data;
        }
        $keys = array_keys($data[0]);

        if (in_array('other_images', $keys)) {
            // Bind other images
            foreach ($data as $key => $record) {
                $data[$key]['other_images'] = trim(implode(', ', collect($record['other_images'])
                    ->map(function ($image) {
                        return $image['url'];
                    })->toArray()));
            }
        }

        if (in_array('category_others_path', $keys)) {
            foreach ($data as $key => $record) {
                $path = null;
                if (! is_null($record['category_others_path'])) {
                    foreach ($record['category_others_path'] as $categoryPath) {
                        $path = $path.collect($categoryPath)->pluck('name')->reverse()->implode(' -> ');
                    }
                }

                $data[$key]['category_others_path'] = $path;
            }
        }

        if (in_array('category_others', $keys)) {
            foreach ($data as $key => $record) {
                $path = null;

                if (! is_null($record['category_others'])) {
                    $path = $path.collect($record['category_others'])->pluck('name')->reverse()->implode(' -> ');
                }

                $data[$key]['category_others'] = $path;
            }
        }

        if (in_array('category_main_path', $keys)) {
            foreach ($data as $key => $record) {
                $path = null;

                if (! is_null($record['category_main_path'])) {
                    $path = $path.collect($record['category_main_path'])->pluck('name')->reverse()->implode(' -> ');
                }

                $data[$key]['category_main_path'] = $path;
            }
        }

        if (in_array('tags', $keys)) {
            foreach ($data as $key => $record) {
                $data[$key]['tags'] = collect($record['tags'])->implode('|');
            }
        }

        return $data;
    }

    /**
     * @return Model|\Illuminate\Database\Query\Builder|object|null
     */
    public static function findSupplierPricing($sku, $pricingTierName, $default = false)
    {
        $query = DB::table('products')
            ->select(['products.id', 'products.name', 'supplier_product_pricing.price'])
            ->join('supplier_products', 'supplier_products.product_id', '=', 'products.id');

        if ($pricingTierName == '' && $default === true) {
            $query->joinWhere('supplier_pricing_tiers', 'supplier_pricing_tiers.is_default', '=', 1);
        } else {
            $query->joinWhere('supplier_pricing_tiers', 'supplier_pricing_tiers.name', '=', $pricingTierName);
        }

        $query->join('supplier_product_pricing', 'supplier_pricing_tier_id', '=', 'supplier_pricing_tiers.id')
            ->where('products.sku', $sku)
            ->whereRaw('supplier_product_pricing.supplier_product_id = supplier_products.id');

        return $query->first();
    }

    public function price(): \Illuminate\Database\Eloquent\Casts\Attribute
    {
        return \Illuminate\Database\Eloquent\Casts\Attribute::make(
            get: fn ($value) => $this->getDefaultPricing(ProductPricingTier::default()->id)?->price ?? 0,
        );
    }

    public function getPotentialInventory()
    {
        if ($this->type === self::TYPE_STANDARD) {
            $standardInventory = ProductInventory::whereIn('product_id', $this->id)
                ->where('warehouse_id', '>', 0)
                ->sum('inventory_available');

            return $standardInventory;
        }

        $componentIds = $this->components->pluck('id')->toArray();
        $componentInventory = ProductInventory::whereIn('product_id', $componentIds)
            ->groupBy('product_id', 'warehouse_id')
            ->selectRaw('product_id, warehouse_id, SUM(inventory_available) as sum_inventory')
            ->where('warehouse_id', '>', 0)
            ->get();
        $minBundleQuantity = $componentInventory->map(function ($inventory) {
            $component = $this->components->firstWhere('id', $inventory->product_id);

            return intval($inventory->sum_inventory / $component->pivot->quantity);
        })->min();

        return $minBundleQuantity;
    }

    private function countProductsByPriceGreaterThanZero(self $bundle): int
    {
        $count = 0;

        $bundle->components->each(function (self $component) use (&$count) {
            if ($component->price > 0) {
                $count++;
            }
        });

        return $count;
    }

    private function countProductsByUnitCostGreaterThanZero(self $bundle): int
    {
        $count = 0;

        $bundle->components->each(function (self $component) use (&$count) {
            if ($component->unit_cost > 0) {
                $count++;
            }
        });

        return $count;
    }

    public function getBundlePriceProration(self $bundle): float
    {
        if ($this->countProductsByPriceGreaterThanZero($bundle) >= $this->countProductsByUnitCostGreaterThanZero($bundle)) {
            return $this->calculatePriceProration($bundle);
        }

        return $this->calculateUnitCostProration($bundle);
    }

    private function calculatePriceProration(self $bundle): float
    {
        $totalPrice = 0;
        $bundle->components->each(function (self $component) use (&$totalPrice) {
            $totalPrice += $component->price * $component->pivot->quantity;
        });

        return $totalPrice == 0 ? 0 : round($this->price / $totalPrice, 4);
    }

    private function calculateUnitCostProration(self $bundle): float
    {
        $totalUnitCost = 0;

        $bundle->components->each(function (self $component) use (&$totalUnitCost) {
            $totalUnitCost += $component->unit_cost * $component->pivot->quantity;
        });

        return $totalUnitCost == 0 ? 0 : round($this->unit_cost / $totalUnitCost, 4);
    }

    public function getReportingDailyFinancialDto(Carbon $date): ReportingDailyFinancialDto
    {
        return app(SalesOrderLineFinancialsRepository::class)->getSalesOrderLineFinancialsSummaryForProductDate($this,
            $date);
    }

    /**
     * Generate Blemished Product SKU pattern
     *
     * @throws Exception
     */
    public function generateSkuPattern(): string
    {
        $pattern = Setting::getBlemishedSkuPattern();
        $sku = $this->sku ?? 'SKU';
        $today = today();

        // Extract date format from the pattern
        preg_match('/{{date:([^}]+)}}/', $pattern, $matches);
        if (! isset($matches[1])) {
            throw new Exception('Make sure you are following the pattern {{date:XXXX}}');
        }
        $dateFormat = $matches[1];

        $date = $today->isoFormat($dateFormat);
        $resolvedPattern = str_replace('{{sku}}', $sku, $pattern);
        $resolvedPattern = str_replace('{{date:'.$dateFormat.'}}', $date, $resolvedPattern);
        $suffix = 1;
        $finalSku = $resolvedPattern;
        while (Product::where('sku', $finalSku)->exists()) {
            $finalSku = $resolvedPattern.'.'.$suffix;
            $suffix++;
        }

        return $finalSku;
    }
}
