<?php

namespace App\Models;

use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * Class PaymentTerm.
 *
 *
 * @property int $id
 * @property string $name
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $archived_at
 */
class PaymentTerm extends Model implements Filterable, Sortable
{
    use Archive, HasFilters, HasSort;
    use HasFactory;

    protected $guarded = [];

    protected $casts = ['archived_at' => 'datetime'];

    protected $fillable = ['name'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function purchaseOrders()
    {
        return $this->hasMany(PurchaseOrder::class);
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.payment_term.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    /**
     * delete payment term
     *
     * @return array|bool|null
     *
     * @throws \Exception
     */
    public function delete()
    {
        // If the model was used from any related relation, we will archived it and its variants
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return parent::delete();
    }

    /**
     * check if there is related purchase orders.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        // load related relations and its children relations
        $this->loadCount('purchaseOrders');

        if ($this->purchase_orders_count) {
            return [
                'purchaseOrders' => trans_choice('messages.currently_used', $this->purchase_orders_count, [
                    'resource' => 'purchase Order',
                    'model' => 'payment term('.$this->name.')',
                ]),
            ];
        }

        return false;
    }
}
