<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Enums\PaymentType as EnumPaymentType;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\CachesOrderCurrency;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Repositories\IntegrationInstanceRepository;
use App\Services\Accounting\AccountingIntegration;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute as Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Modules\Xero\Entities\XeroPayment;
use Modules\Xero\Entities\XeroTransaction;

/**
 * Class Payment.
 *
 *
 * @property int $id
 * @property string $link_type
 * @property int $link_id
 * @property Carbon $payment_date
 * @property int $payment_type_id
 * @property string $external_reference
 * @property ?string $type
 * @property float $amount
 * @property float $cost
 * @property int $currency_id
 * @property float $currency_rate
 * @property int $currency_id_tenant_snapshot
 * @property int $accounting_integration_id
 * @property string $accounting_integration_type
 * @property-read float $amount_in_tenant_currency
 * @property mixed $link
 * @property PaymentType $paymentType
 * @property mixed $accountingIntegration
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 */
class Payment extends Model implements Filterable, Sortable, UniqueFieldsInterface
{
    use BulkImport;
    use CachesOrderCurrency, HandleDateTimeAttributes, HasFactory, HasFilters, HasSort;

    protected $casts = [
        'payment_date' => 'datetime',
        'amount' => 'float',
        'cost' => 'float',
        'type' => EnumPaymentType::class,
    ];

    protected $fillable = [
        'payment_date',
        'payment_type_id',
        'external_reference',
        'amount',
        'type',
        'cost',
        'currency_id',
        'currency_code',
        'accounting_integration_id',
        'accounting_integration_type',
        'link_type',
        'link_id',
    ];

    const STATUS_TO_SYNC = 'toSync';

    const STATUS_SYNCED = 'synced';

    const STATUS_HAS_ERRORS = 'hasErrors';

    const STATUS_INELIGIBLE_TO_SYNC = 'ineligibleToSync';

    const TYPE_DIRECT = 'direct';

    const TYPE_OVERPAYMENT = 'overpayment';

    const TYPE_PREPAYMENT = 'prepayment';

    const STATUSES = [
        self::STATUS_TO_SYNC,
        self::STATUS_SYNCED,
        self::STATUS_HAS_ERRORS,
        self::STATUS_INELIGIBLE_TO_SYNC,
    ];

    public static function getUniqueFields(): array
    {
        return [
            'link_type',
            'link_id',
            'payment_date',
            'payment_type_id',
            'amount',
            'currency_id',
            'external_reference',
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function paymentType(): BelongsTo
    {
        return $this->belongsTo(PaymentType::class);
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    public function link(): MorphTo
    {
        return $this->morphTo('link');
    }

    public function accountingIntegration(): MorphTo|AccountingIntegration
    {
        return $this->morphTo('accountingIntegration');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function amountInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->amount * $this->currency_rate);
    }

    public function setCurrencyCodeAttribute($value)
    {
        $currency = cache()->remember('currency_for_code_' . $value, 60, function () use ($value) {
            return Currency::with([])->where('code', $value)->first();
        });
        $this->currency_id = $currency->id;
        $this->currency_rate = $currency->conversion;
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function scopeFilterLink(Builder $query, array $relation, string $operator, $value, $conjunction): Builder
    {
        $function = $conjunction == 'and' ? 'whereHasMorph' : 'orWhereHasMorph';

        $relationSplit = explode('.', $relation['combined_key']);
        $query->{$function}($relation['name'], ['*'], function ($q) use ($operator, $value, $relationSplit) {
            $q->filterKey(implode('.', array_slice($relationSplit, 1)), $operator, $value);
        });
        if (in_array($operator, ['!=', 'doesNotContain', 'isEmpty'])) {
            $query->orWhereNull($this->{$relation['name']}()->getForeignKeyName());
        }

        return $query;
    }

    public function scopeFilterAccountingIntegration(Builder $query, array $relation, string $operator, $value, $conjunction): Builder
    {
        $function = $conjunction == 'and' ? 'whereHasMorph' : 'orWhereHasMorph';

        $query->{$function}(
            $relation['name'],
            [XeroTransaction::class, XeroPayment::class],
            function ($q) use ($relation, $operator, $value) {
                $q->filterKey($relation['key'], $operator, $value);
            });
        if (in_array($operator, ['!=', 'doesNotContain', 'isEmpty'])) {
            $query->orWhereNull($this->{$relation['name']}()->getForeignKeyName());
        }

        return $query;
    }

    public function scopeFilterStatus(Builder $query, array $relation, string $operator, string $value, $conjunction): Builder
    {
        switch ($value) {
            case 'toSync':
                $query->whereHas('accountingIntegration', function (Builder $query) {
                    $query->whereColumn('payments.updated_at', '>', 'updated_at');
                })
                    ->orWhereNull('accounting_integration_id');
                if (app(IntegrationInstanceRepository::class)->getAccountingInstance()) {
                    $query->where('payment_date', '>=', app(IntegrationInstanceRepository::class)->getAccountingInstance()->integration_settings['settings']['sync_start_date']);
                }

                return $query;
            case 'synced':
                return $query->whereHas('accountingIntegration', function (Builder $query) {
                    $query->whereNull('last_error');
                });
            case 'hasErrors':
                return $query->whereHas('accountingIntegration', function (Builder $query) {
                    $query->whereNotNull('last_error');
                });
            case 'ineligibleToSync':
                if (app(IntegrationInstanceRepository::class)->getAccountingInstance()) {
                    return $query->where('payment_date', '<', app(IntegrationInstanceRepository::class)->getAccountingInstance()->integration_settings['settings']['sync_start_date']);
                }

                return $query;
            default:
                return $query;
        }
    }

    public function save(array $options = [])
    {
        if (empty($this->payment_date)) {
            $this->payment_date = now();
        }

        if (empty($this->currency_id)) {
            $this->currency_id = Currency::default()->id;
        }

        $this->cacheCurrencyRate($this->isDirty(['currency_id']));

        return parent::save($options);
    }

    public function availableColumns()
    {
        return [];
    }

    public function filterableColumns(): array
    {
        return [
            'payment_date',
            'type',
            'last_error',
            'parent_reference',
            'payment_type_id',
            'external_reference',
            'amount',
            'currency_id',
            'created_at',
            'updated_at',
            'status',
        ];
    }

    public function generalFilterableColumns(): array
    {
        return $this->filterableColumns();
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }
}
