<?php

namespace App\Models;

use App\DTO\ReportingDailyFinancialDto;
use App\Exporters\MapsExportableFields;
use App\Importers\DataImporter;
use App\Importers\DataImporters\NominalCodeImporter;
use App\Importers\ImportableInterface;
use App\Models\Abstractions\AbstractReportable;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Repositories\NominalCodeRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;

/**
 * Class NominalCode.
 *
 *
 * @property int $id
 * @property string $code
 * @property string $name
 * @property string $type
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $archived_at
 */
class NominalCode extends AbstractReportable implements Filterable, ImportableInterface, MapsExportableFields, Sortable
{
    use Archive, HasFilters, HasSort;
    use HasFactory;

    /**
     * Nominal Code types.
     */
    const TYPE_REVENUE = 'Revenue';

    const TYPE_EXPENSE = 'Expense';

    const TYPE_ASSET = 'Asset';

    const TYPE_LIABILITY = 'Liability';

    const TYPE_EQUITY = 'Equity';

    const TYPE_BANK = 'Bank';

    const TYPE_CURRENT = 'Current';

    const TYPE_INVENTORY = 'Inventory';

    const TYPE_FIXED = 'Fixed';

    const TYPE_DIRECTCOSTS = 'Directcosts';

    const TYPE_CURRLIAB = 'Currliab';

    const TYPE_TERMLIAB = 'Termliab';

    const TYPE_SALES = 'Sales';

    const TYPES_INCOME_STATEMENT = [
        self::TYPE_SALES,
        self::TYPE_REVENUE,
        self::TYPE_DIRECTCOSTS,
        self::TYPE_EXPENSE,
    ];

    const TYPES = [
        self::TYPE_REVENUE,
        self::TYPE_EXPENSE,
        self::TYPE_ASSET,
        self::TYPE_LIABILITY,
        self::TYPE_EQUITY,
        self::TYPE_BANK,
        self::TYPE_CURRENT,
        self::TYPE_INVENTORY,
        self::TYPE_FIXED,
        self::TYPE_DIRECTCOSTS,
        self::TYPE_CURRLIAB,
        self::TYPE_TERMLIAB,
        self::TYPE_SALES,
    ];

    protected $fillable = ['code', 'name', 'type'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function SalesOrderLines()
    {
        return $this->hasMany(SalesOrderLine::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * {@inheritDoc}
     */
    public function delete()
    {
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        // Set default mapping to null if exists.
        $settings = Setting::with([])->whereIn('key', Setting::KEYS_NC_MAPPING)->get();
        if (($setting = $settings->where('value', null)->firstWhere('default_value', $this->id))) {
            // The nominal code is a default value in some mapping,
            // we remove the default value
            $setting->default_value = null;
            $setting->save();
        }

        return parent::delete();
    }

    /**
     * Determine if the nominal code is used.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        $usage = [];
        // used in mapping
        $settings = Setting::with([])->whereIn('key', Setting::KEYS_NC_MAPPING)->get();
        if ($settings->firstWhere('value', $this->id)) {
            $usage['mappingNominalCodes'] = __('messages.nominal_code.used_in_mapping', ['id' => $this->name]);
        }

        // used in sales order lines
        $this->loadCount('SalesOrderLines');
        if ($this->sales_order_lines_count) {
            $usage['salesOrderLines'] = trans_choice('messages.currently_used', $this->sales_order_lines_count, [
                'resource' => 'sales order line',
                'model' => 'nominal code('.$this->name.')',
            ]);
        }

        return count($usage) ? $usage : false;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.nominal_code.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name', 'code', 'type'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new NominalCodeImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return NominalCodeImporter::getExportableFields();
    }

    public function getReportingDailyFinancialDto(Carbon $date): ReportingDailyFinancialDto
    {
        // TODO: Implement calculateFinancialSummary() method once cost invoices ready.
        // app(NominalCodeRepository::class)->getCostInvoiceLineFinancialsSummaryForNominalCodeDate($this, $date);
    }
}
