<?php

namespace App\Models\Magento;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Exceptions\IntegrationInstance\Magento\BundleComponentUnavailableException;
use App\Exceptions\IntegrationInstance\Magento\UnableToCreateBundleProductException;
use App\Exporters\MapsExportableFields;
use App\Helpers;
use App\Http\Resources\Magento\ProductListingResource;
use App\Http\Resources\Magento\ProductResource;
use App\Integrations\Listing;
use App\Integrations\Listings\ListingsHelper;
use App\Integrations\Listings\ProductMapper;
use App\Jobs\Magento\GetCustomerGroupsJob;
use App\Jobs\Magento\GetProductAttributesJob;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\IntegrationInstance;
use App\Models\Product as SkuProduct;
use App\Models\ProductListing;
use App\Models\ProductPricingTier;
use App\Models\Setting;
use App\Services\Magento\MagentoListingMapper;
use App\Services\Product\CreateUpdateProductService;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property int $variant_id
 * @property string|null $supplier
 * @property  string|null $image
 * @property array $json_object
 * @property Carbon|null $mapped_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property string $updated_by
 * @property-read int $image_count
 * @property-read string $status_name
 * @property-read IntegrationInstance $integrationInstance
 * @property-read ProductListing $productListing
 * @property-read CustomerGroup[]|Collection $customerGroups
 * @property-read ProductAttribute[]|Collection $productAttributes
 *
 * @method static Builder|static latestProduct(int $integrationInstanceId, string $updatedBy = null, string $column = 'json_object->updated_at')
 */
class Product extends Model implements Filterable, Listing, MapsExportableFields, Sortable
{
    use HasFilters, HasSort, ListingsHelper;

    protected $table = 'magento_products';

    public $dataTableKey = 'magento.product';

    protected $guarded = [];

    protected $casts = [
        'json_object' => 'array',
        'name' => 'string',
        'sku' => 'string',
        'status' => 'string',
        'price' => 'float',
        'weight' => 'float',
        'mapped_at' => 'datetime',
    ];

    const SELLER_SKU_COLUMN = 'json_object->sku';

    private static int $max_image_count;

    public function integrationInstance()
    {
        return $this->belongsTo(IntegrationInstance::class);
    }

    public function productListing()
    {
        return $this->belongsTo(ProductListing::class, 'product', 'id');
    }

    public function customerGroups()
    {
        return $this->hasMany(CustomerGroup::class, 'integration_instance_id', 'integration_instance_id');
    }

    public function productAttributes()
    {
        return $this->hasMany(ProductAttribute::class, 'integration_instance_id', 'integration_instance_id');
    }

    public function getStatusNameAttribute(): string
    {
        return $this->status == 1 ? 'Enabled' : 'Disabled';
    }

    private function mapToProduct($productId)
    {
        $mapping = [
            'variant_id' => $this['variant_id'],
            'product_id' => $productId,
        ];

        self::getProductMapper($this->integrationInstance)->mapListings([$mapping]);

        $this->mapped_at = now();
        $this->save();
    }

    /**
     * {@inheritDoc}
     */
    public function createSkuProduct(array $fieldMapping = [])
    {
        return DB::transaction(function () use ($fieldMapping) {
            $skuValue = empty($fieldMapping) ? $this->getValue('sku') : $this->getValue($this->getMappedField('sku', $fieldMapping));

            if (empty(trim($skuValue))) {
                return ['id' => $this['id'], 'success' => false, 'reason' => 'The listing has empty seller sku, skipping.'];
            }

            if ($this->productListing) {
                // Skip if listing is already mapped.
                return [
                    'id' => $this['id'],
                    'success' => true,
                    'product_id' => $this->productListing->product_id,
                    'reason' => 'Listing already mapped to sku product, skipping.',
                ];
            }

            // Make sure the product doesn't exist already
            $product = SkuProduct::with([])->where('sku', $skuValue)->first();

            if ($product) {
                // We check if listing isn't mapped, we map it to the product
                $this->mapToProduct($product->id);

                return [
                    'id' => $this['id'],
                    'success' => true,
                    'product_id' => $product->id,
                    'reason' => "Product with sku '{$skuValue}' already exists, skipping.",
                ];
            }

            $this->setRelation('productAttributes', $this->productAttributes->keyBy('code'));
            // We map the the fields to create the needed payload and
            // create the product
            $defaultDimension = Helpers::setting(Setting::KEY_DEFAULT_DIMENSION_UNIT, null, true);
            $defaultWeight = Helpers::setting(Setting::KEY_DEFAULT_WEIGHT_UNIT, null, true);

            if (! empty($fieldMapping)) {
                // We create the product with mapping.
                $product = $this->createSkuProductWithMapping($fieldMapping, [
                    'sku' => $this['sku'],
                    'type' => $this->json_object['type_id'] == 'bundle' ? SkuProduct::TYPE_BUNDLE : SkuProduct::TYPE_STANDARD,
                    'name' => $this['name'],
                ]);
            } else {
                // We use the needed fields to create the product.
                $payload = [
                    'sku' => $this['sku'],
                    'name' => $this['name'],
                    'type' => $this->json_object['type_id'] == 'bundle' ? SkuProduct::TYPE_BUNDLE : SkuProduct::TYPE_STANDARD,
                    'weight' => $this['weight'] ?? 0,
                    'weight_unit' => $defaultWeight,
                    'dimension_unit' => $defaultDimension,
                    'case_dimension_unit' => $defaultDimension,
                ];

                $product = new SkuProduct($payload);
                $product->save();

                //$product->setBrandName( $this['matchingProduct']['AttributeSets'][0]['Brand'] ?? null );

                $defaultPricingTier = ProductPricingTier::with([])->where('is_default', 1)->first();
                $payload['pricing'] = $defaultPricingTier ? [
                    [
                        'product_pricing_tier_id' => $defaultPricingTier->id,
                        'price' => $this['price'],
                    ],
                ] : [];

                $product->setPricing($payload['pricing'], false);

                if ($this['image']) {
                    $product->addImages([
                        [
                            'url' => $this->getImageUrl($this['image']),
                        ],
                    ]);
                }
            }

            // assign pricing tiers to the product
            $pricing = [];
            foreach ($this->json_object['tier_prices'] as $tierPrice) {
                /*
                 * 32000 id in magento represents all groups and won't be retrieved in GetCustomerGroups
                 */
                if ($tierPrice['customer_group_id'] == '32000') {
                    continue;
                }
                $customerGroup = $this->customerGroups->firstWhere('customer_group_id', $tierPrice['customer_group_id']);
                // load again from Magento is it's not exist
                if (! $customerGroup) {
                    dispatch_sync(new GetCustomerGroupsJob($this->integrationInstance));
                    $this->load('customerGroups');
                    $customerGroup = $this->customerGroups->firstWhere('customer_group_id', $tierPrice['customer_group_id']);
                }

                $pricing[] = [
                    'product_pricing_tier_id' => $customerGroup->product_pricing_tier_id,
                    'price' => $tierPrice['value'],
                ];
            }
            $product->setPricing($pricing ?: null, false);

            // add bundle components to the product
            if ($product->type == SkuProduct::TYPE_BUNDLE) {
                $components = collect($this->json_object['extension_attributes']['bundle_product_options'])
                    ->map(fn ($option) => $option['product_links'][0] ?? null) // TODO: currently, I will only take the first product from the bundle option
                    ->filter()->values()->map(fn ($c) => ['sku' => $c['sku'], 'quantity' => $c['qty']])
                    ->keyBy('sku')->toArray();

                // create magento products
                $componentProducts = self::query()->whereIn('sku', array_keys($components))->get();
                foreach ($components as $component) {
                    $componentProduct = $componentProducts->firstWhere('sku', $component['sku']);
                    if (! $componentProduct) {
                        throw new BundleComponentUnavailableException($component['sku']);
                    }

                    $response = $componentProduct->createSkuProduct($fieldMapping);
                    if ($response['success']) {
                        $components[$componentProduct->sku]['id'] = $response['product_id'];
                    } else {
                        throw new UnableToCreateBundleProductException($componentProduct->sku, $response['reason']);
                    }
                }
                // add bundle components to the product
                CreateUpdateProductService::make($product)->setBundleComponents($components);
            }

            $this->mapToProduct($product->id);

            return ['id' => $this->id, 'success' => true, 'product_id' => $product->id];
        });
    }

    /**
     * {@inheritDoc}
     */
    public function getListingUrl(): ?string
    {
        $storeUrl = $this->integrationInstance->connection_settings['store_base_url'] ?? null;

        return "{$storeUrl}/backend/catalog/product/edit/id/{$this->variant_id}";
    }

    /**
     * {@inheritDoc}
     */
    public static function getResource()
    {
        return ProductResource::class;
    }

    /**
     * {@inheritDoc}
     */
    public static function getProductListingResource()
    {
        return ProductListingResource::class;
    }

    /**
     * {@inheritDoc}
     */
    public static function getProductMapper(IntegrationInstance $integrationInstance): ProductMapper
    {
        return new MagentoListingMapper($integrationInstance);
    }

    /**
     * {@inheritDoc}
     */
    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', false, 'ID'),
            'variant_id' => Exporter::makeExportableField('variant_id', false),
            'sku' => Exporter::makeExportableField('sku', false),
            'price' => Exporter::makeExportableField('price', false),
            'name' => Exporter::makeExportableField('name', false),
            'status' => Exporter::makeExportableField('status', false),
            'mapped_sku.id' => Exporter::makeExportableField('mapped_sku.id', false),
            'mapped_sku.name' => Exporter::makeExportableField('mapped_sku.name', false),
            'mapped_sku.sku' => Exporter::makeExportableField('mapped_sku.sku', false),
            'mapped_sku' => Exporter::makeExportableField('mapped_sku', false),
        ];
    }

    /**
     * {@inheritDoc}
     */
    public static function makeExportData($builder): array
    {
        $records = $builder->get()->map(function (self $listing) {
            return [
                (string) $listing['variant_id'],
                $listing->productListing ? $listing->productListing->product->sku : null,
                $listing['sku'],
                $listing['name'],
                $listing['price'],
                $listing['status'],
            ];
        })->toArray();

        return [
            'filename' => 'magento_listings',
            'headers' => [
                'variant_id',
                'mapped_sku',
                'sku',
                'name',
                'price',
                'status',
            ],
            'records' => $records,
            'xlsxheaders' => [
                'variant_id' => 'string',
                'mapped_sku' => 'string',
                'sku' => 'string',
                'name' => 'string',
                'price' => 'double',
                'status' => 'string',
            ],
        ];
    }

    /**
     * Get available columns to show in datatable.
     */
    public function availableColumns(): array
    {
        return config("data_table.{$this->dataTableKey}.columns");
    }

    /**
     * Get filterable columns, must be inside available columns.
     */
    public function filterableColumns(): array
    {
        $columns = collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
        for ($i = 1; $i <= static::maxImageCount(); $i++) {
            $columns[] = "image$i";
        }

        return $columns;
    }

    /**
     * Get General filterable columns.
     */
    public function generalFilterableColumns(): array
    {
        return ['sku', 'variant_id', 'price', 'name', 'status'];
    }

    /**
     * Get sortable columns.
     */
    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function delete()
    {
        if ($this->productListing) {
            $this->productListing->delete();
        }

        return parent::delete();
    }

    public function isUsed()
    {
        return false;
    }

    public function getValue($field)
    {
        if (Str::startsWith($field, 'image')) {
            $images = $this->json_object['media_gallery_entries'] ?? [];
            if (empty($images)) {
                return null;
            }

            $imageIndex = max(0, (int) str_replace('image', '', $field) - 1);
            if (is_numeric($imageIndex) && isset($images[$imageIndex])) {
                return $this->getImageUrl($images[$imageIndex]['file'] ?? null);
            }

            return null;
        }

        if (Str::startsWith($field, 'attributes.')) {
            $attributeCode = str_replace('attributes.', '', $field);
            $customAttribute = collect($this->json_object['custom_attributes'])->filter(fn ($ca) => strtolower($ca['attribute_code']) == $attributeCode)->first();
            $customAttributeValue = $customAttribute['value'] ?? null;
            if (is_null($customAttributeValue) || strlen(trim($customAttributeValue)) == 0) {
                return null;
            }

            // attributes as image
            if (in_array($attributeCode, ['image', 'small_image', 'thumbnail'])) {
                return $this->getImageUrl($customAttributeValue);
            }

            if (filter_var($customAttributeValue, FILTER_VALIDATE_INT) !== false) {
                $productAttribute = $this->productAttributes[$customAttribute['attribute_code']];
                if (! $productAttribute) {
                    dispatch_sync(new GetProductAttributesJob($this->integrationInstance));
                    $this->load('productAttributes');
                    $this->setRelation('productAttributes', $this->productAttributes->keyBy('code'));
                    $productAttribute = $this->productAttributes[$customAttribute['attribute_code']];
                }

                if (! empty($productAttribute->options)) {
                    $customAttributeValue = collect($productAttribute->options)->firstWhere('value', $customAttributeValue)['label'] ?? $customAttributeValue;
                }
            }

            return $customAttributeValue;
        }

        return $this[$field];
    }

    public function getImageUrl(?string $imagePath): ?string
    {
        if (empty($imagePath)) {
            return null;
        }

        return rtrim($this->integrationInstance->connection_settings['store_base_url'], '/').'/pub/media/catalog/product/'.ltrim($imagePath, '/');
    }

    public function scopeFilterProductListing(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        // without whereHas product
        if ($operator == 'isEmpty' || $operator == 'isNotEmpty') {
            return $builder->whereNull('product', $conjunction, $operator == 'isNotEmpty');
        }

        return false; // default behavior
    }

    public function scopeLatestProduct(Builder $builder, int $integrationInstanceId, ?string $updatedBy = null, string $column = 'json_object->updated_at')
    {
        $builder->where('integration_instance_id', $integrationInstanceId)
            ->when($updatedBy, function (Builder $builder) use ($updatedBy) {
                $builder->where('updated_by', $updatedBy);
            })
            ->latest($column);
    }

    public static function maxImageCount(): int
    {
        if (! isset(static::$max_image_count)) {
            static::$max_image_count = static::query()->latest('image_count')->value('image_count') ?: 1;
        }

        return static::$max_image_count;
    }
}
