<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;

/**
 * Class InventoryForecastCache.
 *
 * @property-read int $id
 * @property int $supplier_id
 * @property int $product_id
 * @property float $suggested_purchase_quantity
 * @property Carbon $forecast_date
 * @property float $available_stock
 * @property float $inbound_stock
 * @property float $in_transit_stock
 * @property ?float $inventory_inbound
 * @property ?float $inventory_available
 * @property ?float $inventory_in_transit
 * @property bool $complete_for_supplier
 * @property-read Collection|PurchaseOrderLine[] $purchaseOrderLines
 * @property-read Warehouse $warehouse
 * @property-read Product $product
 * @property-read Supplier $supplier
 */
class InventoryForecastItem extends Model
{

    use HasFactory;

    /**
     * @var array
     */
    protected $guarded = [];

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function purchaseOrderLines(): BelongsToMany
    {
        return $this->belongsToMany(
            PurchaseOrderLine::class,
            'forecast_item_po_line_links',
            'forecast_item_id',
            'purchase_order_line_id'
        );
    }

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function getReservedStock(): float{
        return $this->product->inWarehousesReservedQuantity()
            ->where('warehouse_id', $this->warehouse_id)
            ->sum('inventory_reserved');
    }

}
