<?php

namespace App\Models;

use App\Contracts\HasReference;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use App\Services\InventoryManagement\PositiveInventoryEvent;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Facades\DB;

/**
 * Class InventoryAssemblyLine.
 *
 * @property int $id
 * @property int $inventory_assembly_id
 * @property string $product_type
 * @property int $quantity
 * @property int $product_id
 * @property float $unit_cost
 * @property-read InventoryAssembly $inventoryAssembly
 * @property-read Product $product
 * @property-read Collection|InventoryMovement[] $inventoryMovements
 */
class InventoryAssemblyLine extends Model implements Filterable, HasReference, PositiveInventoryEvent
{
    use HasFilters;

    protected $fillable = ['product_type'];

    const INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_KIT = 'kit';

    const INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_COMPONENT = 'component';

    const INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPES = [
        self::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_KIT,
        self::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_COMPONENT,
    ];

    public function getFifoLayerIdAttribute()
    {
        return $this->inventoryMovements()
            ->where('link_type', self::class) // When the assembly was created.
            ->firstOrFail(['layer_id'])
      ->layer_id;
    }

    /*
     |--------------------------------------------------------------------------
     | Relations
     |--------------------------------------------------------------------------
     */

    public function inventoryAssembly(): BelongsTo
    {
        return $this->belongsTo(InventoryAssembly::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function inventoryMovements(): MorphMany
    {
        return $this->morphMany(InventoryMovement::class, 'link');
    }

    public function fifoLayers(): MorphMany
    {
        $relation = $this->morphMany(FifoLayer::class, 'link');

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (FifoLayer $fifoLayer) {
                $fifoLayer->delete();
            });
        });

        return $relation;
    }

    public function getReference(): string
    {
        return $this->inventoryAssembly->getReference();
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return ['quantity', 'sku'];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * @return bool|void|null
     */
    public function delete()
    {
        // Remove inventory movements
        $this->load('inventoryMovements');
        DB::transaction(function () {
            if ($this->quantity < 0) {
                // This assembly was a reduction in inventory,
                foreach ($this->inventoryMovements as $inventoryMovement) {
                    $fifoLayer = $inventoryMovement->fifo_layer;
                    $fifoLayer->fulfilled_quantity += $inventoryMovement->quantity;
                    $fifoLayer->save();

                    $inventoryMovement->delete();
                }
            } else {
                // The assembly was an increase in inventory,
                // we apply back any fulfilled inventory from the
                // created fifo layer to other fifo layers on the product.
                /** @var FifoLayer $fifoLayer */
                if (($fifoLayer = $this->fifoLayers()->first())) {
                    /*
         * Instead of the following, we should think about restricting deletion of the assembly if the fifo layer has been used
         */
                    if ($fifoLayer->fulfilled_quantity > 0) {
                        $activeNegativeMovements = $fifoLayer->inventoryMovements()->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                            ->where('quantity', '<', 0)
                            ->orderByDesc('id')
                            ->get();

                        /** @var InventoryMovement $inventoryMovement */
                        foreach ($activeNegativeMovements as $inventoryMovement) {
                            $inventoryMovement->reassignFifoLayer();
                        }
                    }

                    $fifoLayer->delete();
                }
            }

            $this->inventoryMovements()->delete();

            return parent::delete();
        });
    }

    public function getLinkType(): string
    {
        return self::class;
    }

    public function getId(): int
    {
        return $this->id;
    }

    public function getProductId(): int
    {
        return $this->product_id;
    }

    public function getWarehouseId(): int
    {
        return $this->inventoryAssembly->warehouse_id;
    }

    public function getEventDate(): Carbon
    {
        return $this->inventoryAssembly->action_date;
    }

    public function getQuantity(): int
    {
        return $this->quantity;
    }

    public function getType(): string
    {
        return InventoryMovement::TYPE_ASSEMBLY;
    }

    public function getDrawer(): Model
    {
        return $this->inventoryAssembly;
    }

    public function createFifoLayer(int $quantity, ?float $unitCost = null, ?int $productId = null): FifoLayer
    {
        $fifoLayer = new FifoLayer;
        $fifoLayer->fifo_layer_date = $this->getEventDate();
        $fifoLayer->product_id = $this->getProductId();
        $fifoLayer->original_quantity = $this->getQuantity();
        $fifoLayer->fulfilled_quantity = 0;
        $fifoLayer->warehouse_id = $this->getWarehouseId();
        $fifoLayer->total_cost = $unitCost;
        $this->fifoLayers()->save($fifoLayer);

        return $fifoLayer;
    }

    public function createInventoryMovements(int $quantity, FifoLayer $fifoLayer): void
    {
        $inventoryMovement = new InventoryMovement;
        $inventoryMovement->product_id = $this->getProductId();
        $inventoryMovement->inventory_movement_date = $this->getEventDate();
        $inventoryMovement->quantity = $this->getQuantity();
        $inventoryMovement->type = InventoryMovement::TYPE_ASSEMBLY;
        $inventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $inventoryMovement->warehouse_id = $this->getWarehouseId();
        $inventoryMovement->warehouse_location_id = $fifoLayer->warehouse->defaultLocation?->id ?? null;
        $inventoryMovement->fifo_layer = $fifoLayer->id;

        // this is the cost per unit used.  If the unit made is a component, the unit cost should be the prorated value of the component for the bundle
        $this->inventoryMovements()->save($inventoryMovement);
    }

    public function getOriginatingMovement(): ?InventoryMovement
    {
        return $this->inventoryMovements->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '>', 0)
            ->first();
    }

    public function getFifoLayer(): ?FifoLayer
    {
        return $this->fifoLayers()->first();
    }

    public function getUnitCost(): float
    {
        return $this->unit_cost > 0 ? $this->unit_cost : $this->product->getUnitCostAtWarehouse($this->getWarehouseId());
    }

    public function backorderQueueReleases()
    {
        return $this->morphMany(BackorderQueueRelease::class, 'link');
    }

    /**
     * @uses  FifoLayerRepository::recalculateTotalCosts()
     */
    public function refreshFromFifoCostUpdate(float $amount = 0): void
    {
        $this->updated_at = now();
        $this->save();
    }

}
