<?php

namespace App\Models;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Data\IntegrationInstanceInventoryData;
use App\Enums\IntegrationInstanceSyncStatusEnum;
use App\Helpers;
use App\Models\Amazon\AmazonReportLog;
use App\Models\Amazon\FBAInitialInventory;
use App\Models\Amazon\Order as AmazonOrder;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use App\Models\Magento\CustomerGroup;
use App\Models\Magento\InventorySource;
use App\Services\ShippingProvider\ShipStationManager;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Modules\Qbo\Entities\QboAccount;
use Modules\Qbo\Entities\QboBill;
use Modules\Qbo\Entities\QboCustomer;
use Modules\Qbo\Entities\QboInvoice;
use Modules\Qbo\Entities\QboItem;
use Modules\Qbo\Entities\QboJournal;
use Modules\Qbo\Entities\QboPayment;
use Modules\Qbo\Entities\QboPurchaseOrder;
use Modules\Qbo\Entities\QboTaxRate;
use Modules\Qbo\Entities\QboVendor;
use Modules\Xero\Entities\XeroAccount;
use Modules\Xero\Entities\XeroContact;
use Modules\Xero\Entities\XeroPayment;
use Modules\Xero\Entities\XeroTaxRate;
use Modules\Xero\Entities\XeroTransaction;
use Spatie\LaravelData\Optional;
use Throwable;

/**
 * Class IntegrationInstance.
 *
 *
 * @property int $id
 * @property int $integration_id
 * @property string $name
 * @property array $connection_settings
 * @property array $integration_settings
 * @property string $connection_status
 * @property string $country
 * @property bool $is_automatic_sync_enabled
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read Integration $integration
 * @property-read SalesChannel $salesChannel
 * @property-read Warehouse $warehouse
 * @property-read Collection|IntegrationInstanceServiceDisruptions[] $integrationInstanceServiceDisruptions
 * @property-read array $credentials
 * @property-read int|null $payment_type_id
 * @property-read Carbon $order_start_date
 * @property-read Carbon $open_start_date
 * @property-read Carbon $closed_start_date
 * @property-read Carbon $audit_trail_start_date
 * @property-read string $refresh_token
 *
 * @method static IntegrationInstance|Builder amazon
 * @method static IntegrationInstance|Builder shipstation
 * @method IntegrationInstance|Builder shopify
 * @method IntegrationInstance|Builder starshipit
 * @method static IntegrationInstance|Builder magento
 */
class IntegrationInstance extends Model implements Filterable, IntegrationInstanceInterface
{
    use HasFactory;
    use HasFilters;

    protected $table = 'integration_instances';

    protected $primaryKey = 'id';

    const SETTINGS_OPTION_NAME_NEITHER = 'Neither';

    const SETTINGS_OPTION_ID_NEITHER = 'neither';

    const MAX_RULE_TYPE_PERCENT_AVAILABLE = '% of Available';

    const MAX_RULE_TYPE_FIXED_ALLOCATION = 'Fixed Allocation';

    const MAX_RULE_TYPE_NONE = 'None';

    const ORDER_CONDITION_CREATED_AFTER = 'created_after';

    const GIFT_CARD_NOTE_SALES_ORDER_CUSTOM_FIELD = 'gift_card_note_sales_order_custom_field_id';

    protected $fillable = [
        'integration_id',
        'name',
        'connection_settings',
        'integration_settings',
        'is_automatic_sync_enabled',
        'country'
    ];

    protected $casts = [
        'connection_settings' => 'array',
        'integration_settings' => 'array',
        'is_automatic_sync_enabled' => 'boolean',
        'sync_status' => IntegrationInstanceSyncStatusEnum::class,
    ];

    public function scopeAmazon($query)
    {
        return $query->whereHas('integration', function ($q) {
            $q->where('name', Integration::NAME_AMAZON_US);
        });
    }

    public function scopeShopify($query)
    {
        return $query->whereHas('integration', function ($q) {
            $q->where('name', Integration::NAME_SHOPIFY);
        });
    }

    public function scopeWooCommerce($query)
    {
        return $query->whereHas('integration', function ($q) {
            $q->where('name', Integration::NAME_WOOCOMMERCE);
        });
    }

    public function scopeEbay($query)
    {
        return $query->whereHas('integration', function ($q) {
            $q->where('name', Integration::NAME_EBAY);
        });
    }

    public function scopeShipstation($query)
    {
        return $query->whereHas('integration', function ($q) {
            $q->where('name', Integration::NAME_SHIPSTATION);
        });
    }

    public function scopeStarshipit($query)
    {
        return $query->whereHas('integration', function ($q) {
            $q->where('name', Integration::NAME_STARSHIPIT);
        });
    }

    public function scopeShipMyOrders($query)
    {
        return $query->whereHas('integration', function ($q) {
            $q->where('name', Integration::NAME_SHIPMYORDERS);
        });
    }

    public function scopeMagento($query)
    {
        return $query->whereHas('integration', function ($q) {
            $q->where('name', Integration::NAME_MAGENTO);
        });
    }

    public function integrationInstanceServiceDisruptions()
    {
        return $this->hasMany(IntegrationInstanceServiceDisruptions::class, 'integration_instance_id');
    }

    public function integration()
    {
        return $this->belongsTo(Integration::class, 'integration_id');
    }

    public function salesChannel()
    {
        return $this->hasOne(SalesChannel::class, 'integration_instance_id');
    }

    public function warehouse(): HasOne
    {
        return $this->hasOne(Warehouse::class, 'integration_instance_id')->where('type', Warehouse::TYPE_AMAZON_FBA);
    }

    public function shippingProviderShippingMethodMappings()
    {
        return $this->hasMany(ShippingMethodMappingsSkuToShippingProviderMethod::class, 'shipping_provider_id');
    }

    public function storeMappings()
    {
        return $this->hasMany(StoreMapping::class, 'integration_instance_id');
    }

    public function apiLogs()
    {
        return $this->hasMany(ApiLog::class, 'integration_instance_id');
    }

    public function getConnectionSetting(string $settingName): mixed
    {
        return data_get($this->connection_settings, $settingName);
    }

    public function setConnectionSetting(string $settingName, mixed $value): void
    {
        $connectionSettings = $this->getAttribute('connection_settings');
        data_set($connectionSettings, $settingName, $value);
        $this->setAttribute('connection_settings', $connectionSettings);
    }

    public function getIntegrationSetting(string $settingName): mixed
    {
        return data_get($this->getAttribute('integration_settings'), 'settings.'.$settingName);
    }

    public function setIntegrationSetting(string $settingName, mixed $value): void
    {
        $connectionSettings = $this->getAttribute('integration_settings');
        data_set($connectionSettings, 'settings.'.$settingName, $value);
        $this->setAttribute('integration_settings', $connectionSettings);
    }

    /**
     * @throws Throwable
     */
    public function delete()
    {
        DB::transaction(function () {
            $settings = $this->connection_settings;
            $settings['deleting'] = true;

            $this->connection_settings = $settings;
            $this->save();

            // Delete Xero related data
            if ($this->isXero()) {
                /*
                 * clear accounting_integration_id, accounting_integration_type from any tables
                 */
                $this->resetXeroLinks();
                $this->truncateXeroData();
            }

            // Delete QBO related data
            if ($this->isQbo()) {
                /*
                 * clear accounting_integration_id, accounting_integration_type from any tables
                 */
                $this->resetQboLinks();
                $this->truncateQboData();
            }

            // Delete sales channel and its sales orders and product listings
            if ($this->salesChannel) {
                $this->deleteListingsFromSalesChannel();
                $this->deleteOrdersFromSalesChannel();
                $this->salesChannel->delete();
            }

            // Delete amazon reports
            /*if ($this->isAmazonInstance()) {
                $this->cleanAmazonData();

                // If is last Amazon instance, we remove amazon payment type
                $count = self::with([])
                             ->whereHas('integration', function (Builder $builder) {
                                 $builder->where('name', Integration::NAME_AMAZON_US);
                             })->count();

                if ($count === 1) {
                    $amazonPaymentType = PaymentType::with([])->where('name', PaymentType::PAYMENT_TYPE_AMAZON)->first();
                    if ($amazonPaymentType) {
                        // Last Amazon instance is being deleted,
                        // we remove the Amazon payment method
                        if (Payment::query()->where('payment_type_id', $amazonPaymentType->id)->count()) {
                            $defaultPaymentType = Helpers::setting(Setting::KEY_SO_DEFAULT_PAYMENT_TYPE) ?: PaymentType::query()->valueOrFail('id');
                            Payment::query()->where('payment_type_id', $amazonPaymentType->id)->update(['payment_type_id' => $defaultPaymentType]);
                        }
                        $amazonPaymentType->delete();
                    }
                }
            }*/
            if ($this->isShipStation()) {
                $shipStationManager = new ShipStationManager();
                $shipStationManager->deleteWebhooks($this);
            }
            if ($this->isShopify()) {
                // If is last Shopify instance, we remove shopify payment type
                $count = self::with([])->shopify()->count();

                if ($count === 1) {
                    // Last Shopify instance is being deleted,
                    // we remove the Shopify Payment method
                    PaymentType::with([])->where('name', PaymentType::PAYMENT_TYPE_SHOPIFY)->delete();
                }

                // delete inventory_items/delete webhook
                Artisan::call('sku:shopify:delete-webhooks', ['integrationInstance' => $this->id]);
            }

            if ($this->isMagento()) {
                CustomerGroup::query()->where('integration_instance_id', $this->id)->delete();
                InventorySource::query()->where('integration_instance_id', $this->id)->delete();
                \App\Models\Magento\ProductAttribute::query()->where('integration_instance_id', $this->id)->delete();
                \App\Models\Magento\Store::query()->where('integration_instance_id', $this->id)->delete();
            }

            // delete store mappings
            StoreMapping::query()->where('integration_instance_id', $this->id)->delete();

            // delete import mappings
            DataImportMapping::query()->where('integration_instance_id', $this->id)->delete();

            if ($this->warehouse) {
                $this->warehouse()->delete();
            }

            // delete shipping method mappings of shipping provider integrations
            if ($this->integration->integration_type == Integration::TYPE_SHIPPING_PROVIDER) {
                $this->shippingProviderShippingMethodMappings()->delete();
            }

            return parent::delete();
        });
    }

    public function isSalesChannel(): bool
    {
        return $this->integration->isSalesChannel();
    }

    public function isAmazonInstance(): bool
    {
        return $this->integration->name == Integration::NAME_AMAZON_US;
    }

    public function isShopify(): bool
    {
        return $this->integration->name === Integration::NAME_SHOPIFY;
    }

    public function isShipStation(): bool
    {
        return $this->integration->name === Integration::NAME_SHIPSTATION;
    }

    public function isLocal(): bool
    {
        return $this->integration->name === Integration::NAME_SKU_IO;
    }

    public function isMagento(): bool
    {
        return $this->integration->name === Integration::NAME_MAGENTO;
    }

    public function isVeracore(): bool
    {
        return $this->integration->name === Integration::NAME_VERACORE;
    }

    public function isSMO(): bool
    {
        return $this->integration->name === Integration::NAME_SHIPMYORDERS;
    }

    public function isEbay(): bool
    {
        return strtolower($this->integration->name) === strtolower(Integration::NAME_EBAY);
    }

    public function isWooCommerce(): bool
    {
        return strtolower($this->integration->name) === strtolower(Integration::NAME_WOOCOMMERCE);
    }

    public function isXero(): bool
    {
        return $this->integration->name === Integration::NAME_XERO;
    }

    public function isQbo(): bool
    {
        return $this->integration->name === Integration::NAME_QBO;
    }

    public function isFinalized(): bool
    {
        if ($this->isXero()) {
            if (isset($this->connection_settings['tenant_id'])) {
                return true;
            }
        }

        return false;
    }

    public function isAccounting(): bool
    {
        return $this->integration->integration_type === Integration::TYPE_ACCOUNTING;
    }

    /**
     * Deletes amazon reports and products related to the
     * integration instance.
     */
    public function cleanAmazonData(): void
    {
        AmazonReportLog::with([])->where('integration_instance_id', $this->id)->delete();
        AmazonOrder::with([])->where('integration_instance_id', $this->id)->delete();
        FBAInitialInventory::with([])->where('integration_id', $this->id)->delete();

        // delete all data that related to the fba warehouse
        $fbaWarehouseId = $this->warehouse()->where('type', Warehouse::TYPE_AMAZON_FBA)->value('id');
        FifoLayer::query()->where('warehouse_id', $fbaWarehouseId)->eachById(fn (FifoLayer $fifoLayer) => $fifoLayer->delete());
        InventoryAdjustment::query()->where('warehouse_id', $fbaWarehouseId)->eachById(fn (InventoryAdjustment $adjustment) => $adjustment->delete());
        PurchaseOrder::query()->where('destination_warehouse_id', $fbaWarehouseId)->eachById(fn (PurchaseOrder $purchaseOrder) => $purchaseOrder->delete());
        SalesCredit::query()->where('to_warehouse_id', $fbaWarehouseId)->eachById(fn (SalesCredit $salesCredit) => $salesCredit->delete());
        SalesCreditReturn::query()->where('warehouse_id', $fbaWarehouseId)->eachById(fn (SalesCreditReturn $salesCredit) => $salesCredit->delete());
        SalesOrderFulfillment::query()->where('warehouse_id', $fbaWarehouseId)->eachById(fn (SalesOrderFulfillment $fulfillment) => $fulfillment->delete(true));
        SalesOrderLine::query()->where('warehouse_id', $fbaWarehouseId)->eachById(fn (SalesOrderLine $salesOrderLine) => $salesOrderLine->delete());
        WarehouseTransfer::query()->where('to_warehouse_id', $fbaWarehouseId)->eachById(fn (WarehouseTransfer $warehouseTransfer) => $warehouseTransfer->delete());
        WarehouseTransfer::query()->where('from_warehouse_id', $fbaWarehouseId)->eachById(fn (WarehouseTransfer $warehouseTransfer) => $warehouseTransfer->delete());

        ProductListing::with([])->where('sales_channel_id', $this->salesChannel->id)->update(['document_id' => null]);
    }

    public function hasFBA(): bool
    {
        if (! $this->isAmazonInstance()) {
            return false;
        }

        $settings = $this->integration_settings;

        return $settings && isset($settings['fba']) && $settings['fba']['isFbaEnabled'] == true;
    }

    /**
     * Deletes the listings for the integration instance.
     */
    protected function deleteListingsFromSalesChannel()
    {
        /** @var Model $modelPath */
        $modelPath = $this->integration->getProductsModelPath();
        $modelPath::with([])->where('integration_instance_id', $this->id)->delete();
    }

    /**
     * Deletes the orders for the integration instance.
     */
    protected function deleteOrdersFromSalesChannel()
    {
        /** @var Model $modelPath */
        $modelPath = $this->integration->getOrderModelPath();
        $modelPath::with([])->where('integration_instance_id', $this->id)->delete();
    }

    /**
     * Merge Developer(integration) credentials with integration instance connection settings.
     */
    public function getMagentoMappingsAttribute(): ?Collection
    {
        if ($this->isMagento()) {
            return $this->storeMappings;
        }

        return null;
    }

    public function getCredentialsAttribute()
    {
        return array_merge($this->connection_settings ?: [], $this->integration->credentials);
    }

    public function getPaymentTypeIdAttribute()
    {
        return $this->integration_settings['settings']['payment_type'] ??
           Helpers::setting(Setting::KEY_SO_DEFAULT_PAYMENT_TYPE) ?:
             PaymentType::with([])->first()->id ??
             null;
    }

    public function fbaInventoryTrackingStartDate(): ?Carbon
    {
        if (! isset($this->integration_settings['fba']['fbaInventoryTrackingStartDate'])) {
            return null;
        }

        return Carbon::parse($this->integration_settings['fba']['fbaInventoryTrackingStartDate']);
    }

    public function isStartedInventoryTracking(): bool
    {
        return isset($this->integration_settings['fba']['startedInventoryTracking']) && $this->integration_settings['fba']['startedInventoryTracking'] == true;
    }

    public function getOrderStartDateAttribute()
    {
        if (empty($this->integration_settings['orders']['date'])) {
            return null;
        }

        return Carbon::parse($this->integration_settings['orders']['date'], 'UTC');
    }

    public function getOpenStartDateAttribute()
    {
        if (empty($this->integration_settings['orders']['download']['open_start_date'])) {
            return null;
        }

        return Carbon::parse($this->integration_settings['orders']['download']['open_start_date'], 'UTC');
    }

    public function getClosedStartDateAttribute()
    {
        if (empty($this->integration_settings['orders']['download']['closed_start_date'])) {
            return null;
        }

        return Carbon::parse($this->integration_settings['orders']['download']['closed_start_date'], 'UTC');
    }

    public function getAuditTrailStartDateAttribute()
    {
        if (empty($this->integration_settings['orders']['download']['audit_trail_start_date'])) {
            return null;
        }

        return Carbon::parse($this->integration_settings['orders']['download']['audit_trail_start_date'], 'UTC');
    }

    public function ordersLastDownloaded(Carbon $date)
    {
        $settings = $this->integration_settings;

        $settings['orders']['download']['lastSyncedAt'] = $date;
        $settings['orders']['download']['options_changed'] = false;

        $this->integration_settings = $settings;
        $this->save();
    }

    /**
     * Handles amazon payment type setup.
     */
    public function handleAmazonPaymentTypeSetup()
    {
        $paymentType = PaymentType::with([])->where('name', PaymentType::PAYMENT_TYPE_AMAZON)
            ->first();

        if (! $paymentType) {
            $type = new PaymentType;
            $type->name = PaymentType::PAYMENT_TYPE_AMAZON;
            $type->save();
        }
    }

    /**
     * Handles Shopify payment type setup.
     */
    public function handleShopifyPaymentTypeSetup()
    {
        $paymentType = PaymentType::with([])->where('name', PaymentType::PAYMENT_TYPE_SHOPIFY)
            ->first();

        if (! $paymentType) {
            $type = new PaymentType;
            $type->name = PaymentType::PAYMENT_TYPE_SHOPIFY;
            $type->save();
        }
    }

    /**
     * {@inheritDoc}
     */
    public function save(array $options = [])
    {
        if ($this->isDirty('integration_settings')) {
            $integrationSettings = $this->integration_settings;
            $oldIntegrationSettings = $this->getOriginal('integration_settings');

            // set a flag to indicate if the download order options changed
            if (! empty($integrationSettings['orders']['date']) && ! empty($integrationSettings['orders']['order_statuses'])) {
                if ($integrationSettings['orders']['date'] != ($oldIntegrationSettings['orders']['date'] ?? null) ||
                                                                                                         ! empty(array_diff($integrationSettings['orders']['order_statuses'], $oldIntegrationSettings['orders']['order_statuses'] ?? [])) ||
                                                                                                         ! empty(array_diff($oldIntegrationSettings['orders']['order_statuses'] ?? [], $integrationSettings['orders']['order_statuses']))) {
                    $integrationSettings['orders']['download']['options_changed'] = true;

                    $this->integration_settings = $integrationSettings;
                }
            }

            if ($this->integration->isSalesChannel() && $this->isAmazonInstance()) {
                // We don't allow emailing amazon customers.
                $settings = $this->integration_settings;
                $settings['emailCustomers'] = false;
                $this->integration_settings = $settings;
            }
        }

        // mark the connection settings as valid
        if ($this->isDirty('connection_settings') && ($this->getOriginal('connection_settings')['invalid'] ?? false)) {
            $this->connection_settings = array_merge($this->connection_settings, ['invalid' => $this->connection_settings['invalid'] ?? false]);
        }

        return parent::save($options);
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return ['name'];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * @return bool
     */
    public function hasInventoryLocations(): bool
    {

        $settings = is_string($this->integration_settings) ?
            json_decode($this->integration_settings, true) :
            $this->integration_settings;

        return !empty($settings) &&
            !empty(@$settings['inventory']) &&
            !empty(@$settings['inventory']['locations']);
    }

    /*
     * This function is required after updating order settings in order to sanitize the dates
     */
    public function setOrderStartDate()
    {
        if (@$this->integration_settings['orders']['download']['open_start_date'] && @$this->integration_settings['orders']['download']['closed_start_date']) {
            $openStartDate = Carbon::parse(@$this->integration_settings['orders']['download']['open_start_date']);
            $closedStartDate = Carbon::parse(@$this->integration_settings['orders']['download']['closed_start_date']);
            $orderArray = $this->integration_settings['orders'];

            if ($openStartDate->copy()->getTimestamp() < $closedStartDate->copy()->getTimestamp()) {
                $orderStartDate = $openStartDate;
            } else {
                $orderStartDate = $closedStartDate;
            }

            $orderArray['date'] = $orderStartDate;

            $this->integration_settings = array_merge($this->integration_settings, ['orders' => $orderArray]);

            $this->update();
        }
    }

    protected function resetQboLinks()
    {
        AccountingTransaction::query()
            ->whereIn('accounting_integration_type', [QboInvoice::class, QboBill::class, QboPurchaseOrder::class])
            ->update(
                [
                    'accounting_integration_id' => null,
                    'accounting_integration_type' => null,
                ]
            );

        Payment::query()
            ->where('accounting_integration_type', QboPayment::class)
            ->update(
                [
                    'accounting_integration_id' => null,
                    'accounting_integration_type' => null,
                ]
            );

        PaymentType::query()
            ->where('accounting_integration_type', QboAccount::class)
            ->update(
                [
                    'accounting_integration_id' => null,
                    'accounting_integration_type' => null,
                ]
            );

        TaxRate::query()
            ->where('accounting_integration_type', QboTaxRate::class)
            ->update(
                [
                    'accounting_integration_id' => null,
                    'accounting_integration_type' => null,
                ]
            );
    }

    protected function resetXeroLinks()
    {
        AccountingTransaction::query()
            ->where('accounting_integration_type', XeroTransaction::class)
            ->update(
                [
                    'accounting_integration_id' => null,
                    'accounting_integration_type' => null,
                ]
            );

        Payment::query()
            ->where('accounting_integration_type', XeroPayment::class)
            ->update(
                [
                    'accounting_integration_id' => null,
                    'accounting_integration_type' => null,
                ]
            );

        PaymentType::query()
            ->where('accounting_integration_type', XeroAccount::class)
            ->update(
                [
                    'accounting_integration_id' => null,
                    'accounting_integration_type' => null,
                ]
            );

        TaxRate::query()
            ->where('accounting_integration_type', XeroTaxRate::class)
            ->update(
                [
                    'accounting_integration_id' => null,
                    'accounting_integration_type' => null,
                ]
            );
    }

    protected function truncateXeroData()
    {
        XeroAccount::query()->delete();
        XeroContact::query()->delete();
        XeroPayment::query()->delete();
        XeroTaxRate::query()->delete();
        XeroTransaction::query()->delete();
    }

    protected function truncateQboData()
    {
        QboInvoice::query()->delete();
        QboBill::query()->delete();
        QboPurchaseOrder::query()->delete();
        QboCustomer::query()->delete();
        QboVendor::query()->delete();
        QboJournal::query()->delete();
        QboPayment::query()->delete();
        QboItem::query()->delete();
        QboTaxRate::query()->delete();
        QboAccount::query()->delete();
    }

    public function getMasterOfStock(): ?string
    {
        $masterOfStock = $this->getInventoryData()->masterOfStock;
        return $masterOfStock instanceof Optional ? null : $masterOfStock;
    }

    public function getSalesNominalCodeId(): ?int
    {
        return $this->integration_settings['sales_nominal_code_id'] ?? null;
    }

    public function getInventoryData(): IntegrationInstanceInventoryData
    {
        $settings = $this->integration_settings;

        return IntegrationInstanceInventoryData::from($settings['inventory'] ?? []);
    }

    public function getMasterOfPrice(): ?string
    {
        return $this->integration_settings['pricing']['masterOfPrice']['id'] ?? null;
    }

    public function getPricingTierId()
    {
        return $this->integration_settings['pricing']['pricingTier']['id'] ?? null;
    }

    /**
     * Mark connection settings as invalid
     */
    public function unauthorizedConnection(): void
    {
        $connectionSettings = $this->connection_settings;
        $connectionSettings['invalid'] = true;

        $this->connection_settings = $connectionSettings;
        $this->save();
    }

    public function refreshToken(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => @$this->connection_settings['refresh_token']
        );
    }

    // TODO: eventually remove this as we properly instantiate XeroIntegrationInstance everywhere
    public function hasRemainingDailyApiCalls(): bool
    {
        if (! isset($this->integration_settings['remainingApiUsage'])) {
            return true;
        }
        $dailyResetAt = isset($this->integration_settings['dailyResetAt']) ? Carbon::parse($this->integration_settings['dailyResetAt']) : Carbon::now();
        if ($this->integration_settings['remainingApiUsage'] == 0 && Carbon::now()->lt($dailyResetAt)) {
            return false;
        }

        return true;
    }
}
