<?php

namespace App\Models;

use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * Class Incoterm.
 *
 *
 * @property int $id
 * @property string $code
 * @property string|null $description
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $archived_at
 */
class Incoterm extends Model implements Filterable, Sortable
{
    use Archive, HasFilters, HasSort;
    use HasFactory;

    protected $guarded = [];

    protected $casts = ['archived_at' => 'datetime'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
 */

    public function purchaseOrders()
    {
        return $this->hasMany(PurchaseOrder::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * {@inheritDoc}
     */
    public function delete()
    {
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return parent::delete();
    }

    /**
     * Determine if the incoterm is used.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        $usage = [];

        // used in purchase orders
        $this->loadCount('purchaseOrders');
        if ($this->purchase_orders_count) {
            $usage['purchaseOrders'] = trans_choice('messages.currently_used', $this->purchase_orders_count, [
                'resource' => 'purchase order',
                'model' => 'incoterm('.$this->code.')',
            ]);
        }

        return count($usage) ? $usage : false;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.incoterm.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['description', 'code'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }
}
