<?php

namespace App\Models;

use App\DTO\ReportingDailyFinancialDto;
use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Models\Abstractions\AbstractReportable;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use App\Repositories\FinancialLineTypeRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * @property int $id
 * @property string $name
 * @property FinancialLineClassificationEnum $classification
 * @property bool $allocate_to_products
 * @property FinancialLineProrationStrategyEnum $proration_strategy
 * @property int $nominal_code_id
 * @property-read EloquentCollection|FinancialLine[] $financialLines
 * @property-read NominalCode $nominalCode
 */
class FinancialLineType extends AbstractReportable implements Filterable
{
    use HasFactory,
        HasFilters;

    protected $guarded = [];

    protected $casts = [
        'classification' => FinancialLineClassificationEnum::class,
        'proration_strategy' => FinancialLineProrationStrategyEnum::class,
    ];

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function financialLines(): HasMany
    {
        return $this->hasMany(FinancialLine::class);
    }

    public function nominalCode(): BelongsTo
    {
        return $this->belongsTo(NominalCode::class);
    }

    public function save(array $options = []): bool
    {
        if (! isset($this->proration_strategy)) {
            $this->proration_strategy = $this->classification == FinancialLineClassificationEnum::REVENUE ?
                FinancialLineProrationStrategyEnum::REVENUE_BASED : FinancialLineProrationStrategyEnum::COST_BASED;
        }

        return parent::save($options);
    }

    public function getReportingDailyFinancialDto(Carbon $date): ?ReportingDailyFinancialDto
    {
        return app(FinancialLineTypeRepository::class)->getFinancialLineFinancialsSummaryForFinancialLineTypeDate($this, $date);
    }

    public function availableColumns()
    {
        return ['id', 'name'];
    }

    public function filterableColumns(): array
    {
        return ['id', 'name'];
    }

    public function generalFilterableColumns(): array
    {
        return [];
    }
}
