<?php

namespace App\Models;

use App\Abstractions\FinancialDocumentLineInterface;
use App\Abstractions\UniqueFieldsInterface;
use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Models\Concerns\BulkImport;
use App\Repositories\SalesOrderLineFinancialsRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * @property int $id
 * @property int $sales_order_id
 * @property int $financial_line_type_id
 * @property int $nominal_code_id
 * @property string $description
 * @property int $quantity
 * @property float $amount
 * @property float $amount_in_tenant_currency
 * @property-read float $extended_amount
 * @property-read float $extended_amount_in_tenant_currency
 * @property float $tax_allocation
 * @property float $tax_allocation_in_tenant_currency
 * @property int $tax_rate_id
 * @property float $tax_rate
 * @property bool $allocate_to_products
 * @property FinancialLineProrationStrategyEnum $proration_strategy
 * @property ?int $allocate_to_id
 * @property ?string $allocate_to_type
 * @property ?string $sales_channel_line_id
 * @property ?string $external_source_id
 * @property ?string $external_source_type
 * @property ?Carbon $created_at
 * @property ?Carbon $updated_at
 * @property-read FinancialLineType $financialLineType
 * @property-read NominalCode $nominalCode
 * @property-read SalesOrder $salesOrder
 */
class FinancialLine extends Model implements FinancialDocumentLineInterface, UniqueFieldsInterface
{
    use BulkImport;
    use BulkImport;
    use HasFactory;

    protected $guarded = [];

    protected $casts = [
        'proration_strategy' => FinancialLineProrationStrategyEnum::class,
    ];

    public static function getUniqueFields(): array
    {
        return [
            'sales_order_id',
            'financial_line_type_id',
            'description',
            'quantity',
            'amount',
            'proration_strategy',
            'sales_channel_line_id',
            'allocate_to_id',
            'allocate_to_type',
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function financialLineType(): BelongsTo
    {
        return $this->belongsTo(FinancialLineType::class);
    }

    public function nominalCode(): BelongsTo
    {
        return $this->belongsTo(NominalCode::class);
    }

    public function salesOrder(): BelongsTo
    {
        return $this->belongsTo(SalesOrder::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function save(array $options = []): bool
    {
        if (! isset($this->description)) {
            $this->description = $this->financialLineType->name;
        }

        if (! isset($this->allocate_to_products)) {
            $this->allocate_to_products = $this->financialLineType->allocate_to_products;
        }

        if (! isset($this->proration_strategy)) {
            $this->proration_strategy = $this->financialLineType->proration_strategy;
        }

        if (! isset($this->nominal_code_id)) {
            $this->nominal_code_id = $this->financialLineType->nominal_code_id;
        }

        $this->invalidateLines();

        return parent::save($options);
    }

    public function delete(): ?bool
    {
        $this->invalidateLines();

        return parent::delete();
    }

    public function invalidateLines(): void
    {
        if ($this->salesOrder->salesOrderLines->isNotEmpty()) {
            app(SalesOrderLineFinancialsRepository::class)->invalidateForSalesOrderLineIds($this->salesOrder->salesOrderLines->pluck('id')->toArray());
        }
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeAllocatable(Builder $builder, bool $allocateToProducts = true): Builder
    {
        return $builder->where('allocate_to_products', $allocateToProducts);
    }

    public function scopeRevenue(Builder $builder): Builder
    {
        return $builder->whereHas('financialLineType', function (Builder $builder) {
            $builder->where('classification', FinancialLineClassificationEnum::REVENUE->value);
        });
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function amountInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->amount * $this->salesOrder->currency_rate);
    }

    public function extendedAmountInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->extended_amount * $this->salesOrder->currency_rate);
    }

    public function taxAllocationInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->tax_allocation * $this->salesOrder->currency_rate);
    }
}
