<?php

namespace App\Models;

use App\Contracts\HasReference;
use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Exporters\MapsExportableFields;
use App\Importers\DataImporter;
use App\Importers\DataImporters\InitialInventoryDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Concerns\LogsActivity;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Repositories\SalesOrderLineFinancialsRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Spatie\Activitylog\LogOptions;

/**
 * Class FifoLayer.
 *
 *
 * @property int $id
 * @property Carbon $fifo_layer_date
 * @property int $product_id
 * @property float $original_quantity
 * @property float $fulfilled_quantity
 * @property float $available_quantity
 * @property float $total_cost
 * @property int $warehouse_id
 * @property int $link_id
 * @property string $link_type
 * @property string|null $link_reference
 * @property Product $product
 * @property Warehouse|null $warehouse
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $archived_at
 * @property-read Collection|InventoryMovement[] $inventoryMovements
 * @property-read float $unfulfilled_quantity
 * @property-read float $avg_cost
 * @property-read float $actual_cost
 */
class FifoLayer extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable
{
    use Archive,
        BulkImport,
        HandleDateTimeAttributes,
        HasFactory,
        HasFilters,
        HasParentLink,
        HasSort,
        LogsActivity;

    /**
     * Link types for send in request only.
     */
    const REQUEST_LINK_TYPE_PO = 'purchase_order';

    const REQUEST_LINK_TYPE_IB = 'inventory_bundling';

    const REQUEST_LINK_TYPE_IM = 'inventory_movement';

    const REQUEST_LINK_TYPE_IA = 'inventory_adjustment';

    const REQUEST_LINK_TYPE_SC = 'sales_credit';

    const REQUEST_LINK_TYPE_WT = 'warehouse_transfer';

    const REQUEST_LINK_TYPE_ST = 'stock_take';

    const REQUEST_LINK_TYPE_IAS = 'assembly';

    const REQUEST_LINK_TYPES = [
        PurchaseOrderShipmentReceiptLine::class => self::REQUEST_LINK_TYPE_PO,
        // This should be first in PO group for filters to work.
        PurchaseOrderLine::class => self::REQUEST_LINK_TYPE_PO,
        InventoryMovement::class => self::REQUEST_LINK_TYPE_IM,
        InventoryAdjustment::class => self::REQUEST_LINK_TYPE_IA,
        SalesCreditReturnLine::class => self::REQUEST_LINK_TYPE_SC,
        WarehouseTransferShipmentReceiptLine::class => self::REQUEST_LINK_TYPE_WT,
        StockTakeItem::class => self::REQUEST_LINK_TYPE_ST,
        InventoryAssemblyLine::class => self::REQUEST_LINK_TYPE_IAS,
    ];

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    protected $casts = [
        'fifo_layer_date' => 'datetime',
        'original_quantity' => 'float',
        'fulfilled_quantity' => 'float',
        'total_cost' => 'float',
    ];

    protected $guarded = [];

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [];
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->id;
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function link(): MorphTo
    {
        return $this->morphTo();
    }

    public function inventoryMovements(): MorphMany
    {
        return $this->morphMany(InventoryMovement::class, 'layer');
    }

    public function salesOrderLines()
    {
        return $this->morphToMany(SalesOrderLine::class, 'layer', 'sales_order_line_layers')
            ->using(SalesOrderLineLayer::class)
            ->withPivot('quantity')
            ->withTimestamps();
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getAvgCostAttribute(): float
    {
        return $this->original_quantity != 0 ? $this->total_cost / $this->original_quantity : 0;
    }

    public function getActualCostAttribute(): float
    {
        return $this->avg_cost * $this->available_quantity;
    }

    public function availableQuantity(): \Illuminate\Database\Eloquent\Casts\Attribute{
        return \Illuminate\Database\Eloquent\Casts\Attribute::get(
            fn() => $this->original_quantity - $this->fulfilled_quantity
        );
    }

    public function getLinkNameAttribute(): ?string
    {
        if (! isset(self::REQUEST_LINK_TYPES[$this->link_type])) {
            return null;
        }

        return self::REQUEST_LINK_TYPES[$this->link_type];
    }

    public function getLinkReferenceAttribute()
    {
        if (! $this->link_type || ! class_exists($this->link_type)) {
            return null;
        }

        if ($this->link_type && $this->link instanceof HasReference) {
            return $this->link->getReference();
        }

        return null;
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * @throws OversubscribedFifoLayerException
     * @throws Exception
     */
    public function save(array $options = [], bool $allowOverage = false): bool
    {
        // Unarchive if already archived and there is unfulfilled quantity.
        if ($this->available_quantity != 0 && $this->isArchived()) {
            $this->archived_at = null;
        }

        if ($this->fulfilled_quantity > $this->original_quantity) {
            if (!$allowOverage) {
                throw new OversubscribedFifoLayerException($this, 'Fulfilled quantity cannot exceed original quantity for fifo layer: '.$this->id);
            }
        } elseif ($this->fulfilled_quantity < 0) {
            throw new Exception('Fulfilled quantity cannot be negative for fifo layer: '.$this->id);
        } elseif ($this->original_quantity < 0) {
            throw new Exception('Original quantity cannot be negative for fifo layer: '.$this->id);
        }

        return parent::save($options);
    }

    public function delete()
    {
        return DB::transaction(function () {
            $this->load(['inventoryMovements']);
            $this->inventoryMovements->each->delete();

            $salesOrderLineLayersQuery = SalesOrderLineLayer::with([])->where('layer_id', $this->id)
                ->where('layer_type', self::class);

            $salesOrderLineIds = $salesOrderLineLayersQuery->pluck('sales_order_line_id')->toArray();
            app(SalesOrderLineFinancialsRepository::class)->invalidateForSalesOrderLineIds($salesOrderLineIds);

            $salesOrderLineLayersQuery->delete();

            // Only delete fifo layer if all movements and sales order line layers are deleted.
            $hasLineLayers = SalesOrderLineLayer::with([])->where('layer_id', $this->id)
                ->where('layer_type', self::class)->count() > 0;
            if ($this->inventoryMovements()->count() > 0 || $hasLineLayers) {
                throw new Exception("Unable to delete Fifo Layer: {$this->id} due to existing movements or sales order line layers.");
            }

            return parent::delete();
        });
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.fifo_layer.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['id', 'product.sku'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())
            ->where('sortable', 1)
            ->pluck('data_name')
            ->all();
    }

    private function makeLinkTypeFromName($name): array
    {
        if (in_array($name, self::REQUEST_LINK_TYPES)) {
            return array_keys(self::REQUEST_LINK_TYPES, $name);
        }

        return [__NAMESPACE__.'\\'.ucfirst(Str::camel($name))];
    }

    public function scopeFilterLink(Builder $builder, array $relation, string $operator, string $value, $conjunction): Builder
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';
        $builder->{$function}('link_type', $operator, $this->makeLinkTypeFromName($value));

        return $builder;
    }

    public function scopeSortLink(Builder $builder, array $relation, bool $ascending): Builder
    {
        $builder->orderBy('link_type', $ascending ? 'asc' : 'desc');

        return $builder;
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', false, 'ID'),
            'sku' => Exporter::makeExportableField('sku', false, 'SKU'),
            'product_name' => Exporter::makeExportableField('product_name', false),
            'original_quantity' => Exporter::makeExportableField('original_quantity', false),
            'fulfilled_quantity' => Exporter::makeExportableField('fulfilled_quantity', false),
            'total_cost' => Exporter::makeExportableField('total_cost', false),
            'origin.name' => Exporter::makeExportableField('origin_link_name', false),
            'origin.link_id' => Exporter::makeExportableField('origin_link_id', false),
            'origin.reference' => Exporter::makeExportableField('origin_link_reference', false),
        ];
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new InitialInventoryDataImporter(null, $filePath);
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeFilterPurchaseOrderForeignCurrency(Builder $builder, $operator, $value, $conjunction): Builder
    {
        if ($value) {
            $builder->whereHasMorph('link', PurchaseOrderShipmentReceiptLine::class, function ($query) {
                $query->whereHas('purchaseOrderShipmentLine', function ($query) {
                    $query->whereHas('purchaseOrderLine', function ($query) {
                        $query->whereHas('purchaseOrder', function ($query) {
                            $query->whereHas('currency', function ($query) {
                                $query->where('is_default', 0);
                            });
                        });
                    });
                });
            });
        } else {
            $builder->whereHasMorph('link', PurchaseOrderShipmentReceiptLine::class, function ($query) {
                $query->whereHas('purchaseOrderShipmentLine', function ($query) {
                    $query->whereHas('purchaseOrderLine', function ($query) {
                        $query->whereHas('purchaseOrder', function ($query) {
                            $query->whereHas('currency', function ($query) {
                                $query->where('is_default', 1);
                            });
                        });
                    });
                });
            });
        }

        return $builder;
    }

    public function scopeFilterPurchaseOrderHasDiscount(Builder $builder, $operator, $value, $conjunction): Builder
    {
        if ($value) {
            $builder->whereHasMorph('link', PurchaseOrderShipmentReceiptLine::class, function ($query) {
                $query->whereHas('purchaseOrderShipmentLine', function ($query) {
                    $query->whereHas('purchaseOrderLine', function ($query) {
                        $query->whereHas('purchaseOrder', function ($query) {
                            $query->where('discount_amount', '!=', 0);
                        });
                    });
                });
            });
        }

        return $builder;
    }

    public function scopeFilterPurchaseOrderHasNonProductLine(Builder $builder, $operator, $value, $conjunction): Builder
    {
        if ($value) {
            $builder->whereHasMorph('link', PurchaseOrderShipmentReceiptLine::class, function ($query) {
                $query->whereHas('purchaseOrderShipmentLine', function ($query) {
                    $query->whereHas('purchaseOrderLine', function ($query) {
                        $query->whereHas('purchaseOrder', function ($query) {
                            $query->whereHas('purchaseOrderLines', function ($query) {
                                $query->whereNull('product_id');
                            });
                        });
                    });
                });
            });
        }

        return $builder;
    }

    /**
     * @throws Exception
     */
    public function calculatedAvailableQuantity(): float
    {
        if (!$this->loadExists('inventoryMovements')) {
            throw new Exception("Inventory movements must be eager loaded for calculatedAvailableQuantity");
        }
        return $this->inventoryMovements->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)->sum('quantity');
    }
}
