<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Casts\EncryptValue;
use App\Exporters\MapsExportableFields;
use App\Exporters\TransformsExportData;
use App\Helpers;
use App\Importers\DataImporter;
use App\Importers\DataImporters\CustomerDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Repositories\CustomerRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

/**
 * Class Customer.
 *
 *
 * @property int $id
 * @property int $default_shipping_address_id
 * @property int|null $default_billing_address_id
 * @property string $name
 * @property string $firstName
 * @property string $middleInitial
 * @property string $lastName
 * @property string|null $email
 * @property string|null $zip
 * @property string|null address1
 * @property string|null company
 * @property string|null phone
 * @property string|null fax
 * @property int|null $sales_channel_origin_id
 * @property int|null $product_pricing_tier_id
 * @property int|null $store_id
 * @property int|null $shipping_method_domestic_id
 * @property int|null $shipping_method_international_id
 * @property int|null $warehouse_id
 * @property string $hash
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $archived_at
 * @property-read Collection $all_addresses
 * @property-read Address|null $shippingAddress
 * @property-read Address|null $billingAddress
 * @property-read Address|null $address
 * @property-read SalesChannel|null $salesChannelOrigin
 */
class Customer extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable, TransformsExportData, UniqueFieldsInterface
{
    use Archive, HasFilters, HasSort, Notifiable;
    use BulkImport;
    use HasFactory;

    protected $fillable = [
        'name',
        'email',
        'zip',
        'address1',
        'company',
        'phone',
        'fax',
        'product_pricing_tier_id',
        'store_id',
        'shipping_method_domestic_id',
        'shipping_method_international_id',
        'warehouse_id',
        'sales_channel_origin_id',
        'default_shipping_address_id',
        'default_billing_address_id',
    ];

    protected $casts = [
        'name' => EncryptValue::class,
        'email' => EncryptValue::class,
        'address1' => EncryptValue::class,
        'company' => EncryptValue::class,
        'phone' => EncryptValue::class,
    ];

    public static function getUniqueFields(): array
    {
        return [
            'name',
            'email',
            'zip',
            'address1',
        ];
    }

    //  protected $attributes = [ 'sales_channel_origin_id' => SalesChannel::LOCAL_CHANNEL_ID ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
     */

    public function salesOrders()
    {
        return $this->hasMany(SalesOrder::class);
    }

    public function salesCredits()
    {
        return $this->hasMany(SalesCredit::class);
    }

    public function address()
    {
        return $this->belongsTo(Address::class, 'default_shipping_address_id');
    }

    public function billingAddress()
    {
        return $this->belongsTo(Address::class, 'default_billing_address_id');
    }

    public function shippingAddress()
    {
        return $this->belongsTo(Address::class, 'default_shipping_address_id');
    }

    public function addresses()
    {
        $relation = $this->morphToMany(Address::class, 'addressable')->withTimestamps()->withPivot('is_default');

        // overwrite delete function to delete its pricing
        $relation->onDelete(function (Builder $builder) {
            return $this->addresses->each(function (Address $address) {
                $address->delete();
            });
        });

        return $relation;
    }

    public function salesChannelOrigin()
    {
        return $this->belongsTo(SalesChannel::class, 'sales_channel_origin_id');
    }

    public function productPricingTier()
    {
        return $this->belongsTo(ProductPricingTier::class);
    }

    public function store()
    {
        return $this->belongsTo(Store::class);
    }

    public function shippingMethodDomestic()
    {
        return $this->belongsTo(ShippingMethod::class, 'shipping_method_domestic_id');
    }

    public function shippingMethodInternational()
    {
        return $this->belongsTo(ShippingMethod::class, 'shipping_method_international_id');
    }

    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
     */

    public function firstName(): \Illuminate\Database\Eloquent\Casts\Attribute{
        return \Illuminate\Database\Eloquent\Casts\Attribute::get(fn() => explode(' ', $this->name)[0] ?? '');
    }

    public function middleInitial(): \Illuminate\Database\Eloquent\Casts\Attribute{
        return \Illuminate\Database\Eloquent\Casts\Attribute::get(function(){
            $split = explode(' ', $this->name);
            if(count($split) > 2){
                return strtoupper(Str::substr($split[1], 0, 1) . '.');
            } else {
                return '';
            }
        });
    }

    public function lastName(): \Illuminate\Database\Eloquent\Casts\Attribute{
        return \Illuminate\Database\Eloquent\Casts\Attribute::get(function(){
            $split = explode(' ', $this->name);
            if(count($split) > 1){
                return end($split);
            } else {
                return '';
            }
        });
    }

    public function getAllAddressesAttribute()
    {
        $this->load(['shippingAddress', 'billingAddress', 'addresses']);

        $addresses = [];

        if ($this->shippingAddress) {
            $this->shippingAddress->is_default_shipping = true;

            $addresses[] = $this->shippingAddress;
        }

        if ($this->billingAddress) {
            if ($this->billingAddress->id == ($this->shippingAddress->id ?? null)) {
                $addresses[0]->is_default_billing = true;
            } else {
                $this->billingAddress->is_default_billing = true;

                $addresses[] = $this->billingAddress;
            }
        }

        return collect($addresses)->merge($this->addresses);
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
     */

    public function save(array $options = [])
    {
        // handling phone number
        if ($this->isDirty('phone')) {
            $this->phone = $this->phone ? Helpers::getPhoneAttribute($this->phone, $this->shippingAddress?->constantCountry) : null;
        }

        return parent::save($options);
    }

    /**
     * {@inheritDoc}
     */
    public function delete()
    {
        return DB::transaction(function () {
            if ($usage = $this->isUsed()) {
                return $usage;
            }

            $this->loadMissing('billingAddress', 'shippingAddress', 'addresses');

            $deleted = parent::delete();

            if ($deleted) {
                if ($this->default_shipping_address_id === $this->default_billing_address_id) {
                    // Both shipping and billing addresses are the same, we only delete 1
                    if ($this->shippingAddress) {
                        $this->shippingAddress->delete();
                    }
                } else {
                    // Different addresses, delete them both if available.
                    if ($this->billingAddress) {
                        $this->billingAddress->delete();
                    }

                    if ($this->shippingAddress) {
                        // Only delete if different from billing address
                        $this->shippingAddress->delete();
                    }
                }

                $this->addresses()->delete();
                $this->addresses()->detach();
            }

            return $deleted;
        });
    }

    /**
     * Determine if the nominal code is used.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        $usage = [];

        // used in sales orders
        $this->loadCount('salesOrders');
        if ($this->sales_orders_count) {
            $usage['salesOrders'] = trans_choice('messages.currently_used', $this->sales_orders_count, [
                'resource' => 'sales order',
                'model' => 'customer('.$this->name.')',
            ]);
        }

        return count($usage) ? $usage : false;
    }

    /**
     * @return Customer[]
     */
    public static function findMatch($request): array
    {
        $name = $request->get('name');
        $zip = $request->get('zip');
        $address1 = $request->get('address1');

        return self::with([])->where('name', $name)->where('zip', $zip)->whereRaw('MATCH(address1) AGAINST(?) > .9', [$address1])->get();
    }

    public function addSalesCredit(SalesCredit $salesCredit)
    {
        // Fill in fields
        $salesCredit->from_address_id = $this->default_shipping_address_id;
        $salesCredit->customer_id = $this->id;
        $salesCredit->store_id = $this->store_id ?? ($this->salesChannelOrigin ? $this->salesChannelOrigin->store->id : null);
        $salesCredit->save();
    }

    /**
     * Check if address matches with a customer or with any customer`s addresses.
     *
     *
     * @return Customer|bool
     */
    public static function exists(array $address)
    {
        $customer_repository = new CustomerRepository();

        // exists based on email
        if (! empty($address['email'])) {
            $customer = $customer_repository->findOneByEmail($address['email']);

            if ($customer) {
                return $customer;
            } else {
                // We have an email address that's not already tied to a customer.
                // We return false so that the customer won't be looked up with
                // address details only but may be potentially created as a new customer.
                // @see SKU-6492
                return false;
            }
        }

        // exists based on combination of name, zip, and address1
        if (! empty($address['name']) && ! empty($address['zip']) && ! empty($address['address1'])) {
            $customer = $customer_repository->findOneByAddress($address);

            if ($customer) {
                return $customer;
            }
        }

        return false;
    }

    public function scopeWhereAddress(Builder $builder, array $address)
    {
        return $builder->where('name', $address['name'])
            ->where('zip', $address['zip'])
            ->where('address1', $address['address1']);
    }

    public function scopeFilterAddress(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        $lastKey = $relation['key'];

        return $builder->{$function.'Has'}('address', function (Builder $builder) use ($lastKey, $value, $operator) {
            $builder->filterKey([
                'key' => $builder->qualifyColumn($lastKey),
                'is_relation' => false,
            ], $operator, $value);
        });
    }

    /**
     * @return Address|Model
     */
    public function appendAddress(array $address)
    {
        $customerAddresses = $this->all_addresses;

        // Label not required by sales order creation API.  If label not provided:
        if (empty($address['label'])) {
            // See if any existing address matches all the data fields provided, if found, use that address id
            if ($exactAddress = $this->exactAddress($customerAddresses, $address)) {
                return $exactAddress;
            }

            // If no match, use auto generated label of “Address 1”, “Address 2”, etc.  Whatever # does not yet exist
            $address['label'] = $this->generateLabel($customerAddresses, 'Address');

            return $this->addresses()->create($address);
        } else {
            // If label is provided
            // If label matches existing label and match address details
            if ($exactAddress = $this->exactAddress($customerAddresses, $address, true)) {
                return $exactAddress;
            }

            // but details of address are different than that label,
            // look for match with other addresses with the exact same address info.

            // If label is provided but not already used,
            // still check if address details match any existing addresses, and just use that id instead of creating a new one
            if ($exactAddress = $this->exactAddress($customerAddresses, $address)) {
                return $exactAddress;
            }

            // If label matches existing label
            if ($customerAddresses->where('label', $address['label'])->isNotEmpty()) {
                // if still not found, create a new label with the provided label plus a “ (2)” or whatever number is needed to make it unique
                $address['label'] = $this->generateLabel($customerAddresses, $address['label']);

                return $this->addresses()->create($address);
            } else {
                // If label is provided but not already used,
                // create a new address with the provided info
                return $this->addresses()->create($address);
            }
        }
    }

    private function exactAddress(Collection $addresses, array $address, $withLabel = false)
    {
        if (empty($address)) {
            return null;
        }

        $clone = clone $addresses;

        $allowedColumns = [
            'name',
            'email',
            'phone',
            'address1',
            'address2',
            'address3',
            'city',
            'province',
            'province_code',
            'zip',
            'country_code',
            'country',
        ];

        foreach ($address as $field => $value) {
            if ($withLabel || in_array($field, $allowedColumns)) {
                $clone = ($field == 'phone' && ! empty($value)) ? $clone->where($field, Helpers::getPhoneAttribute($value)) :
                    $clone->where($field, $value);
            }
        }

        return $clone->first();
    }

    private function generateLabel($addresses, $prefix)
    {
        $lastNum = 1;

        foreach ($addresses as $address) {
            if (Str::startsWith($address['label'], $prefix)) {
                $num = (int) filter_var((explode("{$prefix} ", $address['label'])[1] ?? null), FILTER_SANITIZE_NUMBER_INT);
                if ($num >= $lastNum) {
                    $lastNum = $num + 1;
                }
            }
        }

        return "{$prefix} ({$lastNum})";
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.customer.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        if (Str::startsWith(func_get_arg(0), ['sales_channel_origin.'])) {
            return func_get_args();
        }

        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name', 'email', 'zip', 'address1', 'company_name'];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    /**
     * Set/Unset default shipping address.
     */
    public function setDefaultShippingAddress($addressId, bool $set = true): bool
    {
        if ($set === true && $this->default_shipping_address_id != $addressId) {
            // detach the address from other addresses and set as default shipping address
            $this->addresses()->detach($addressId);

            if ($addressId == $this->default_billing_address_id || ($addressId != $this->default_billing_address_id && $this->default_billing_address_id != $this->default_shipping_address_id)) {
                // attach previous shipping address to other address
                $this->addresses()->syncWithoutDetaching([$this->default_shipping_address_id]);
            }

            $this->default_shipping_address_id = $addressId;
        } elseif ($set === false) {
            if ($this->default_shipping_address_id == $addressId) {
                if ($addressId != $this->default_billing_address_id) {
                    // attach previous shipping addresses to other address
                    $this->addresses()->syncWithoutDetaching([$this->default_shipping_address_id]);
                }

                $this->default_shipping_address_id = null;
            }
        }

        return $this->save();
    }

    /**
     * Set/Unset default billing address.
     */
    public function setDefaultBillingAddress($addressId, bool $set = true): bool
    {
        if ($set === true && $this->default_billing_address_id != $addressId) {
            // detach the address from other addresses and set as default shipping address
            $this->addresses()->detach($addressId);

            if ($addressId == $this->default_shipping_address_id || ($addressId != $this->default_shipping_address_id && $this->default_billing_address_id != $this->default_shipping_address_id)) {
                // attach previous shipping address to other address
                $this->addresses()->syncWithoutDetaching([$this->default_billing_address_id]);
            }

            $this->default_billing_address_id = $addressId;
        } elseif ($set === false) {
            if ($this->default_billing_address_id == $addressId) {
                if ($addressId != $this->default_shipping_address_id) {
                    // attach previous shipping addresses to other address
                    $this->addresses()->syncWithoutDetaching([$this->default_billing_address_id]);
                }

                $this->default_billing_address_id = null;
            }
        }

        return $this->save();
    }

    public function isAddressBelongsToMe($addressId)
    {
        return $this->all_addresses->where('id', $addressId)->isNotEmpty();
    }

    public function newQuery()
    {
        return parent::newQuery()->with([
            'productPricingTier',
            'store',
            'warehouse',
            'shippingMethodDomestic',
            'shippingMethodInternational',
        ]);
    }

    /**
     * Indicates if the customer is an amazon customer.
     */
    public function isAmazon(): bool
    {
        if (! $this->salesChannelOrigin || ! $this->salesChannelOrigin->integrationInstance) {
            return false;
        }

        return $this->salesChannelOrigin->integrationInstance->isAmazonInstance();
    }

    public function canNotifyAboutOrder(): bool
    {
        return ! $this->isAmazon();
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new CustomerDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return CustomerDataImporter::getExportableFields();
    }

    public function resetPii(): void
    {
        $placeholder = '****Redacted****';

        $this->fill([
            'name' => $placeholder,
            'email' => $placeholder,
            'address1' => $placeholder,
            'company' => $placeholder,
            'phone' => $placeholder,
        ]);

        $this->save();
    }

    public static function transformExportData(array $data): array
    {
        // Prevent exporting of Amazon Customer Data
        return array_map(function ($customer) {
            if (@$customer['sales_channel_origin']['name'] == Integration::NAME_AMAZON_US) {
                return [];
            }

            return $customer;
        }, $data);
    }
}
