<?php

namespace App\Models;

use App\ImportExport\ImportExport;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Response;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use InvalidArgumentException;

/**
 * Class CsvTemplate.
 *
 *
 * @property int $id
 * @property string $name
 * @property string $model
 * @property array $columns
 * @property int $import_count
 * @property int $export_count
 * @property Carbon|null $last_imported_at
 * @property Carbon|null $last_exported_at
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read  string $sample
 * @property-read  array $fields
 * @property-read  Builder|HasFilters|HasSort|Archive $model_builder
 */
class CsvTemplate extends Model implements Filterable, Sortable
{
    use Archive, HasFilters, HasSort;

    protected $fillable = ['name', 'model', 'columns'];

    protected $casts = [
        'last_imported_at' => 'datetime',
        'last_exported_at' => 'datetime',
        'archived_at' => 'datetime', 'columns' => 'array', 'import_count' => 'integer', 'export_count' => 'integer',
    ];

    protected $appends = ['sample'];

    const MODEL_PRODUCT = 'Product';

    const MODEL_SALES_ORDER = 'Sales Order';

    const MODEL_PRODUCT_BRAND = 'Product Brand';

    const MODEL_SUPPLIER = 'Supplier';

    const MODELS = [
        self::MODEL_PRODUCT,
        self::MODEL_SALES_ORDER,
        self::MODEL_PRODUCT_BRAND,
        self::MODEL_SUPPLIER,
    ];

    /**
     * @return Response|array
     *
     * @throws Exception
     */
    public function import(string $file, bool $preview = false, ?int $limit = null)
    {
        $importClass = null;
        switch ($this->model) {
            case self::MODEL_PRODUCT:
                // update import count and date
                $importClass = \App\ImportExport\Product::class;
                break;
            case self::MODEL_SALES_ORDER:
                $importClass = \App\ImportExport\SalesOrder::class;
                break;
            default:
                throw new InvalidArgumentException(__('messages.import_export.model_not_support', ['input' => $this->model]));
        }

        if (! $preview) {
            $this->recordImporting();
        }

        return ( new $importClass($this->fields) )->import($file, ! $preview, $limit);
    }

    /**
     * @param  iterable|callable|Model|null  $exportModels
     *
     * @throws Exception
     */
    public function export(bool $sample = false, ?int $limit = null, $exportModels = null): Response
    {
        /** @var ImportExport $exportClass */
        $exportClass = null;
        switch ($this->model) {
            case self::MODEL_PRODUCT:
                $exportClass = \App\ImportExport\Product::class;
                break;
            case self::MODEL_SALES_ORDER:
                $exportClass = \App\ImportExport\SalesOrder::class;
                break;
            case self::MODEL_PRODUCT_BRAND:
                $exportClass = \App\ImportExport\ProductBrand::class;
                break;
            case self::MODEL_SUPPLIER:
                $exportClass = \App\ImportExport\Supplier::class;
                break;
            default:
                throw new InvalidArgumentException(__('messages.import_export.model_not_support', ['input' => $this->model]));
        }

        // update export count and date
        if (! $sample) {
            $this->recordExporting();
        }

        $exportClass = new $exportClass($this->fields);

        $exportModels = $sample ? $this->getModelBuilder($sample)->firstOrNew([]) : $exportModels;

        return $exportClass->setExportModels($exportModels)
            ->export($sample ? $this->getSampleFileName() : null, $limit);
    }

    /**
     * Determine whether the file has same template fields.
     *
     *
     * @throws Exception
     */
    public function isSame(string $file): bool
    {
        switch ($this->model) {
            case self::MODEL_PRODUCT:
                $fileFields = ( new \App\ImportExport\Product($this->fields) )->getColumnsFromFile($file);
                $templateFields = ( new \App\ImportExport\Product($this->fields) )->getColumnsFromFields();

                return ! boolval(array_diff($fileFields, $templateFields));
        }

        return false;
    }

    public function getSampleAttribute()
    {
        return $this->export(true)->getOriginal('url');
    }

    public function getFieldsAttribute()
    {
        return array_merge($this->columns['attributes'] ?? [], $this->columns['relations'] ?? []);
    }

    public function getModelBuilder(bool $sample = false)
    {
        switch ($this->model) {
            case self::MODEL_PRODUCT:
                return \App\Queries\Product::with([
                    'productAttributes',
                    'suppliers',
                    'productPricingTiers',
                    'supplierProducts',
                    'supplierProducts.supplierPricingTiers',
                    'primaryCategory',
                    'otherCategories',
                ]);
            case self::MODEL_SALES_ORDER:
                if ($sample) {
                    return SalesOrderLine::with(
                        [
                            'salesOrder.salesChannel',
                            'salesOrder.customer.address',
                            'salesOrder.warehouse',
                            'product',
                            'nominalCode',
                            'salesOrder.shippingMethod',
                        ]
                    );
                }

                return SalesOrder::with([
                    'salesChannel',
                    'customer.address',
                    'warehouse',
                    'salesOrderLines',
                    'salesOrderLines.product',
                    'salesOrderLines.nominalCode',
                    'shippingMethod',
                ]);
            case self::MODEL_PRODUCT_BRAND:
                return ProductBrand::with([]);
            case self::MODEL_SUPPLIER:
                return Supplier::with([]);
            default:
                throw new InvalidArgumentException(__('messages.import_export.model_not_support', ['input' => $this->model]));
        }
    }

    public function getUsedCountAttribute()
    {
        if ($this->getOriginal('used_count', false) !== false) {
            return $this->getOriginal('used_count');
        }

        return $this->export_count + $this->import_count;
    }

    public function getLastUsedAtAttribute()
    {
        if ($this->getOriginal('last_used_at', false) !== false) {
            return $this->getOriginal('last_used_at') ? Carbon::parse($this->getOriginal('last_used_at')) : null;
        }

        if (! $this->last_exported_at && ! $this->last_imported_at) {
            return null;
        }

        if (! $this->last_exported_at || ! $this->last_imported_at) {
            return $this->last_exported_at ?: $this->last_imported_at;
        }

        return $this->last_exported_at->gt($this->last_imported_at) ? $this->last_exported_at : $this->last_imported_at;
    }

    /**
     * Get sample file name.
     */
    private function getSampleFileName(): string
    {
        return "template-{$this->id}.csv";
    }

    /**
     * Update export count and data fot the template.
     */
    private function recordExporting(): bool
    {
        $this->last_exported_at = now();
        $this->export_count += 1;

        return $this->save();
    }

    /**
     * Update import count and data fot the template.
     */
    private function recordImporting(): bool
    {
        $this->last_imported_at = now();
        $this->import_count += 1;

        return $this->save();
    }

    public function customColumns()
    {
        return [
            'used_count' => '`export_count` + `import_count`',
            'last_used_at' => 'IF(ISNULL(`last_imported_at`), `last_exported_at`, GREATEST(`last_imported_at`, IFNULL(`last_exported_at`, 0)))',
        ];
    }

    public function calculatedColumns()
    {
        return ['used_count', 'last_used_at'];
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return [
            'name',
            'model',
            'export_count',
            'import_count',
            'used_count',
            'last_exported_at',
            'last_imported_at',
            'last_used_at',
        ];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name', 'model'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return $this->availableColumns();
    }
}
